<?php
/**
 * Legacy VIA wrapper implementation
 *
 *
 * @author Andrew Gilmour <andrew.gilmour@red61.com>
 * @author Will Tatam <will.tatam@red61.com>
 * @copyright 2014 Red61 Ltd
 * @licence   proprietary
 */
use Red61\Via\Cache\ViaCacheDriver;
use Red61\Via\Cache\ViaCacheKeyGenerator;
use Red61\Via\Cache\ViaCachePlugin;
use Red61\Via\Cache\ViaFilecacheDriver;
use Red61\Via\Cache\ViaMemcacheDriver;
use Red61\Via\DataObject\ViaApiAddressDetails;
use Red61\Via\DataObject\ViaApiBasketEventItem;
use Red61\Via\DataObject\ViaApiBasketSummary;
use Red61\Via\DataObject\ViaApiQuestionResponse;
use Red61\Via\DataObject\ViaApiTicketRequest;
use Red61\Via\DataObject\ViaApiTicketAttribute;
use Red61\Via\DataObject\ViaApiTicketAttributeSelection;
use Red61\Via\DataObject\ViaApiTicketAttributeRequest;
use Red61\Via\Exception\BasketChangedException;
use Red61\Via\Exception\ClientInvalidRequestException;
use Red61\Via\Exception\GiftAidDeclarationOverlapException;
use Red61\Via\Exception\ViaExceptionMapper;
use Red61\Via\Plugin\ViaCorePluginManager;
use Red61\Via\Plugin\ViaPluginManager;
use Red61\Via\Profiler\ViaProfilerPlugin;
use Red61\Via\DataObject\ViaApiBasket;
use Red61\Via\SessionStorage\ViaSessionStorage;
use Red61\Via\SoapClientFactory;
use Red61\Via\ViaApiClient;
use Red61\Via\ViaApiService;

/**
 * This class contains all the functions required to list events, list performances for events, add tickets for a
 * performance to a shopping basket, manage the shopping basket, create an order, create customers, update customer's
 * details and for users to log in.
 *
 * [!!] New projects should consider instead using the RPC-style Red61\Via\ViaApiService interface. This interface is
 *      provided for backwards compatability.
 *
 * @package red61_via
 */
class Red61_Via{

	private $_localData;

	/**
	 * @var ViaApiService
	 */
	protected $api_service;

	/**
	 * @var ViaApiClient
	 */
	protected $api_client;

	/**
	 * @var ViaPluginManager
	 */
	protected $plugin_manager;

	/**
	 * @var ViaProfilerPlugin - may be null if no profiling is configured
	 */
	protected $profiler;

	/**
	 * @var bool if TRUE, then any exception during an API call will always be thrown.
	 */
	protected $always_rethrow = FALSE;

	/**
	 * The constuctor for the red61_via class, must be called at the start of each page.
	 * 
	 * [!!] Exceptions: getBasketSummary in particular will by default swallow any exceptions (including
	 *      CartNotFound and others for legacy compatibility reasons. You may prefer to guarantee that
	 *      exceptions will bubble up to your application code - if so, pass an array including
	 *      'always_rethrow' => TRUE to the $conf options.
	 *
	 * @param string $wsdl     url of the wsdl file, used to create the soap client.
	 * @param string $web_key  credentials, defines the level of access
	 * @param string $basketId The unique identifier for the shopping basket, NULL until items in basket.
	 * @param array  $conf     Optional config options such as cache=>false
	 * @param array  $arr      Used for creating the soap client. Optional as default value sufficient.
	 *
	 */

	protected $dayOffset = null;

	function __construct($wsdl, $web_key, $basketId = NULL, $conf=array('cache'=>'true'), $arr=array()) {
		$soap_factory         = new SoapClientFactory($arr);
		$exception_mapper     = new ViaExceptionMapper();
		$this->plugin_manager = new ViaCorePluginManager;
		$this->api_client     = new ViaApiClient($soap_factory, $exception_mapper, $this->plugin_manager, $wsdl, $web_key);
		$this->api_service    = new ViaApiService($this->api_client);
		if ($basketId) {
			$this->api_client->setBasketId($basketId);
		}

		if (isset($conf['cache']) AND $conf['cache']) {
			$this->configureCachePlugin($conf, $wsdl, $web_key);
		}

		if (isset($conf['profile']) AND $conf['profile']) {
			$this->profiler = new ViaProfilerPlugin;
			$this->profiler->registerWithManager($this->plugin_manager);
		}

		$this->always_rethrow = isset($conf['always_rethrow']) ? $conf['always_rethrow'] : FALSE;
	}

	/**
	 * Attempts to set up the caching plugin using the legacy configuration format. To use your own driver,
	 * pass an instance of ViaCacheDriver in the cache_handler option.
	 *
	 * @param array  $conf
	 * @param string $wsdl
	 * @param string $web_key
	 *
	 * @throws InvalidArgumentException if the cache configuration is not valid
	 */
	protected function configureCachePlugin($conf, $wsdl, $web_key)
	{
		$driver = NULL;
		if ( ! isset($conf['cache_handler'])) {
			if ( ! isset($conf['cache_dir'])) {
				list($user_id) = explode(':', $web_key);
				$conf['cache_dir'] = "FS:".sys_get_temp_dir().DIRECTORY_SEPARATOR."api".DIRECTORY_SEPARATOR.preg_replace('/[^A-Za-z0-9]/','',$wsdl).DIRECTORY_SEPARATOR.$user_id;
			}

			if( ! preg_match('/^(FS|memcache):(.+)/i', $conf['cache_dir'], $matches)) {
				throw new InvalidArgumentException("Could not parse cache backend configuration ".$conf['cache_dir']);
			}

			if ($matches[1] === 'FS')
			{
				$driver = new ViaFilecacheDriver($matches[2]);
			}
			else
			{
				$driver = new ViaMemcacheDriver(new Memcache, array('host' => $matches[2]));
			}
		}

		if ( ! $driver instanceof ViaCacheDriver)
		{
			throw new \InvalidArgumentException('The cache driver provided must implement the ViaCacheDriver interface');
		}

		$cache_plugin = new ViaCachePlugin(
			$driver,
			new ViaCacheKeyGenerator,
			array()
		);
		$this->plugin_manager->registerPlugin($cache_plugin);
	}

	/**
	 * Assigns a ViaSessionStorage implementation to persist the user's basket ID between requests.
	 *
	 * This will trigger the API client to load the current basket ID from the session. The session storage
	 * will then receive a saveBasketId call whenever a basket is created or invalidated.
	 *
	 * [!!] Providing SessionStorage is optional - by default no session storage will be set and you will need
	 *      to manage persistence of the basket ID in your application code.
	 *
	 * @param ViaSessionStorage $storage
	 *
	 * @return void
	 */
	public function connectSessionStorage(ViaSessionStorage $storage)
	{
		$this->api_client->connectSessionStorage($storage);
	}

	/**
	 * Uses the authentication code set in the constructor (<i>$web_key</i>) to return an array of event objects.
	 * If there are no events it returns an empty array.
	 *
	 * @deprecated in favour of getEvents
	 * @return \Red61\Via\DataObject\ViaApiEvent[]
	 */
	public function getEventsList() {

		trigger_error("Call to deprecated method getEventsList(), use getEvents()", E_USER_NOTICE);
		return $this->getEvents();
	}

	/**
	 * Uses the authentication code set in the constructor (<i>$web_key</i>) to return an array of event objects.
	 * If there are no events it returns an empty array.
	 *
	 * @return \Red61\Via\DataObject\ViaApiEvent[]
	 * @see    \Red61\Via\ViaApiService::getEvents
	 */
	public function getEvents() {
		return $this->api_service->getEvents();
	}

	/**
	 * Uses the authentication code set in the constructor (<i>$web_key</i>) to return an array of Customer Category
	 * objects. If $customerId is non zero then the customer categories applied to this customer will be marked as active.
	 * If there are no Customer Categories it returns an empty array.
	 *
	 * @param int $customerId
	 *
	 * @return \Red61\Via\DataObject\ViaApiCustomerCategory[]
	 * @see    \Red61\Via\ViaApiService::getCustomerCategories
	 */
	public function getCustomerCategories($customerId) {
		return $this->api_service->getCustomerCategories(
			\Red61\Via\ApiRequest\apiGetCustomerCategoriesRequest::create()
				->setCustomerId($customerId)
		);
	}

	/**
	 * Uses the authentication code set in the constructor (<i>$web_key</i>) to set the categories of a customer.
	 *
	 * @param int   $customerId
	 * @param int[] $categories an array of ids to set
	 *
	 * @return void
	 * @see \Red61\Via\ViaApiService::setCustomerCategories
	 */
	public function setCustomerCategories($customerId, $categories) {
		$this->api_service->setCustomerCategories(
			\Red61\Via\ApiRequest\apiSetCustomerCategoriesRequest::create()
				->setCustomerId($customerId)
				->setCategories($categories)
		);
	}

	/**
	 * Uses the authentication code set in the constructor (<i>$web_key</i>) to return an array of Postage Charge
	 * objects. If there are no Postage Charges it returns an empty array.
	 *
	 * @return \Red61\Via\DataObject\ViaApiPostageCharge[]
	 * @see    \Red61\Via\ViaApiService::getPostageCharges
	 * @deprecated for getDeliveryOptions
	 */
	public function getPostageCharges() {
		$this->logDepCall("getPostageCharges replaced by getDeliveryOptions");
		return $this->api_service->getPostageCharges();
	}

	/**
	 * Uses the authentication code set in the constructor (<i>$web_key</i>) to set the postage charge of the
	 * basket to be the the postage charge indicated by the id passed in. 0 should be used if this order is to be
	 * collected.
	 *
	 * @param int $chargeId Charge to set, 0 if order is to be collected
	 *
	 * @return void
	 * @see    \Red61\Via\ViaApiService::setPostageCharge
	 * @deprecated for setDeliveryOption
	 */
	public function setPostageCharge($chargeId) {
		$this->logDepCall("setPostageCharge replaced by setDeliveryOption");
		$this->api_service->setPostageCharge(
			\Red61\Via\ApiRequest\apiSetPostageChargeRequest::create()
				->setChargeId($chargeId)
		);
	}

	/**
	 * Cleans the given order
	 *
	 * @param string $orderRef      The reference of the order to clean.
	 * @param bool   $emailCustomer Whether to email the customer to inform them of the clean.
	 *
	 * @return bool
	 * @see \Red61\Via\ViaApiService::cleanUnpaidOrder
	 *
	 */
	public function cleanUnpaidOrder($orderRef, $emailCustomer) {
		return $this->api_service->cleanUnpaidOrder(
			\Red61\Via\ApiRequest\apiCleanUnpaidOrderRequest::create()
				->setOrderRef($orderRef)
				->setEmailCustomer($emailCustomer)
		);
	}

	/**
	 * Creates a new gift aid declaration for a given customer. The start date is the system default or date now
	 * and the end date is assumed to be forever.
	 *
	 * @param int $customerId the id of the customer the gift aid declaration will be created against
	 *
	 * @throws GiftAidDeclarationOverlapException if the customer already has an active declaration covering date now.
	 * @return void
	 * @see \Red61\Via\ViaApiService::createGiftAidDeclaration
	 */
	public function createGiftAidDeclaration($customerId) {
		$this->api_service->createGiftAidDeclaration(
			\Red61\Via\ApiRequest\apiCreateGiftAidDeclarationRequest::create()
				->setCustomerId($customerId)
		);
	}

	/**
	 * Uses the authentication code set in the constructor (<i>$web_key</i>) and  the parameter <i>$serverIdAndEventId</i>
	 * to return an array of performance objects. If there are no performances it returns an empty array.
	 *
	 * @param string $serverIdAndEventId The id of the server/event you wish to view performances for.
	 *
	 * @return \Red61\Via\DataObject\ViaApiPerformance[]
	 * @deprecated in favour of getPerformances
	 */
	public function getPerformancesList($serverIdAndEventId) {
		trigger_error("Call to deprecated method getPerformancesList(), use getPerformances()", E_USER_NOTICE);
		return $this->getPerformances($serverIdAndEventId);
	}
	/**
	 * Uses the authentication code set in the constructor (<i>$web_key</i>) and  the parameter <i>$serverIdAndEventId</i>
	 * to return an array of performance objects. If there are no performances it returns an empty array.
	 *
	 * @param string $serverIdAndEventId The id of the server/event you wish to view performances for.
	 *
	 * @return \Red61\Via\DataObject\ViaApiPerformance[]
	 * @see    \Red61\Via\ViaApiService::getPerformances
	 */
	public function getPerformances($serverIdAndEventId) {
		return $this->api_service->getPerformances(
			\Red61\Via\ApiRequest\apiGetPerformancesRequest::create()
				->setEventRef($serverIdAndEventId)
		);
	}

	/**
	 * Uses the authentication code set in the constructor (<i>$web_key</i>) and  the parameter <i>$serverIdAndEventId</i>
	 * to return an array of performance objects. If there are no performances it returns an empty array.
	 *
	 * @param string   $serverIdAndEventId id of the server/event you wish to view performances for.
	 * @param string[] $schemeRefs         schemes customer is subscribed to, each element in the form "serverId:schemeId"
	 *
	 * @return \Red61\Via\DataObject\ViaApiPerformance[]
	 * @see    \Red61\Via\ViaApiService::getPerformancesByScheme
	 * @deprecated for getPerformancesByScheme
	 */
	public function getPerformancesListByScheme($serverIdAndEventId, $schemeRefs) {
		trigger_error("Call to deprecated method getPerformancesListByScheme(),
			use getPerformancesByScheme()", E_USER_NOTICE);
		return $this->getPerformancesByScheme($serverIdAndEventId, $schemeRefs);
	}

	/**
	 * Uses the authentication code set in the constructor (<i>$web_key</i>) and  the parameter <i>$serverIdAndEventId</i>
	 * to return an array of performance objects. If there are no performances it returns an empty array.
	 *
	 * @param string   $serverIdAndEventId id of the server/event you wish to view performances for.
	 * @param string[] $schemeRefs         schemes customer is subscribed to, each element in the form "serverId:schemeId"
	 *
	 * @return \Red61\Via\DataObject\ViaApiPerformance[]
	 * @see    \Red61\Via\ViaApiService::getPerformancesByScheme
	 */
	public function getPerformancesByScheme($serverIdAndEventId, $schemeRefs) {
		return $this->api_service->getPerformancesByScheme(
			\Red61\Via\ApiRequest\apiGetPerformancesBySchemeRequest::create()
				->setEventRef($serverIdAndEventId)
				->setSchemeRefs($schemeRefs)
		);
	}

	/**
	 * Uses the parameter <i>$performance</i> to return an array of performanceConcession objects. If there are no
	 * concessions for this performance it returns an empty array. This method is used in conjunction with the
	 * method <i>getPerformances($eventId)</i>. While looping through the array of Performance objects returned from
	 * that method you pass a Performance object to this method.
	 *
	 * @param \Red61\Via\DataObject\ViaApiPerformance $performance The Performance you wish to view the concessions for.
	 *
	 * @return \Red61\Via\DataObject\ViaApiPerformanceConcession[]
	 * @see    \Red61\Via\DataObject\ViaApiPerformance::getConcessions
	 * @deprecated it is now safe simply to call \Red61\Via\DataObject\ViaApiPerformance::getConcessions
	 */
	public function getPerformanceConcessions(\Red61\Via\DataObject\ViaApiPerformance $performance){
		return $performance->getConcessions();
	}

	/**
	 * Uses the authentication code set in the constructor (<i>$web_key</i>) and  the parameter <i>$performanceId</i>
	 * to return an array of performancePrice objects. If there are no prices it returns an empty array.
	 *
	 * [!!] This method returns the price array found within the apiGetPerformancePrices result, not the raw result.
	 *
	 * @param string $serverIdAndPerformanceId The id of the server/performance you wish to view prices for.
	 *
	 * @return \Red61\Via\DataObject\ViaApiPerformancePrice[]
	 * @see    \Red61\Via\ViaApiService::getPerformancePrices
	 */
	public function getPerformancePrices($serverIdAndPerformanceId) {
		$prices = $this->getPerformancePriceDetails($serverIdAndPerformanceId);
		return $prices->getPrices();
	}

	/**
	 * Uses the authentication code set in the constructor (<i>$web_key</i>) and  the parameter <i>$performanceId</i> to
	 * return a ViaApiPerformancePrices object which can be used to retrieve the allocation details
	 *
	 * @param string $serverIdAndPerformanceId The id of the server/performance you wish to view prices for.
	 *
 	 * @return \Red61\Via\DataObject\ViaApiPerformancePrices
	 * @see    \Red61\Via\ViaApiService::getPerformancePrices
	 */
	public function getPerformancePriceDetails($serverIdAndPerformanceId) {
		return $this->api_service->getPerformancePrices(
			\Red61\Via\ApiRequest\apiGetPerformancePricesRequest::create()
				->setPerformanceRef($serverIdAndPerformanceId)
		);
	}

	/**
	 * Fetches live performance availability for the specified performance.
	 *
	 * [!!] This method is expensive and should only be used following consultation with Red61.
	 *
	 * @param string $serverIdAndPerformanceId
	 *
	 * @return int
	 * @see    getPerformanceAvailability
	 * @deprecated for getPerformanceAvailability
	 */
	public function getPerformanceAvailibility($serverIdAndPerformanceId) {
		return $this->getPerformanceAvailability($serverIdAndPerformanceId);
	}

	/**
	 * Fetches live performance availability for the specified performance.
	 *
	 * [!!] This method is expensive and should only be used following consultation with Red61.
	 *
	 * @param string $serverIdAndPerformanceId
	 *
	 * @return int
	 * @see    \Red61\Via\ViaApiService::getPerformanceAvailability
	 */
	public function getPerformanceAvailability($serverIdAndPerformanceId) {
		return $this->api_service->getPerformanceAvailability(
			\Red61\Via\ApiRequest\apiGetPerformanceAvailabilityRequest::create()
				->setPerformanceRef($serverIdAndPerformanceId)
		);
	}

	/**
	 * Fetches a list of live performance availability for the specified performances
	 *
	 * [!!] This method is expensive and should only be used following consultation with Red61.
	 *
	 * @param string[] $serverIdAndPerformanceIds
	 *
	 * @return \Red61\Via\DataObject\ViaApiListValue[]
	 * @see    getPerformanceAvailabilityList
	 * @deprecated for getPerformanceAvailabilityList
	 */
	public function getPerformanceAvailibilityList(array $serverIdAndPerformanceIds) {
		return $this->getPerformanceAvailabilityList($serverIdAndPerformanceIds);
	}

	/**
	 * Fetches a list of live performance availability for the specified performances
	 *
	 * [!!] This method is expensive and should only be used following consultation with Red61.
	 *
	 * @param string[] $serverIdAndPerformanceIds
	 *
	 * @return \Red61\Via\DataObject\ViaApiListValue[]
	 * @see    \Red61\Via\ViaApiService::getPerformanceAvailabilityList
	 */
	public function getPerformanceAvailabilityList(array $serverIdAndPerformanceIds) {
		return $this->api_service->getPerformanceAvailabilityList(
			\Red61\Via\ApiRequest\apiGetPerformanceAvailabilityListRequest::create()
				->setPerformanceRefs($serverIdAndPerformanceIds)
		);
	}

	/**
	 * This method is used for fetching the current number of seats for sale to the current user for the supplied list of
	 * performances, broken down by price band.
	 *
	 * [!!] This method is expensive and should only be used following consultation with Red61.
	 *
	 * @param string $serverIdAndPerformanceId
	 *
	 * @return \Red61\Via\DataObject\ViaApiPerformanceAvailability
	 * @see    \Red61\Via\ViaApiService::getPriceBandAvailability
	 */
	public function getPriceBandAvailability($serverIdAndPerformanceId) {
		return $this->api_service->getPriceBandAvailability(
			\Red61\Via\ApiRequest\apiGetPriceBandAvailabilityRequest::create()
				->setPerformanceRef($serverIdAndPerformanceId)
		);
	}

	/**
	 * Uses the parameter <i>$serverIdAndEventId</i> to return an event object from the server
	 *
	 * @param string $serverIdAndEventId The id of the server/event you wish to view details for, in form "serverId:eventId"
	 *
	 * @return \Red61\Via\DataObject\ViaApiEvent
	 * @see    \Red61\Via\ViaApiService::getEventDetails
	 */
	public function getEventDetails($serverIdAndEventId) {
		$result = $this->api_service->getEventDetails(
			\Red61\Via\ApiRequest\apiGetEventDetailsRequest::create()
				->setEventRef($serverIdAndEventId)
		);
		return $result;
	}

	/**
	 * Returns a list of the donation funds linked to a specific event
	 *
	 * @param String $serverIdAndEventId the id of the server/event you wish to view the linked donation funds for in form "serverId:eventId"
	 *
	 * @return \Red61\Via\DataObject\ViaApiDonationFund[]
	 * @see    \Red61\Via\ViaApiService::getDonationsForEvent
	 */
	public function getDonationsForEvent($serverIdAndEventId) {
		return $this->api_service->getDonationsForEvent(
			\Red61\Via\ApiRequest\apiGetDonationsForEventRequest::create()
				->setEventRef($serverIdAndEventId)
		);
	}

	/**
	 * Uses the parameter <i>$serverIdAndPerformanceId</i> to return a performance object.
	 *
	 * @param string $serverIdAndPerformanceId The id of the server/performance you wish to view details for, in form "serverId:performanceId"
	 *
	 * @return \Red61\Via\DataObject\ViaApiPerformance
	 * @see    \Red61\Via\ViaApiService::getPerformanceDetails
	 */
	public function getPerformanceDetails($serverIdAndPerformanceId) {
		return $this->api_service->getPerformanceDetails(
		    \Red61\Via\ApiRequest\apiGetPerformanceDetailsRequest::create()
		        ->setPerformanceRef($serverIdAndPerformanceId)
		);
	}

	/**
	 * Takes an array of ViaApiPerformance objects and groups them by Edinburgh Fringe month and day
	 *
	 * [!!} This method groups performances based on the Edinburgh Fringe clock, with performances
	 *      starting up to 7am being listed with the date of the day before.
	 *
	 * @param \Red61\Via\DataObject\ViaApiPerformance[] $performances Array of performance objects.
	 *
	 * @return array 3-dimensional array of the provided performances grouped by month and day
	 *
	 * @deprecated for groupPerformancesByFringeTimeDate
	 * @see        groupPerformancesByFringeTimeDate
	 */
	public function groupPerformancesByDate($performances)
	{
		return $this->groupPerformancesByFringeTimeDate($performances);
	}

	/**
	 * Takes an array of ViaApiPerformance objects and groups them by Edinburgh Fringe month and day
	 * (eg performances up to 7am are listed with the date of the day before).
	 *
	 * @param \Red61\Via\DataObject\ViaApiPerformance[] $performances Array of performance objects.
	 *
	 * @return array 3-dimensional array of the provided performances grouped by month and day
	 */
	public function groupPerformancesByFringeTimeDate($performances)
	{
		$lists = array();
		$dayOffset = $this->getDayChangeOffset();
		foreach($performances as $perf) {
			$month = date("Y-m",strtotime($perf->getDatetime()) - ( $dayOffset* 60 * 60));
			$day = date("d",strtotime($perf->getDatetime()) - ( $dayOffset * 60 * 60));
			$lists[$month][$day][] = $perf;
		}
		return($lists);
	}

	/**
	 * Either creates a new customer record (if <i>$create</i> is true and <i>$customerId</i> is 0) or updates an
	 * existing customer record if <i>$create</i> is false and the <i>$customerId</i> is known. If <i>$create</i> is
	 * false then the <i>$customerId</i> must not be 0 or the update will fail. If the create or update fails then the
	 * 0 is returned, otherwise the  customer id is returned (newly created if <i>$create</i> is true or the same as
	 * the parameter <i>$customerId</i> if <i>$create</i> is false.
	 *
	 * @param boolean $create                 If true this will create a new customer record, if false will just update the current customer record defined by the parameter $customerId. If $create is true then parameter $customerId should be set set to 0.
	 * @param string  $title                  The title of the customer, Mr, Mrs etc..
	 * @param string  $forename               The forename of the customer.
	 * @param string  $surname                The surname of the customer.
	 * @param string  $company                The customer's company.
	 * @param string  $address1               The address1 field of the customer's address.
	 * @param string  $address2               The address2 field of the customer's address.
	 * @param string  $address3               The address3 field of the customer's address.
	 * @param string  $city                   The city of the customer's address.
	 * @param string  $county                 The county/state of the customer's address.
	 * @param string  $postcode               The postcode of the customer's address.
	 * @param string  $country                The country of the customer's address
	 * @param string  $dayPhone               The day phone number of the customer.
	 * @param string  $eveningPhone           The evening phone number of the customer.
	 * @param string  $mobile                 The mobile phone number of the customer.
	 * @param string  $email                  The email of the customer.
	 * @param string  $password               The password of the customer.
	 * @param string  $passwordConfirm        The password confirm field of the customer.
	 * @param string  $preferredContactMethod The preferred contact method of the customer.
	 * @param boolean $organisationMailing    Whether the customer allows organisation mailing.
	 * @param boolean $thirdPartyMailing      Whether the customer allows third part mailing.
	 * @param boolean $additionalMailingPermission Whether the customer allows additional mailing
	 * @param string gender					  The customer's gender in the format "M" (for Male), "F" (for Female), "N" (for Non-binary)
	 * Note: passing through NULL will result in the customer's current gender not being updated.
	 * Passing through an empty string will delete the customer's current stored gender
	 * @param string dateOfBirth 			   The customer's date of birth e.g. "1990-12-01"
	 * Note: passing through NULL will result in the customer's current date of birth not being updated.
	 * Passing through "0000-00-00" will delete the customer's current stored date of birth
	 * @param integer $customerId             The id of the customer. This is only relevant for customer detail updates when $create is false. Becomes available once the customer has logged in.
	 * @param string gender					  The customer's gender in the format "M" (for Male), "F" (for Female), "N" (for Non-binary)
	 * Note: passing through NULL will result in the customer's current gender not being updated.
	 * Passing through an empty string will delete the customer's current stored gender
	 * @param string dateOfBirth 			   The customer's date of birth e.g. "1990-12-01"
	 * Note: passing through NULL will result in the customer's current date of birth not being updated.
	 * Passing through "0000-00-00" will delete the customer's current stored date of birth
	 *
	 * @throws SoapFault
	 * @return int integer - customer id
	 * @see    \Red61\Via\ViaApiService::createAccount
	 */
	public function createAccount($create,$title,$forename,$surname,$company,$address1,$address2,$address3,$city,$county,
		$postcode,$country,$dayPhone,$eveningPhone,$mobile,$email,$password,$passwordConfirm,$preferredContactMethod,
		$organisationMailing,$thirdPartyMailing,$additionalMailingPermission, $customerId, $gender=null, $dateOfBirth=null
	) {
		return $this->api_service->createAccount(
		    \Red61\Via\ApiRequest\apiCreateAccountRequest::create()
				->setCreate($create)
				->setTitle($title)
				->setForename($forename)
				->setSurname($surname)
				->setCompany($company)
				->setAddress1($address1)
				->setAddress2($address2)
				->setAddress3($address3)
				->setCity($city)
				->setCounty($county)
				->setCountry($country)
				->setPostcode($postcode)
				->setCountry($country)
				->setDayPhone($dayPhone)
				->setEveningPhone($eveningPhone)
				->setMobile($mobile)
				->setEmail($email)
				->setPassword($password)
				->setPasswordConfirm($passwordConfirm)
				->setPreferredContactMethod($preferredContactMethod)
				->setOrganisationMailingPermission($organisationMailing)
				->setThirdPartyMailingPermission($thirdPartyMailing)
				->setAdditionalMailingPermission($additionalMailingPermission)
				->setGender($gender)
				->setDateOfBirth($dateOfBirth)
				->setCustomerId($customerId)
		);
	}

	/**
	 * Updates the otherDetails of the customer specified by <i>$customerId</i> using the given new details.
	 *
	 * @param int                                              $customerId   Specifies the id of the customer to update
	 * @param \Red61\Via\DataObject\ViaApiCustomerOtherDetails $otherdetails The updated other details
	 *
	 * @throws SoapFault
	 * @return void
	 * @see    \Red61\Via\ViaApiService::updateCustomerOtherDetails
	 */
	public function updateCustomerOtherDetails($customerId, $otherdetails){
		$this->api_service->updateCustomerOtherDetails(
		    \Red61\Via\ApiRequest\apiUpdateCustomerOtherDetailsRequest::create()
				->setCustomerId($customerId)
				->setApiOtherDetails($otherdetails)
		);
	}

	/**
	 * Uses the parameter <i>$price</i> to return an array of performancePriceConcession objects. If there are no
	 * concession for this price band it returns an empty array. This method is used in conjunction with the method
	 * <i>getPerformancePrices($performanceId)</i>. While looping through the array of performancePrice objects
	 * returned from that method you pass a performancePrice object to this method.
	 *
	 *
	 * @param \Red61\Via\DataObject\ViaApiPerformancePrice $price The performancePrice object you wish to view the concessions for.
	 *
	 * @return \Red61\Via\DataObject\ViaApiPerformanceConcession[]
	 * @see    \Red61\Via\DataObject\ViaApiPerformancePrice::getConcessions
	 * @deprecated it is now safe to always call ViaApiPerformancePrice::getConcessions
	 */
	public function getPerformancePriceConcessions(\Red61\Via\DataObject\ViaApiPerformancePrice $price) {
		return $price->getConcessions();
	}

	/**
	 * Adds a scheme to the basket, initialising the basket if necessary. $giftAidResponse is now deprecated
	 *
	 * @param string  $schemeRef       The reference of the scheme to add to the basket.
	 * @param integer $concessionId    The id of the scheme tier concession or 0 for no concession
	 * @param integer $customerId      The id of the customer wishing to subscribe.
	 * @param boolean $giftAidResponse Answer to the scheme gift aid text, if appropriate. This param is optional and deprecated to use since 4.1
	 *
	 * @return boolean boolean indicating if the add succeeded or not.
	 * @deprecated in favour of addSchemeToBasket
	 */
	public function addScheme($schemeRef,$concessionId,$customerId,$giftAidResponse=null){
		trigger_error("Call to deprecated method addScheme(), use addSchemeToBasket()", E_USER_NOTICE);
		return $this->addSchemeToBasket($schemeRef,$concessionId,$customerId,$giftAidResponse);
	}

	/**
	 * Adds a scheme to the basket, initialising the basket if necessary. $giftAidResponse is now deprecated
	 *
	 * @param string $schemeRef The reference of the scheme to add to the basket.
	 * @param integer $concessionId The id of the scheme tier concession or 0 for no concession
	 * @param integer $customerId The id of the customer wishing to subscribe.
	 * @param boolean $giftAidResponse Answer to the scheme gift aid text, if appropriate. This param is optional and deprecated to use since 4.1
	 *
	 * @return boolean boolean indicating if the add succeeded or not.
	 * 
	 * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
	 * @throws \Red61\Via\Exception\SchemeUnavailableException  If the scheme tier requested or the scheme is not available to the web
	 * 
	 * 
	 * @see    \Red61\Via\ViaApiService::addSchemeToBasket
	 */
	public function addSchemeToBasket($schemeRef,$concessionId,$customerId,$giftAidResponse=null){
		if ($giftAidResponse !== null) {
			$this->setCreateGiftAidDeclaration($giftAidResponse);
		}

		return $this->api_service->addSchemeToBasket(
		    \Red61\Via\ApiRequest\apiAddSchemeToBasketRequest::create()
		        ->setSchemeRef($schemeRef)
				->setConcessionId($concessionId)
				->setCustomerId($customerId)
		);
	}

	/**
	 * Marks the basket as about to create a gift aid declaration for that customer
	 *
	 * @param boolean $createGiftAidDeclaration
	 *
	 * @return void
	 * @see    \Red61\Via\ViaApiService:setCreateGiftAidDeclaration
	 */
	public function setCreateGiftAidDeclaration($createGiftAidDeclaration) {
		$this->api_service->setCreateGiftAidDeclaration(
		    \Red61\Via\ApiRequest\apiSetCreateGiftAidDeclarationRequest::create()
		        ->setCreateGiftAidDeclaration($createGiftAidDeclaration)
		);
	}

	/**
	 * Removes a scheme from the basket.
	 *
	 * @param string  $schemeRef  The reference of the scheme to remove from the basket (of the form 'serverid:schemeid').
	 * @param integer $customerId The id of the customer associated with the scheme
	 *
	 * @return boolean boolean indicating if the remove succeeded or not.
	 * @deprecated in favour of removeSchemeFromBasket
	 */
	public function removeScheme($schemeRef,$customerId){
		trigger_error("Call to deprecated method removeScheme(), use removeSchemeFromBasket()", E_USER_NOTICE);
		return $this->removeSchemeFromBasket($schemeRef,$customerId);
	}

	/**
	 * Removes a scheme from the basket.
	 *
	 * @param String  $schemeRef  The reference of the scheme to remove from the basket (of the form 'serverid:schemeid').
	 * @param integer $customerId The id of the customer associated with the scheme
	 *
	 * @return boolean boolean indicating if the remove succeeded or not.
	 * @see    \Red61\Via\ViaApiService::removeSchemeFromBasket
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 */
	public function removeSchemeFromBasket($schemeRef,$customerId){
		return $this->api_service->removeSchemeFromBasket(
			\Red61\Via\ApiRequest\apiRemoveSchemeFromBasketRequest::create()
				->setSchemeRef($schemeRef)
				->setCustomerId($customerId)
		);
	}


	/**
	 * Returns all the vouchers with non zero credit that this customer has attached to them.
	 *
	 * @param int $customerId
	 *
	 * @return \Red61\Via\DataObject\ViaApiVoucherDetails[]
	 * @see    \Red61\Via\ViaApiService::getCustomerVouchers
	 */
	public function getCustomerVouchers($customerId){
		return $this->api_service->getCustomerVouchers(
		    \Red61\Via\ApiRequest\apiGetCustomerVouchersRequest::create()
		        ->setCustomerId($customerId)
		);
	}

	/**
	 * Returns an array of ViaApiVoucherDetails objects containing data on the voucher. Returns an empty array if there
	 * are no voucher types.
	 *
	 * @return \Red61\Via\DataObject\ViaApiVoucherType[]
	 * @see    \Red61\Via\ViaApiService::getVoucherTypes
	 */
	public function getVoucherTypes(){
		return $this->api_service->getVoucherTypes();
	}

	/**
	 * Adds a voucher to the basket, initialising the basket if necessary.
	 *
	 * @param string  $voucherTypeRef The reference of the voucher to add to the basket.
	 * @param integer $quantity       The amount of vouchers to add
	 *
	 * @return boolean boolean indicating if the add succeeded or not.
	 * @deprecated for addVoucherTypeToBasket
	 */
	public function addVoucher($voucherTypeRef,$quantity){
		trigger_error("Call to deprecated method addVoucher(), use addVoucherTypeToBasket()", E_USER_NOTICE);
		return $this->addVoucherTypeToBasket($voucherTypeRef,$quantity);
	}

	/**
	 * Adds a voucher to the basket, initialising the basket if necessary.
	 *
	 * @param string  $voucherTypeRef The reference of the voucher to add to the basket.
	 * @param integer $quantity       The amount of vouchers to add
	 *
	 * @return boolean boolean indicating if the add succeeded or not.
	 * 
	 * @throws \Red61\Via\Exception\VoucherTypeUnavailableException  If the voucher type is off sale, is not available to the be sold on the web or 
	 * is restrict to a customer and the basket has no customer attached.
	 * 
	 * @see \Red61\Via\ViaApiService::addVoucherTypeToBasket
	 */
	public function addVoucherTypeToBasket($voucherTypeRef,$quantity){
		return $this->api_service->addVoucherTypeToBasket(
		    \Red61\Via\ApiRequest\apiAddVoucherTypeToBasketRequest::create()
		        ->setVoucherTypeRef($voucherTypeRef)
				->setQuantity($quantity)
		);
	}

	/**
	 * Removes the given number of vouchers of the given type from the basket.
	 *
	 * @param string  $voucherTypeRef The reference of the voucher type to remove from the basket.
	 * @param integer $quantity       The amount of vouchers to remove
	 *
	 * @return boolean boolean indicating if the remove succeeded or not.
	 * @deprecated for removeVoucherTypeFromBasket
	 */
	public function removeVoucher($voucherTypeRef,$quantity){
		trigger_error("Call to deprecated method removeVoucher(), use removeVoucherTypeFromBasket()", E_USER_NOTICE);
		return $this->removeVoucherTypeFromBasket($voucherTypeRef,$quantity);
	}

	/**
	 * Removes the given number of vouchers of the given type from the basket.
	 *
	 * @param string  $voucherTypeRef The reference of the voucher type to remove from the basket.
	 * @param integer $quantity       The amount of vouchers to remove
	 *
	 * @return boolean boolean indicating if the remove succeeded or not.
	 * @see \Red61\Via\ViaApiService::removeVoucherTypeFromBasket
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 */
	public function removeVoucherTypeFromBasket($voucherTypeRef,$quantity){
		return $this->api_service->removeVoucherTypeFromBasket(
		    \Red61\Via\ApiRequest\apiRemoveVoucherTypeFromBasketRequest::create()
		        ->setVoucherTypeRef($voucherTypeRef)
				->setQuantity($quantity)
		);
	}

	/**
	 * Removes all vouchers types for purchasing from the basket.
	 *
	 * @return boolean boolean indicating if the remove succeeded or not.
	 * @deprecated for removeVoucherTypesFromBasket
	 */
	public function removeVouchers(){
		trigger_error("Call to deprecated method removeVouchers(), use removeVoucherTypesFromBasket()", E_USER_NOTICE);
		return $this->removeVoucherTypesFromBasket();
	}

	/**
	 * Removes all vouchers types for purchasing from the basket.
	 *
	 * @return boolean boolean indicating if the remove succeeded or not.
	 * @see    \Red61\Via\ViaApiService::removeVoucherTypesFromBasket
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 */
	public function removeVoucherTypesFromBasket(){
		return $this->api_service->removeVoucherTypesFromBasket();
	}


	/**
	 * Redeems a voucher.
	 *
	 * @param integer $customerId The id of the customer redeeming the voucher.
	 * @param string $voucherCode The code of the voucher to redeem
	 *
	 * @return boolean boolean indicating if the redeem succeeded or not.
	 * 
	 * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
	 * @throws \Red61\Via\Exception\InvalidCodeException If there was a problem with the supplied voucher code or voucher linked to the code, see the message for details. 
	 * 
	 * @see    \Red61\Via\ViaApiService::redeemVoucher
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 */
	public function redeemVoucher($customerId, $voucherCode){
		return $this->api_service->redeemVoucher(
			\Red61\Via\ApiRequest\apiRedeemVoucherRequest::create()
				->setCustomerId($customerId)
				->setVoucherCode($voucherCode)
		);
	}


	/**
	 * Removes a redeemed voucher from the basket.
	 *
	 * @param string $voucherRef The reference of the voucher to remove from the basket.
	 *
	 * @return boolean boolean indicating if the remove succeeded or not.
	 * @see    \Red61\Via\ViaApiService::removeVoucherRedeem
	 * @deprecated for removeVoucherRedeemFromBasket
	 */
	public function removeVoucherRedeem($voucherRef){
		trigger_error("Call to deprecated method removeVoucherRedeem(), use removeVoucherRedeemFromBasket()", E_USER_NOTICE);
		return $this->removeVoucherRedeemFromBasket($voucherRef);
	}

	/**
	 * Removes a redeemed voucher from the basket.
	 *
	 * @param string $voucherRef The reference of the voucher to remove from the basket.
	 *
	 * @return boolean boolean indicating if the remove succeeded or not.
	 * @see    \Red61\Via\ViaApiService::removeVoucherRedeem
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 */
	public function removeVoucherRedeemFromBasket($voucherRef){
		return $this->api_service->removeVoucherRedeemFromBasket(
			\Red61\Via\ApiRequest\apiRemoveVoucherRedeemFromBasketRequest::create()
				->setVoucherRef($voucherRef)
		);
	}


	/**
	 * Removes all redeemed vouchers from the basket.
	 *
	 * @return boolean boolean indicating if the remove succeeded or not.
	 * @deprecated for removeVoucherRedeemsFromBasket
	 */
	public function removeVoucherRedeems(){
		trigger_error("Call to deprecated method removeVoucherRedeems(), use removeVoucherRedeemsFromBasket()", E_USER_NOTICE);
		return $this->removeVoucherRedeemsFromBasket();
	}

	/**
	 * Removes all redeemed vouchers from the basket.
	 *
	 * @return boolean boolean indicating if the remove succeeded or not.
	 * @see    \Red61\Via\ViaApiService::removeVoucherRedeemsFromBasket
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 */
	public function removeVoucherRedeemsFromBasket(){
		return $this->api_service->removeVoucherRedeemsFromBasket();
	}


	/**
	 * Returns all redeemed vouchers that are currently in the basket.
	 *
	 * @return \Red61\Via\DataObject\ViaApiVoucherDetails[]
	 * @deprecated in favour of getting the details from getBasketItems response
	 */
	public function getVoucherRedeems(){
		trigger_error("Call to deprecated method getVoucherRedeems(), use getVoucherRedeemsFromBasket()", E_USER_NOTICE);
		/** @noinspection PhpDeprecationInspection */
		return $this->getVoucherRedeemsFromBasket();
	}

	/**
	 * Returns all redeemed vouchers that are currently in the basket.
	 *
	 * @return \Red61\Via\DataObject\ViaApiVoucherDetails[]
     * @deprecated in favour of getting the details from getBasketItems response
	 */
	public function getVoucherRedeemsFromBasket(){
		// @todo reimplement caching the basket items for a single request?
		return $this->getBasketItems()->getVoucherRedeems();
	}

	/**
	 * Returns an array of ViaApiMerchandiseCategory objects containing information on the Merchandise Category. Returns
	 * an empty array if there is no Merchandise categories.
	 *
	 * @return \Red61\Via\DataObject\ViaApiMerchandiseCategory[]
	 * @see    \Red61\Via\ViaApiService::getMerchandiseCategories
	 */
	public function getMerchandiseCategories(){
		return $this->api_service->getMerchandiseCategories();
	}

	/**
	 * Returns a ViaApiMerchandise object containing information on the Merchandise.
	 *
	 * @param string $merchandiseRef The reference of the merchandise to return information on (of the form 'serverid:merchandiseId').
	 *
	 * @return \Red61\Via\DataObject\ViaApiMerchandise
	 * @see    \Red61\Via\ViaApiService::getMerchandise
	 */
	public function getMerchandise($merchandiseRef){
		return $this->api_service->getMerchandise(
		    \Red61\Via\ApiRequest\apiGetMerchandiseRequest::create()
		        ->setMerchandiseRef($merchandiseRef)
		);
	}

	/**
	 * Returns an array of ViaApiMerchandise objects containing information on the Merchandise. Returns an empty array
	 * if there is no Merchandise.
	 *
	 * @param string $merchandiseCategoryRef The reference of the merchandise category (of the form 'serverid:merchandiseCategoryId').
	 *
	 * @return \Red61\Via\DataObject\ViaApiMerchandise[]
	 * @see    \Red61\Via\ViaApiService::getMerchandiseByCategory
	 */
	public function getMerchandiseByCategory($merchandiseCategoryRef){
		return $this->api_service->getMerchandiseByCategory(
		    \Red61\Via\ApiRequest\apiGetMerchandiseByCategoryRequest::create()
		        ->setMerchandiseCategoryRef($merchandiseCategoryRef)
		);
	}

	/**
	 * Adds merchandise to the basket, initialising the basket if necessary.
	 *
	 * @param string  $merchandiseVariationRef The reference of the merchandise variation to add to the basket.
	 * @param integer $quantity                The amount of merchandise to add
	 *
	 * @return boolean boolean indicating if the add succeeded or not.
	 * @deprecated for addMerchandiseToBasket
	 */
	public function addMerchandise($merchandiseVariationRef,$quantity){
		trigger_error("Call to deprecated method addMerchandise(), use addMerchandiseToBasket()", E_USER_NOTICE);
		return $this->addMerchandiseToBasket($merchandiseVariationRef,$quantity);
	}

	/**
	 * Adds merchandise to the basket, initialising the basket if necessary.
	 *
	 * @param string  $merchandiseVariationRef The reference of the merchandise variation to add to the basket.
	 * @param integer $quantity                The amount of merchandise to add
	 *
	 * @return boolean boolean indicating if the add succeeded or not.
	 * 
	 * @throws \Red61\Via\Exception\MerchandiseUnavailableException  If the merchandise is not available to the web, or is disabled
	 * 
	 * @see    \Red61\Via\ViaApiService::addMerchandiseToBasket
	 */
	public function addMerchandiseToBasket($merchandiseVariationRef,$quantity){
		return $this->api_service->addMerchandiseToBasket(
		    \Red61\Via\ApiRequest\apiAddMerchandiseToBasketRequest::create()
		        ->setMerchandiseVariationRef($merchandiseVariationRef)
				->setQty($quantity)
		);
	}

	/**
	 * Removes the given number of merchandise of the given type from the basket.
	 *
	 * @param String  $merchandiseVariationRef The reference of the merchandise to remove from the basket.
	 * @param integer $quantity                The amount of merchandise to remove
	 *
	 * @return boolean boolean indicating if the remove succeeded or not.
	 * @deprecated
	 */
	public function removeMerchandise($merchandiseVariationRef,$quantity){
		trigger_error("Call to deprecated method removeMerchandise(), use removeMerchandiseFromBasket()", E_USER_NOTICE);
		return $this->removeMerchandiseFromBasket($merchandiseVariationRef,$quantity);
	}

	/**
	 * Removes the given number of merchandise of the given type from the basket.
	 *
	 * @param String  $merchandiseVariationRef The reference of the merchandise to remove from the basket.
	 * @param integer $quantity                The amount of merchandise to remove
	 *
	 * @return boolean boolean indicating if the remove succeeded or not.
	 * @see    \Red61\Via\ViaApiService::removeMerchandiseFromBasket
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 */
	public function removeMerchandiseFromBasket($merchandiseVariationRef,$quantity){
		return $this->api_service->removeMerchandiseFromBasket(
			\Red61\Via\ApiRequest\apiRemoveMerchandiseFromBasketRequest::create()
				->setMerchandiseVariationRef($merchandiseVariationRef)
				->setQty($quantity)
		);
	}

	/**
	 * @throws BadMethodCallException
	 * @deprecated
	 * @see addDonationToBasket
	 */
	public function addDonation(){
		throw new \BadMethodCallException("Call to removed method addDonation(), use addDonationToBasket()");
	}

	/**
	 * Adds a donation to the basket, initialising the basket if necessary.
	 *
	 * @param string $donationFundRef Fund to donate to
	 * @param double $donationAmount  The amount of the donation
	 *
	 * @return boolean boolean indicating if the add succeeded or not
	 * @see    \Red61\Via\ViaApiService::addDonationToBasket
	 */
	public function addDonationToBasket($donationFundRef,$donationAmount){
		return $this->api_service->addDonationToBasket(
		    \Red61\Via\ApiRequest\apiAddDonationToBasketRequest::create()
		        ->setDonationFundref($donationFundRef)
				->setAmount($donationAmount)
		);
	}

	/**
	 * Removes the donation from the basket.
	 *
	 * @param string $donationFundRef
	 *
	 * @return boolean boolean indicating if the remove succeeded or not.
	 * @see    \Red61\Via\ViaApiService::removeDonationFromBasket
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 */
	public function removeDonationFromBasket($donationFundRef){
		return $this->api_service->removeDonationFromBasket(
			\Red61\Via\ApiRequest\apiRemoveDonationFromBasketRequest::create()
				->setDonationFundRef($donationFundRef)
		);
	}

	/**
	 * Legacy method, see getBasketDonations instead
	 *
	 * @throws BadMethodCallException
	 * @deprecated
	 */
	public function getDonationBasketItem(){
		throw new \BadMethodCallException("Call to deprecated method getDonationBasketItem(), use getBasketDonations()");
	}

	/**
	 * Returns an array of ViaApiDonationFund which is the list of available donation funds
	 *
	 * @return \Red61\Via\DataObject\ViaApiDonationFund[]
	 * @see    \Red61\Via\ViaApiService::getDonationFunds
	 */
	public function getDonationFunds() {
		return $this->api_service->getDonationFunds();
	}

	/**
	 * Return a ViaApiSchemeDetails object - this will contain an array of assocaited ViaApiSchemeTierDetails objects
	 *
	 * @param string $schemeRef The reference of the scheme, of the form serverId:SchemeId
	 *
	 * @return \Red61\Via\DataObject\ViaApiSchemeDetails
	 * 
	 * @throws \Red61\Via\Exception\SchemeUnavailableException if the scheme is not available on the web
	 * 
	 * @see    \Red61\Via\ViaApiService::getSchemeDetails
	 */
	public function getSchemeDetails($schemeRef){
		return $this->api_service->getSchemeDetails(
		    \Red61\Via\ApiRequest\apiGetSchemeDetailsRequest::create()
		        ->setSchemeRef($schemeRef)
		);
	}

	/**
	 * Returns details of all schemes a customer is a member of.
	 *
	 * @param integer $customerId The id of the customer
	 *
	 * @return \Red61\Via\DataObject\ViaApiSchemeTierDetails[]
	 * @see    \Red61\Via\ViaApiService::getCustomerSchemes
	 */
	public function getCustomerSchemes($customerId){
		return $this->api_service->getCustomerSchemes(
		    \Red61\Via\ApiRequest\apiGetCustomerSchemesRequest::create()
		        ->setCustomerId($customerId)
		);
	}

	/**
	 * Returns tier details for the selected tier
	 *
	 * @param string $subscribeRef The customer subscription reference in form serverId:schemeId:tierId
	 *
	 * @return \Red61\Via\DataObject\ViaApiSchemeTierDetails
	 * 
	 * @throws \Red61\Via\Exception\SchemeUnavailableException if the scheme or scheme tier is not available on the web
	 * 
	 * @see    \Red61\Via\ViaApiService::getSchemeTierDetails
	 */
	public function getSchemeTierDetails($subscribeRef){
		return $this->api_service->getSchemeTierDetails(
		    \Red61\Via\ApiRequest\apiGetSchemeTierDetailsRequest::create()
				->setSubscribeRef($subscribeRef)
		);
	}

	/**
	 * This is the first call that should be made to the shopping basket so at this point the shopping basket is created by calling the private method <i>_initialiseViaBasketId()</i>. This call sets the private field <i>$_basketId</i> of this class to be the unique basket id. This method then uses the authentication code set in the constructor (<i>$web_key</i>),  <i>$_basketId</i> and the parameters <i>$qty</i> and <i>$performanceId</i> to add the selected performance tickets to the shopping basket.
	 *
	 * @param array   $qtyList       This an associative array with the index of the form  <b>'$performancePriceObject->pricebandid_0'</b> if it is a full price ticket OR of the form <b>'$performancePriceObject->pricebandid_$performancePriceConcessionObject->concessionid'</b> if it is a concession ticket. The value is the quantity of that ticket. See listprices.php for an example of this.
	 * @param string  $performanceId The server id / id of the performance you wish to add tickets for.
	 * @param integer $customerId    The id of the customer requesting tickets.
	 * @param null    $promocode
	 * @param null    $seats
	 *
	 * @return boolean boolean indicating tickets have or have not been successfully added to the shopping basket
	 * @deprecated
	 */
	public function requestTickets($qtyList, $performanceId, $customerId = null, $promocode = null, $seats = null) {
		trigger_error("Call to deprecated method requestTickets(), use addTicketsToBasket()", E_USER_NOTICE);
		return $this->addTicketsToBasket($qtyList, $performanceId, $customerId, $promocode, $seats);
	}

	/**
	 * This is the first call that should be made to the shopping basket so at this point the shopping basket is created by calling the private method <i>_initialiseViaBasketId()</i>. This call sets the private field <i>$_basketId</i> of this class to be the unique basket id. This method then uses the authentication code set in the constructor (<i>$web_key</i>),  <i>$_basketId</i> and the parameters <i>$qty</i> and <i>$performanceId</i> to add the selected performance tickets to the shopping basket.
	 *
	 * @param array[]  $qtyList       This an associative array with the index of the form  <b>'$performancePriceObject->pricebandid_0'</b> if it is a full price ticket OR of the form <b>'$performancePriceObject->pricebandid_$performancePriceConcessionObject->concessionid'</b> if it is a concession ticket. The value is the quantity of that ticket. See listprices.php for an example of this.
	 * @param string   $performanceId The server id / id of the performance you wish to add tickets for.
	 * @param integer  $customerId    The id of the customer requesting tickets.
	 * @param string[] $promocode     Any promocodes to apply as array with the same keys as qtylist
	 * @param string[] $seats         The seats the customer wants, as an array of comma-separated seats keyed by pricebandId
	 *
	 * @throws Exception
	 * @return boolean boolean indicating tickets have or have not been successfully added to the shopping basket
	 * @see    \Red61\Via\ViaApiService::addTicketsToBasket
	 */
	public function addTicketsToBasket($qtyList, $performanceId, $customerId = null, $promocode = array(), $seats = array()) {
		foreach ($seats as $priceBandId => $seat_string) {
			if($seat_string) $seatIds[$priceBandId] = explode(',', $seat_string);
		}
		$total          = array_sum($qtyList);
		$ticket_list    = array();
		$preferredSeats = array();


		foreach($qtyList as $item => $quantity) {
			// Ensure quantity is always a number or 0
			$quantity = $quantity ? : 0;

			list($priceBandId,$concessionId) = explode("_",$item);

			if (isset($seatIds[$priceBandId])) {
				$pbSeatCount = count($seatIds[$priceBandId]);

				// slice the first X seats off the list (e.g might use the first 2 for concessionId=0 and the last 3 for concessionId=1)
				if($pbSeatCount < $quantity) {
					throw new Exception("Insufficient seats selected");
				}

				// @todo: What's this trying to achieve? Is it so we set the quantity from the selected seats?
				if(($concessionId == 0)&&($pbSeatCount > 0)&&($quantity == 0)&&($total == 0)) {
					$quantity = $pbSeatCount;
				}

				$preferredSeats = array_splice($seatIds[$priceBandId],0,$quantity);
			}

			if ( ! $quantity) {
				continue;
			}
			$ticket_request  = new ViaApiTicketRequest($priceBandId, $concessionId, $quantity);
			if (isset($promocode[$item])) {
				$ticket_request->setPromoCode($promocode[$item]);
			}
			if ($preferredSeats)
			{
				$ticket_request->setPerferredSeats($preferredSeats);
			}

			$ticket_list[] = $ticket_request;
		}

		return $this->api_service->addTicketsToBasket(
			\Red61\Via\ApiRequest\apiAddTicketsToBasketRequest::create()
				->setTicketsList($ticket_list)
				->setPerformanceRef($performanceId)
				->setCustomerId($customerId)
		);
	}

	public function addTicketsWithAttributesToBasket($qtyList, $performanceId, $seats = array(), $attributeId, $attributeTitle) {
		foreach ($seats as $priceBandId => $seat_string) {
			if($seat_string) $seatIds[$priceBandId] = explode(',', $seat_string);
		}
		$total          = array_sum($qtyList);
		$ticket_list    = array();
		$preferredSeats = array();


		foreach($qtyList as $item => $quantity) {
			// Ensure quantity is always a number or 0
			$quantity = $quantity ? : 0;

			list($priceBandId,$concessionId) = explode("_",$item);

			if (isset($seatIds[$priceBandId])) {
				$pbSeatCount = count($seatIds[$priceBandId]);

				// slice the first X seats off the list (e.g might use the first 2 for concessionId=0 and the last 3 for concessionId=1)
				if($pbSeatCount < $quantity) {
					throw new Exception("Insufficient seats selected");
				}

				// @todo: What's this trying to achieve? Is it so we set the quantity from the selected seats?
				if(($concessionId == 0)&&($pbSeatCount > 0)&&($quantity == 0)&&($total == 0)) {
					$quantity = $pbSeatCount;
				}

				$preferredSeats = array_splice($seatIds[$priceBandId],0,$quantity);
			}

			if ( ! $quantity) {
				continue;
			}
			$attributeSelection =  new ViaApiTicketAttributeSelection(new ViaApiTicketAttribute($attributeId, $attributeTitle), array(), false, false, false, false);
			$ticket_request  = new ViaApiTicketAttributeRequest($priceBandId, $concessionId, $quantity, $attributeSelection);

			if ($preferredSeats)
			{
				$ticket_request->setPerferredSeats($preferredSeats);
			}

			$ticket_list[] = $ticket_request;
		}

		return $this->api_service->addTicketsWithAttributesToBasket(
			\Red61\Via\ApiRequest\apiAddTicketsWithAttributesToBasketRequest::create()
				->setTicketAttributes($ticket_list)
				->setPerformanceRef($performanceId)
		);
	}

	/**
	 *
	 * @param int[]  $productIds
	 * @param string $performanceRef
	 * @param int    $priceBandId
	 * @param int    $priceBandConcessionId
	 * @param int    $newPriceBandConcessionId
	 *
	 * @return bool
	 * @deprecated
	 */
	public function changeTicketConcession($productIds, $performanceRef, $priceBandId, $priceBandConcessionId, $newPriceBandConcessionId)
	{
		trigger_error("Call to deprecated method changeTicketConcession(), use updateTicketConcession()", E_USER_NOTICE);
		return $this->updateTicketConcession($productIds, $performanceRef, $priceBandId, $priceBandConcessionId, $newPriceBandConcessionId);
	}

	/**
	 * Change the concession applied to a given set of tickets
	 *
	 * @param int[]  $productIds
	 * @param string $performanceRef
	 * @param int    $priceBandId
	 * @param int    $priceBandConcessionId
	 * @param int    $newPriceBandConcessionId
	 *
	 * @return bool
	 * @see \Red61\Via\ViaApiService::updateTicketConcession
	 */
	public function updateTicketConcession($productIds, $performanceRef, $priceBandId, $priceBandConcessionId, $newPriceBandConcessionId)
	{
		return $this->api_service->updateTicketConcession(
		    \Red61\Via\ApiRequest\apiUpdateTicketConcessionRequest::create()
		        ->setProductIds($productIds)
				->setPerformanceRef($performanceRef)
				->setPriceBandId($priceBandId)
				->setPriceBandConcessionId($priceBandConcessionId)
				->setNewPriceBandConcessionId($newPriceBandConcessionId)
		);
	}

	/**
	 * Uses the parameter unique basket id (<i>$_basketId</i>) to return an the ViaApiBasket object or NULL if no basket exists yet
	 *
	 * @return \Red61\Via\DataObject\ViaApiBasket
	 * @see    \Red61\Via\ViaApiService::getBasketItems
	 */
	public function getBasketItems() {
		return $this->api_service->getBasketItems();
	}

	/**
	 * This method is used to associate a customer with the current basket. The purpose of this is to bring customer
	 * specific ticket offers into scope so they can be applied to items in the basket. An example of this being scheme
	 * membership benefits.
	 *
	 * This call may be sent before a basket is created, and will be queued and sent if a basket is created later
	 * in the execution.
	 *
	 * @param integer $customerId The id of the customer to be associated with the current basket
	 *
	 * @return void
	 * @see    \Red61\Via\ViaApiService::setBasketCustomer
	 * @see    \Red61\Via\ApiRequest\LazyBasketMetadataSettingRequest
	 * @throws CartNotFoundException if the basket cannot be found, usually because it has expired
	 * @throws BasketChangedException When associating a customer with the current basket the basket offers are refreshed
	 *                                and this may result in different offers being applied to the items in the basket.
	 *                                If these new offers mean that the basket total has changed this exception will be
	 *                                thrown. It is advisable to refresh the basket contents if receiving
	 *                                this exception to display the new offers on the items in the basket.
	 */
	public function setBasketCustomer($customerId) {
		$this->api_service->setBasketCustomer(
			\Red61\Via\ApiRequest\apiSetBasketCustomerRequest::create()
				->setCustomerId($customerId)
		);
	}

	/**
	 * Returns the Gift Aid declaration text
	 *
	 * @return string
	 * @see    \Red61\Via\ViaApiService::getGiftAidText
	 */
	public function getGiftAidText() {
		return $this->api_service->getGiftAidText();
	}

	/**
	 * Returns an array of venue > subvenue > events information.
	 *
	 * Each BasketVenueItem object has an array of BasketSubVenueItem objects, each BasketSubVenueItem object has an
	 * array of BasketPerformanceItem objects and each BasketPerformanceItem object has an array of ticket objects.
	 * If a performance is not reserved then each ticket object represents a group of tickets identified uniquely by
	 * priceband and concession; in this case quantity (size of group) is included. If the performance is reserved
	 * then each ticket object represents one ticket and includes area/block/row/seat information; so quantity is
	 * always one. The ticket object contains a boolean flag for reserved. Use the function <i>print_r_html()</i> on
	 * the return of this method to display the nested object structure described above. If the unique basket id
	 * (<i>$_basketId</i>) is NULL or there are no tickets in the basket and empty array is returned.
	 *
	 * @return object[] array containing venues => subvenues => events
	 *
	 * @deprecated
	 */
	public function getBasketItemsSortedByVenue() {
		trigger_error("Call to deprecated method getBasketItemsSortedByVenue, use new basket format", E_USER_NOTICE);
		$items = array();
		foreach($this->getBasketItems()->getEvents() as $event) {
			$venue_id    = $event->getVenue()->getVenueid();
			$subvenue_id = $event->getSubvenue()->getSubvenueid();
			$items[$venue_id][$subvenue_id][] = $event;
		}

		$basketItems = array();
		foreach($items as $venue_id => $subvenue_array) {
			$venue_item = new stdClass;
			$venue_item->venueid = $venue_id;
			foreach ($subvenue_array as $events_array) {
				/** @var ViaApiBasketEventItem[] $events_array */
				$subvenue_item = $events_array[0]->getSubvenue();
				/** @noinspection PhpUndefinedFieldInspection */
				$subvenue_item->events = $events_array;

				$venue_item->name = $events_array[0]->getVenue()->getName();
				$venue_item->subvenues[] = $subvenue_item;
			}
			$basketItems[] = $venue_item;
		}
		return $basketItems;
	}

	/**
	 * [!!] Removed - throws BadMethodCallException
	 *
	 * @deprecated
	 * @throws \BadMethodCallException
	 */
	public function getSubVenuesForVenue() {
		throw new \BadMethodCallException("Call to removed method getSubVenuesForVenue - use new basket format");
	}

	/**
	 * [!!] Removed - throws BadMethodCallException
	 *
	 * @deprecated
	 * @throws \BadMethodCallException
	 */
	public function getEventsForSubVenue() {
		throw new \BadMethodCallException("Call to removed method getEventsForSubVenue - use new basket format");
	}

	/**
	 * Takes a BasketEventItem object and returns the array of BasketPerformanceItem objects.
	 *
	 * @param \Red61\Via\DataObject\ViaApiBasketEventItem $event
	 *
	 * @return \Red61\Via\DataObject\ViaApiBasketPerformanceItem[]
	 * @deprecated for the getPerformances getter method on the object passed in
	 */
	public function getPerformancesForEvent(\Red61\Via\DataObject\ViaApiBasketEventItem $event) {
		return $event->getPerformances();
	}

	/**
	 * Takes a BasketPerformanceItem object and returns the array of ticket objects.
	 *
	 * @param \Red61\Via\DataObject\ViaApiBasketPerformanceItem $performance
	 *
	 * @return \Red61\Via\DataObject\ViaApiProductDetails[] array of ticket objects
	 * @deprecated for the getTickets method on the object passed in
	 */
	public function getTicketsForPerformance(\Red61\Via\DataObject\ViaApiBasketPerformanceItem $performance) {
		return $performance->getTickets();
	}

	/**
	 * Takes a ticket object and returns the array of ticketConcession objects.
	 *
	 * @param \Red61\Via\DataObject\ViaApiProductDetails a ticket object
	 *
	 * @return \Red61\Via\DataObject\ViaApiProductDetails[]
	 * @deprecated - the concession details are now in each ViaApiProductDetails object
	 */
	public function getTicketConcessionCharges($ticket) {
		trigger_error("Call to deprecated method getTicketConcessionCharges(), the concession fields are in the ticket object", E_USER_NOTICE);
		/** @noinspection PhpDeprecationInspection */
		return $this->getTicketConcessions($ticket);
	}

	/**
	 * Takes a ticket object and returns the array of ticketConcession objects.
	 *
	 * @param \Red61\Via\DataObject\ViaApiProductDetails a ticket object
	 *
	 * @return \Red61\Via\DataObject\ViaApiProductDetails[]
	 * @deprecated - the concession details are now in each ViaApiProductDetails object
	 */
	public function getTicketConcessions($ticket) {
		trigger_error("Call to deprecated method getTicketConcessions(), use the concession related fields from each ticket in the event", E_USER_NOTICE);
		return array($ticket);
	}

	/**
	 * Creates an array of the individual tickets in the basket, rather than nested by event and performance as in the
	 * getBasketItems response. The ticket objects have references to their parent  event, venue, subvenue and
	 * performance mapped  as extra properties.
	 *
	 * @param \Red61\Via\DataObject\ViaApiBasket $basketItems the current basket items to group - will fetch a fresh version if not passed
	 *
	 * @return \Red61\Via\DataObject\ViaApiProductDetails[] the tickets, with extra properties
	 * @deprecated this method is likely to be removed or moved elsewhere in a future wrapper version
	 */
	public function getBasketTicketItemsFlat(ViaApiBasket $basketItems = null) {
		if($basketItems == null) {
		  $basketItems = $this->getBasketItems();
		}

		$basket = array();
		foreach($basketItems->getEvents() as $event) {
			foreach($event->getPerformances() as $performance) {
				foreach ($performance->getTickets() as $ticket) {
					/** @noinspection PhpUndefinedFieldInspection */
					$ticket->event       = $event;
					/** @noinspection PhpUndefinedFieldInspection */
					$ticket->venue       = $event->getVenue();
					/** @noinspection PhpUndefinedFieldInspection */
					$ticket->subvenue    = $event->getSubvenue();
					/** @noinspection PhpUndefinedFieldInspection */
					$ticket->performance = $performance;
					$basket[] = $ticket;
				}
			}
		}
		return($basket);
	}

	/**
	 * Uses the unique basket id (<i>_basketId</i>) to get the current sub total of the shopping basket, excluding fees.
	 * If the basket has not been created or there are no tickets in the basket it returns 0.00.
	 *
	 * [!!] This method internally caches the basket summary for the lifetime of this instance. Code that calls for
	 *      example getBasketSubTotal, addTicketsToBasket, getBasketSubTotal within a single execution will return
	 *      incorrect (out of date) values for the second getBasketSubTotal request.
	 *
	 * @return string the subtotal of the basket without the currency
	 * @see    getBasketSummary
	 * @see    \Red61\Via\ViaApiService::getBasketSummary
	 */
	public function getBasketSubTotal() {
		return $this->getBasketSummary()->getSubtotal();
	}

	/**
	 * Uses the unique basket id (<i>_basketId</i>) to get the current total of the shopping basket. If the basket has
	 * not been created or there are no tickets in the basket it returns 0.00.
	 *
	 * [!!] This method internally caches the basket summary for the lifetime of this instance. Code that calls for
	 *      example getBasketTotal, addTicketsToBasket, getBasketTotal within a single execution will return
	 *      incorrect (out of date) values for the second getBasketTotal request.
	 *
	 * @return string the total of the basket without the currency
	 * @see    getBasketSummary
	 * @see    \Red61\Via\ViaApiService::getBasketSummary
	 */
	public function getBasketTotal() {
		return $this->getBasketSummary()->getTotal();
	}

	/**
	 * Uses the unique basket id <i>$_basketId</i> to get the number of items in the shopping basket.
	 * Returns 0 if <i>$_basketId</i> is NULL.
	 *
	 * [!!] This method internally caches the basket summary for the lifetime of this instance. Code that calls for
	 *      example getNoOfItemsInBasket, addTicketsToBasket, getNoOfItemsInBasket within a single execution will return
	 *      incorrect (out of date) values for the second getNoOfItemsInBasket request.
	 *
	 * @return int number of items (tickets and miscellaneous) in the basket
	 * @see    getBasketSummary
	 * @see    \Red61\Via\ViaApiService::getBasketSummary
	 */
	public function getNoOfItemsInBasket() {
		$summary = $this->getBasketSummary();
		return ($summary->getNotickets() + $summary->getNomisc());
	}

	/**
	 * Uses the unique basket id <i>$_basketId</i> to get a basketSummary object which contains the sub total,
	 * handling fee, total and no of tickets in the basket.
	 *
	 * [!!] This method will by default swallow any SoapFault (including CartNotFound and others you may wish to handle
	 *      directly in your application. You can disable this legacy compatibility mode by passing an
	 *      array including 'always_rethrow' => TRUE to the $conf constructor option.
	 *
	 * @throws SoapFault on exception if the class was created with the always_rethrow config option set
	 * @return \Red61\Via\DataObject\ViaApiBasketSummary - either a live result or an empty object if basketId is NULL or basket has expired
	 * @see    \Red61\Via\ViaApiService::getBasketSummary
	 */
	public function getBasketSummary() {
		// @todo: Improve basket response caching to invalidate/flush when we make a basket-altering call
		if ( ! isset($this->_localData['basketSummary'])) {
			try {
				$this->_localData['basketSummary'] = $this->api_service->getBasketSummary();
			} catch (SoapFault $e) {
				if ($this->always_rethrow) {
					throw $e;
				} else {
					return new ViaApiBasketSummary;
				}
			}
		}
		return $this->_localData['basketSummary'];
	}

	/**
	 * Returns a list of the suggested donations for this basket with {@link ViaApiDonationFund#amount} set to the
	 * suggested amount.
	 *
	 * @param int $filterOption The types of donations to filter - see the apiGetBasketSuggestedDonationsRequest object for options
	 *
	 * @return \Red61\Via\DataObject\ViaApiDonationFund[]
	 * @see    \Red61\Via\ViaApiService::getBasketSuggestedDonations
	 * @see    \Red61\Via\ApiRequest\apiGetBasketSuggestedDonationsRequest::setFilterOption
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 */
	public function getBasketSuggestedDonations($filterOption) {
		return $this->api_service->getBasketSuggestedDonations(
			\Red61\Via\ApiRequest\apiGetBasketSuggestedDonationsRequest::create()
				->setFilterOption($filterOption)
		);
	}

	/**
	 * Uses the authentication key to send a confirmation email to the order referenced by order id.
	 *
	 * @param string $orderId The id of the order to send a confirmation to
	 *
	 * @return bool
	 * @see    \Red61\Via\ViaApiService::sendConfirmationEmail
	 */
	public function sendConfirmationEmail($orderId) {
		return $this->api_service->sendConfirmationEmail(
		    \Red61\Via\ApiRequest\apiSendConfirmationEmailRequest::create()
		        ->setOrderId($orderId)
		);
	}

	/**
	 * Uses the unique basket id (<i>_basketId</i>) to remove any items from the basket and optionally removes the basket.
	 *
	 * Removing the basket invalidates the current basket id and sets the field <i>_basketId</i> in this class to NULL.
	 * This method returns true only if clearing the basket and removing the basket return true. Otherwise it returns false.
	 *
	 * @param boolean $removeBasket if true then will invalidate the field _basketId in this class, if false then this
	 *                              _basketId remains valid
	 *
	 * @return boolean indicating that the basket has been cleared successfully and the basket has been removed
	 * @see    \Red61\Via\ViaApiService::clearBasket
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 */
	public function clearBasket($removeBasket = true) {
		$success = $this->api_service->clearBasket(
			\Red61\Via\ApiRequest\apiClearBasketRequest::create()
				->setRemoveBasket($removeBasket)
		);
		return $success;
	}

	/**
	 * This method uses the parameters <i>$tickets</i>, <i>$performanceId</i>, <i>$newQTY</i>, <i>$oldQTY</i>, the
	 * authentication code set in the constructor (currently <i>$roleId</i>) and <i>$_basketId</i> and the makes the
	 * decision to either (1) add more tickets for this ticket type or (2) remove some or all of the currently selected
	 * tickets or (3) do nothing as the old and new quantities are equal. This method is designed to handle single
	 * tickets or tickets grouped by ticket type. See basket.php for an example of its use.
	 *
	 * @param string  $tickets       This is the delimited string of ticket ids. This string is
	 *                               <i>ticketObject->ticketids</i> if <i>getBasketItemsSortedByVenue($fullListing)</i>
	 *                               is called with <i>$fullListing</i> set to false(ticket types grouped together).
	 *                               If <i>$fullListing</i> is true the id for a single ticket is <i>ticketObject->ticketid</i>.
	 *                               They are required if the quantity of the ticket type is to be reduced.
	 * @param string  $performanceId The id of the server/performance that these tickets are for.
	 * @param array   $newQTY        This an associative array (size 1 as this method is used for only one ticket type
	 *                               at a time) with the index of the form  <b>'$ticket->pricebandid_$ticket->priceband_concessionid'</b>.
	 *                               This combination of pricebandid and concessionid defines the unique ticket type.
	 *                               The value is the new quantity required for that ticket type. See basket.php for an
	 *                               example of this.
	 * @param integer $oldQTY        The current amount of tickets for the ticket type
	 * @param int     $customerId
	 *
	 * @return boolean boolean indicating that either add/remove tickets has succeeded or not. Returns false if <i>_basketId</i>
	 * is NULL or new and old quantity are equal.
	 * @deprecated
	 */
	public function changeTickets($tickets, $performanceId, $newQTY, $oldQTY, $customerId = null) {
		trigger_error("Call to deprecated method changeTickets(), use updateTickets()", E_USER_NOTICE);
		return $this->updateTickets($tickets, $performanceId, $newQTY, $oldQTY, $customerId);
	}

	/**
	 * This method uses the parameters <i>$tickets</i>, <i>$performanceId</i>, <i>$newQTY</i>, <i>$oldQTY</i>, the
	 * authentication code set in the constructor (currently <i>$roleId</i>) and <i>$_basketId</i> and the makes the
	 * decision to either (1) add more tickets for this ticket type or (2) remove some or all of the currently selected
	 * tickets or (3) do nothing as the old and new quantities are equal. This method is designed to handle single
	 * tickets or tickets grouped by ticket type. See basket.php for an example of its use.
	 *
	 *
	 * @param string  $tickets       This is the delimited string of ticket ids. This string is <i>ticketObject->ticketids</i>
	 *                               if <i>getBasketItemsSortedByVenue($fullListing)</i> is called with <i>$fullListing</i>
	 *                               set to false(ticket types grouped together). If <i>$fullListing</i> is true the id
	 *                               for a single ticket is <i>ticketObject->ticketid</i>. They are required if the quantity
	 *                               of the ticket type is to be reduced.
	 * @param string  $performanceId The id of the server/performance that these tickets are for.
	 * @param array   $newQTY        This an associative array (size 1 as this method is used for only one ticket type
	 *                               at a time) with the index of the form  <b>'$ticket->pricebandid_$ticket->priceband_concessionid'</b>.
	 *                               This combination of pricebandid and concessionid defines the unique ticket type. The
	 *                               value is the new quantity required for that ticket type. See basket.php for an
	 *                               example of this.
	 * @param integer $oldQTY        The current amount of tickets for the ticket type
	 * @param int     $customerId
	 *
	 * @throws InvalidArgumentException if passed anything other than exactly one price/qty pair in $newQTY
	 * @return boolean boolean indicating that either add/remove tickets has succeeded or not. Returns false if <i>_basketId</i> is NULL or new and old quantity are equal.
	 * @see    \Red61\Via\ViaApiService::addTicketsToBasket
	 * @see    \Red61\Via\ViaApiService::removeTicket
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 */
	public function updateTickets($tickets, $performanceId, $newQTY, $oldQTY, $customerId = null) {
		if (count($newQTY) !== 1) {
			throw new \InvalidArgumentException(__CLASS__.'::updateTickets expects exactly 1 price/quantity pair in newQTY');
		}
		reset($newQTY);
		$item     = key($newQTY);
		$qty      = current($newQTY) ? : 0;

		if ($qty > $oldQTY) {
			// increase the amount of this ticket type (defined by pricebandid and concessionid)
			$qty = $qty - $oldQTY;
			// add new tickets to the basket
			return $this->addTicketsToBasket(
				array($item => $qty),
				$performanceId,
				$customerId
			);
		} else if ($qty < $oldQTY) {
			//remove some of the current tickets using the tickets var
			$remove = $oldQTY - $qty;
			$ticketIDs = explode("|",$tickets);
			list($priceBandId, $concessionId) = explode('_', $item);
			return $this->api_service->removeTicket(
				\Red61\Via\ApiRequest\apiRemoveTicketRequest::create()
					->setProductIds(array_slice($ticketIDs, (count($ticketIDs) - $remove)))
					->setPerformanceRef($performanceId)
					->setPriceBandId($priceBandId)
					->setPriceBandConcessionId($concessionId)
			);
		}	else {
			// do nothing
			return false;
		}
	}

	/**
	 * A utility method to calculate the price of <i>$amount</i> of tickets priced <i>$price</i>.
	 * @param integer $amount Amount of the ticket
	 * @param double $price Price of the ticket
	 * @return string the total for this amount of tickets priced <i>$price</i>
	 */
	public function getTicketTotal($amount, $price) {
		//@todo why is getTicketTotal's parameter multiplication conditional on whether the user has a basket
		if ($this->getBasketId() != NULL) {
			$result = sprintf("%01.2f", $amount * $price);
		} else {
			return 0;
		}
		return $result;
	}

	/**
	 * Uses the unique basket id <i>_basketId</i> to clear and remove the basket and sets <i>_basketId</i> to NULL.
	 *
	 * @return boolean boolean indicating whether this procedure was successful
	 * @deprecated for clearBasket
	 * @see clearBasket
	 */
	public function removeBasket() {
		return $this->clearBasket(true);
	}

	/**
	 * This method takes the parameters <i>$email</i> and <i>$password</i> and checks to see if a customer record
	 * exists with these identifiers. Returns the customer id if there is only one record in the db that matches the
	 * <i>$email</i> and <i>$password</i>. Returns 0 to indicate failure to log in.
	 *
	 * @param string $email    The email of the customer.
	 * @param string $password The password of the customer.
	 *
	 * @return integer integer - customer id, 0 indicates failure
	 * 
	 * @throws \Red61\Via\Exception\LoginDisabledException if the customer has had their ability to login disabled
	 * 
	 * @see    \Red61\Via\ViaApiService::login
	 */
	public function login($email, $password) {
		return $this->api_service->login(
		    \Red61\Via\ApiRequest\apiLoginRequest::create()
		        ->setEmail($email)
				->setPassword($password)
		);
	}

	/**
	 * Uses the parameter <i>$customerId</i> to get the full name of the customer. Returns an empty string to indicate
	 * failure.
	 *
	 * @param integer $customerId The id of a customer, usually returned by logging in or by creating an account.
	 *
	 * @return string customer title, forename and surname
	 * @see    \Red61\Via\ViaApiService::getCustomerName
	 */
	public function getCustomerName($customerId) {
		return $this->api_service->getCustomerName(
		    \Red61\Via\ApiRequest\apiGetCustomerNameRequest::create()
		        ->setCustomerId($customerId)
		);
	}

	/**
	 * Uses the authentication code set in the constructor (<i>$web_key</i>) to get an array of ListElement objects.
	 * A listElement object contains two fields, <i>integer id</i> and <i>string title</i>. In this case the id is the
	 * venueid and the title is the name of the venue. See /includes/filter.php. Returns an empty array if there are no
	 * venues.
	 *
	 * @return \Red61\Via\DataObject\ViaApiListElement[]
	 */
	public function getVenues() {
		$venues = $this->api_service->getVenues();
		usort($venues,"sort_ViaApiListElement");
		return $venues;
	}

	/**
	 * Uses the authentication code set in the constructor (<i>$web_key</i>) to get an array of ListElement objects.
	 * A listElement object contains two fields, <i>integer id</i> and <i>string title</i>. In this case the id is the
	 * subvenueid and the title is the name of the subvenue and the venue name. See /includes/filter.php.
	 * Returns an empty array if there are no subvenues.
	 *
	 * @param string $venueRef optional venueRef filter (serverId:VenueId) If not supplied then will list all sub venues.
	 *
	 * @return \Red61\Via\DataObject\ViaApiListElement[]
	 */
	public function getSubVenues($venueRef="0:0") {
		$subvenues = $this->api_service->getSubVenues(
		    \Red61\Via\ApiRequest\apiGetSubVenuesRequest::create()
		        ->setVenueRef($venueRef)
		);
		usort($subvenues,"sort_ViaApiListElement");
		return $subvenues;
	}

	/**
	 * Return a Venue Details object
	 *
	 * @param string $venueRef The reference of the venue, of the form serverId:VenueId
	 *
	 * @return \Red61\Via\DataObject\ViaApiVenueDetails
	 */
	public function getVenueDetails($venueRef){
		return $this->api_service->getVenueDetails(
		    \Red61\Via\ApiRequest\apiGetVenueDetailsRequest::create()
		        ->setVenueRef($venueRef)
		);
	}

	/**
	 * Return a SubVenue Details object
	 *
	 * @param string $subVenueRef The reference of the subVenue, of the form serverId:SubVenueId
	 *
	 * @return \Red61\Via\DataObject\ViaApiSubVenueDetails
	 */
	public function getSubVenueDetails($subVenueRef){
		return $this->api_service->getSubVenueDetails(
		    \Red61\Via\ApiRequest\apiGetSubVenueDetailsRequest::create()
		        ->setSubVenueRef($subVenueRef)
		);
	}

	/**
	 * Returns an array of ListElement objects. A listElement object contains two fields, <i>integer id</i> and
	 * <i>string title</i>. In this case the id is the schemeid and the title is the name of the scheme. Returns
	 * an empty array if there are no schemes.
	 *
	 * @return \Red61\Via\DataObject\ViaApiListElement[]
	 */
	public function getSchemes(){
		return $this->api_service->getSchemes();
	}

	/**
	 * Returns an array of genreListElement objects. A genreListElement object contains four fields,
	 * <i>integer typeid</i>, <i>integer subtypeid</i>, <i>string typetitle</i> and <i>string subtypetitle</i>.
	 * See /includes/filter.php. Returns an empty array if there are no types.
	 *
	 * @return \Red61\Via\DataObject\ViaApiListElement[]
	 * @deprecated for getEventTypes
	 */
	public function getGenres() {
		trigger_error("Call to deprecated method getGenres(), use getEventTypes()", E_USER_NOTICE);
		return $this->getEventTypes();
	}

	/**
	 * Returns an array of genreListElement objects. A genreListElement object contains four fields,
	 * <i>integer typeid</i>, <i>integer subtypeid</i>, <i>string typetitle</i> and <i>string subtypetitle</i>.
	 * See /includes/filter.php. Returns an empty array if there are no types.
	 *
	 * @return \Red61\Via\DataObject\ViaApiListElement[]
	 */
	public function getEventTypes() {
		return $this->api_service->getEventTypes();
	}

	/**
	 * Returns an array of ListElement objects. A listElement object contains two fields, <i>integer id</i> and
	 * <i>string title</i>. In this case the id is the artist/company id and the title is the name of the
	 * artist/company. See /includes/filter.php. Returns an empty array if there are no artists.
	 *
	 * @return \Red61\Via\DataObject\ViaApiListElement[]
	 */
	public function getArtists() {
		return $this->api_service->getArtists();
	}

	/**
	 * This method uses the authentication code set in the constructor (<i>$web_key</i>) and the parameters
	 * <i>$venueId</i>, <i>$subvenueid</i>, <i>$typeids</i>, <i>$companyid</i> and <i>$date</i> and returns an array of
	 * event objects filtered by the parameters that have been set. See /includes/filter.php. If there are no events
	 * then an empty array is returned.
	 *
	 * @param string $venueid    The id of the server:venue.
	 * @param string $subvenueid The id of the server:subvenue.
	 * @param string $typeids    The id of the event type (in form of 'typeid_subtypeid').
	 * @param string $companyid  The id of the server:artist/company.
	 * @param string $start      either LEGACY: the time period, current allowed values are "today", "tomorrow", "thisweek", "nextweek", "thismonth" and "nextmonth". See /includes/filter.php OR the date and start time of a performance for an event
	 * @param string $end        date and start time of a performance of an event. In combination with <i>$start</i> creates a datetime window for the start datetimes of event performances.
	 * @param string $seasonCode can be used to filter results by season
	 *
	 * @return \Red61\Via\DataObject\ViaApiEvent[]
	 * @see    \Red61\Via\ViaApiService::filterEvents
	 */
	public function filterEvents($venueid,$subvenueid,$typeids,$companyid,$start,$end="LEGACY", $seasonCode) {
		if (!empty($typeids)) {
			list ($typeid, $subtypeid) = explode("_", $typeids);
		} else {
			$typeid = 0;
			$subtypeid = 0;
		}
		$venueid = $venueid ? : "";
		$subvenueid = $subvenueid ? : "";
		$companyid = $companyid ? : "";

		//@todo: review this legacy code implementation

		if($end == "LEGACY") {
			// must be legacy - if $start not empty then $start will be string:

			/**
			1. today
			2. tomorrow
			3. week
			4. nextweek
			5. month
			6. nextmonth
			 */

			$end = "";

			if ($start == "today") {

				$start = mktime(0,0,0);
				$end = mktime(24,0,0);
			} else if ($start == "tomorrow") {
				$start = mktime(0,0,0,date("m"),date("d")+1);
				$end = mktime(24,0,0,date("m"),date("d")+1);
			}	else if ($start == "week") {
				$start = mktime(0,0,0,date("m"),date("d")-date("w")+1);
				$end =  mktime(0,0,0,date("m"),date("d")-date("w")+8);
			}else if ($start == "nextweek") {
				$start = mktime(0,0,0,date("m"),date("d")-date("w")+8);
				$end = mktime(0,0,0,date("m"),date("d")-date("w")+15);
			}else if ($start == "month") {
				$start = mktime(0,0,0,date("m"),1);
				$end = mktime(0,0,0,date("m")+1,1);
			}else if ($start == "nextmonth") {
				$start = mktime(0,0,0,date("m")+1,1);
				$end = mktime(0,0,0,date("m")+2,1);
			}
			else{
				if($start) {
					$start = strtotime($start);
					$end = $start + (3600 * 24);
				}
			}
		}
		else {
			if($start) $start = strtotime($start);
			if($end) $end = strtotime($end);
		}

		$start_datetime = $start ? date("Y-m-d H:i",$start) : "";
		$end_datetime = $end ? date("Y-m-d H:i",$end) : "";

		return $this->api_service->filterEvents(
		    \Red61\Via\ApiRequest\apiFilterEventsRequest::create()
		        ->setVenueRef($venueid)
				->setSubVenueRef($subvenueid)
				->setTypeId($typeid)
				->setSubTypeId($subtypeid)
				->setCompanyRef($companyid)
				->setStartDateTime($start_datetime)
				->setEndDateTime($end_datetime)
				->setSeasonCode($seasonCode)
		);
	}


	/**
	 * This method uses the authentication code set in the constructor (<i>$web_key</i>) and the parameters
	 * <i>$venueId</i>, <i>$subvenueid</i>, <i>$typeids</i> and <i>$timeinterval</i> and returns an array of
	 * event objects filtered by the parameters that have been set. See /includes/filter.php. If there are no
	 * events then an empty array is returned.
	 *
	 * @param string $venueids     An array of the server:venue.
	 * @param string $subvenueids  An array of the server:subvenue.
	 * @param string $typeids      An array of the event types (in form of 'typeid_subtypeid').
	 * @param string $timeinterval The times in which to search in the form starthour:endhour 24 hour based 0-23
	 *
	 * @return \Red61\Via\DataObject\ViaApiEvent[]
	 */
	public function filterEventsByTime($venueids,$subvenueids,$typeids,$timeinterval) {
		//Build up arrays of types and sub types
		$typeid = array();
		$subtypeid = array();
		if(is_array($typeids)){
			foreach($typeids as $genre){
				list($type, $subtype) = explode("_", $genre);
				$typeid[] = $type;
				if($subtype != 0){
					$subtypeid[] = $subtype;
				}
			}
		}

		//Get the start and end hour
		list($starthour, $endhour) = explode(":",$timeinterval);

		return $this->api_service->filterEventsByTime(
		    \Red61\Via\ApiRequest\apiFilterEventsByTimeRequest::create()
				->setStartHour($starthour)
				->setEndHour($endhour)
				->setVenueRefs($venueids ? : array())
				->setSubvenueRefs($subvenueids ? : array())
				->setTypeIds($typeid)
				->setSubTypeIds($subtypeid)
		);
	}

	/**
	 * Returns the cards saved against this customer/
	 *
	 * @param int $customerId
	 *
	 * @return \Red61\Via\DataObject\ViaApiSavedCardDetails[]
	 * @see    \Red61\Via\ViaApiService::getCustomerSavedCards
	 */
	public function getCustomerSavedCards($customerId) {
		return $this->api_service->getCustomerSavedCards(
		    \Red61\Via\ApiRequest\apiGetCustomerSavedCardsRequest::create()
		        ->setCustomerId($customerId)
		);
	}

	/**
	 * Creates the entire order : order record, payment record, line item records and processes payment.
	 *
	 * Processes the payment using the relevant Payment Processor linked to the WEB location. If the payment is taken
	 * sucessfully and the order is completed this method call removes the basket thus invalidating the unique basket id
	 * so it sets the field <i>$_basketId</i> to NULL.
	 *
     * @param string  $cc_name         This is the name on the card.
	 * @param string  $cc_number       This is the card number.
	 * @param string  $start_date      This is the start date of the card if it has one, in the form MM/YY.
	 * @param string  $expiry_date     This is the expiry date of the card, in the form MM/YY.
	 * @param string  $issue           This is the issue number of the card.
	 * @param string  $cv2             This the security number on the back of the card.
	 * @param string  $workstationId   This is the web server host name.
	 * @param integer $deliveryOption  How these tickets should be delivered. 0 = COBO, 1 = Post to Primary address, 2 = Post to Alt address
	 * @param integer $customerId      this is the id of the customer who is making the purchase.
	 * @param bool    $saveCardDetails whether to save card for future purchases
	 *
	 * @return \Red61\Via\DataObject\ViaApiOrderDetails $orderDetails->code is the transactionId in the form 'serverId:orderId'.
	 * @see    \Red61\Via\ViaApiClient::createOrder
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 *
	 * Use the entire string as the transactionId. If the orderId part is < 1 then an error has occurred,
	 * see $orderDetails->statusmessage for details. If an error has occurred this method can be called again to try
	 * again. If orderId part is -2 then some of the tickets in this order have been set off sale and removed from the
	 * basket, the basket should therefore be refreshed. If orderId part = '-302' then extra 3D secure authorisation is
	 * required. $orderDetails->statusmessage provides the acs URL to redirect to as well as the MD and paReq fields
	 * which must be passed to the acs URL. The statusmessage field will be in the form 'acs_url?MD=*****&paReq=*******'.
	 * See checkout.php for an example of this redirection. The return from the acs URL (MD and paRes) must then be
	 * passed to send3DsecureReturn() to complete the 3D secure transaction.
	 */
	public function createOrder($cc_name,$cc_number,$start_date,$expiry_date,$issue,$cv2,$workstationId,$deliveryOption,$customerId,$saveCardDetails=false) {
		$browserType = 0; // Normal PC (1 for wap)

		return $this->api_service->createOrder(
			\Red61\Via\ApiRequest\apiCreateOrderRequest::create()
				->setCcName($cc_name)
				->setCcNumber($cc_number)
				->setStartDate($start_date)
				->setExpiryDate($expiry_date)
				->setIssue($issue)
				->setCv2($cv2)
				->setWorkstationId($workstationId)
				->setDeliveryOption($deliveryOption)
				->setCustomerId($customerId)
				->setBrowserType($browserType)
				->setUserAgent(isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : 'Unknown')
				->setAcceptHeaders(isset($_SERVER['HTTP_ACCEPT']) ? $_SERVER['HTTP_ACCEPT']     : '*')
				->setSaveCardDetails($saveCardDetails)
		);
	}

	/**
	 * Creates the entire initial unpaid order : order record, line items and 'unpaid' payment record.  If this does not
	 * return a valid payment Id (greater than ZERO) then DO NOT proceed to take payment. If this method returns a valid
	 * paymentId then it is safe to proceed to take payment. The result of that payment attempt must be passed to
	 * completeOrder() to complete the order. If the payment  fails or is declined then still call completeOrder()
	 * and to try again this method can be called again and will persist a new unpaid payment against the original
	 * order(created first time round) for processing. Remember to only proceed to take payment if this method returns
	 * a valid paymentId (greater than ZERO).
	 *
	 * @access public
	 *
	 * @param string  $cc_name        This is the name on the card.
	 * @param string  $cc_number      This is the card number.
	 * @param string  $start_date     This is the start date of the card if it has one, in the form MM/YY.
	 * @param string  $expiry_date    This is the expiry date of the card, in the form MM/YY.
	 * @param string  $issue          This is the issue number of the card.
	 * @param string  $workstationId  This is the web server host name.
	 * @param integer $deliveryOption How these tickets should be delivered. 0 = COBO, 1 = Post to Primary address, 2 = Post to Alt address
	 * @param integer $customerId     this is the id of the customer who is making the purchase.
	 *
	 * @return \Red61\Via\DataObject\ViaApiOrderDetails orderDetails object or NULL if <i>$_basketId</i> is NULL.
	 * @see    \Red61\Via\ViaApiClient::createInitialUnpaidOrder
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 *
	 * $orderDetails->code is the 'serverId:paymentId'. The paymentId part is required to pass to completeOrder()
	 * once the payment has been processed independently. If paymentId < 1 then an error has occurred, see
	 * $orderDetails->statusmessage for details and DO NOT proceed to take payment. Instead call this function again.
	 * VERY IMPORTANT: Only proceed to take payment if the paymentId > 0 and, this paymentId must be passed to
	 * completeOrder() once the payment has been processed indendently. If paymentId is -2 then some of the tickets in
	 * this order have been set off sale and removed from the basket, the basket should therefore be refreshed.
	 */
	public function createInitialUnpaidOrder($cc_name,$cc_number,$start_date,$expiry_date,$issue,$workstationId,$deliveryOption,$customerId) {
		// $paymentDetails->code :
		// '1003:45' => success example paymentId=45, use '45' as paymentId in call to completeOrder()
		// '1003:-1' specifically => server error see $paymentDetails->statusmessage for details
		return $this->api_service->createInitialUnpaidOrder(
			\Red61\Via\ApiRequest\apiCreateInitialUnpaidOrderRequest::create()
				->setCcName($cc_name)
				->setCcNumber($cc_number)
				->setStartDate($start_date)
				->setExpiryDate($expiry_date)
				->setIssue($issue)
				->setWorkstationId($workstationId)
				->setDeliveryOption($deliveryOption)
				->setCustomerId($customerId)
		);
	}

	/**
	 * This function persists the payment information to the database, and completes the order depending
	 * on whether the payment was taken successfully. If payment was taken successfully all remote inventory servers in the order are dispatched with payment and order information. If the payment did not succeed then returning code will be a paymentError allowing another payment to be attempted by starting the whole process again by calling createInitialUnpaidOrder() -> process payment independently -> completeOrder().
	 *
	 * @param integer $paymentId                The paymentId (>0) returned by createInitialUnpaidOrder().
	 * @param boolean $paymentSuccess           whether the payment processed independently was successful or not.
	 * @param string  $requestId                Unique identifier for a transaction communication (if available).
	 * @param string  $terminalId               The terminal Id of the transaction (if available).
	 * @param string  $merchantId               The merchant Id supplied by the bank (if available).
	 * @param string  $stan                     The sequence transaction number (if available).
	 * @param string  $authCode                 The auth code identifier  for a transaction.
	 * @param string  $receiptId                Unique identifier for a transaction communication - see <i>$requestId</i> (if available).
	 * @param string  $paymentTrackingReference the tracking reference used or returned from the independent payment processor, if not set '<i>$_basketId|$paymentId</i>' will be used.
	 *
	 * @return \Red61\Via\DataObject\ViaApiOrderDetails orderDetails object or NULL if <i>$_basketId</i> is NULL.
	 * @see    \Red61\Via\ViaApiClient::completeOrder
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 *
	 * $orderDetails->code is the transactionId in the form 'serverId:orderId'. Use the entire string as the transactionId.
	 * If the orderId part is < 1 then an error has occurred, see $orderDetails->statusmessage for details. If an error
	 * has occurred start the whole process again by calling createInitialUnpaidOrder() -> process payment
	 * independently -> completeOrder().
	 */
	public function completeOrder($paymentId, $paymentSuccess, $requestId, $terminalId, $merchantId, $stan, $authCode, $receiptId, $paymentTrackingReference) {
		return $this->api_service->completeOrder(
			\Red61\Via\ApiRequest\apiCompleteOrderRequest::create()
				->setPaymentId($paymentId)
				->setPaymentSuccess($paymentSuccess)
				->setRequestId($requestId)
				->setTerminalId($terminalId)
				->setMerchantId($merchantId)
				->setStan($stan)
				->setAuthCode($authCode)
				->setReceiptId($receiptId)
				->setPaymentProcessorTrackingValue($paymentTrackingReference)
		);
	}

	/**
	 * Creates the entire unpaid account order : order record, line items and 'unpaid' account payment record.
	 *
	 * If this  does not return a valid transaction Id (greater than ZERO) then DO NOT proceed to take payment. If this
	 * method returns a valid transactionId then it is safe to proceed to take payment. The result of that payment attempt
	 * must be passed to completeAccountOrder() to complete the order. If the payment  fails or is declined then still
	 * call completeAccountOrder() and to try again this method can be called again and will persist a new unpaid account
	 * payment against the original order(created first time round) for processing. Remember to only proceed to take
	 * payment if this method returns a valid transactionId (greater than ZERO).
	 *
	 * @param string  $accountCode    The unique account code
	 * @param string  $workstationId  This is the web server host name.
	 * @param integer $deliveryOption How these tickets should be delivered. 0 = COBO, 1 = Post to Primary address, 2 = Post to Alt address, 3 = Print Option NOW
	 * @param integer $customerId     this is the id of the customer who is making the purchase.
	 *
	 * @return \Red61\Via\DataObject\ViaApiOrderDetails orderDetails object or NULL if <i>$_basketId</i> is NULL.
	 * @see    \Red61\Via\ViaApiClient::createUnpaidAccountOrder
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 * $orderDetails->code is the 'serverId:transactionId'. If transactionId < 1 then an error has occurred, see
	 * $orderDetails->statusmessage for details and DO NOT proceed to take payment. Instead call this function again.
	 *
	 * VERY IMPORTANT: Only proceed to take payment if the transactionId > 0. If transactionId is -2 then some of the
	 * tickets in this order have been set off sale and removed from the basket, the basket should therefore be refreshed.
	 */
	public function createUnpaidAccountOrder($accountCode, $workstationId, $deliveryOption, $customerId) {
		return $this->api_service->createUnpaidAccountOrder(
			\Red61\Via\ApiRequest\apiCreateUnpaidAccountOrderRequest::create()
				->setAccountCode($accountCode)
				->setWorkstationId($workstationId)
				->setDeliveryOption($deliveryOption)
				->setCustomerId($customerId)
		);
	}

	/**
	 * This function persists the payment information to the database, and completes the account order depending
	 * on whether the payment was taken successfully. If payment was taken successfully all remote inventory servers
	 * in the order are dispatched with payment and order information. If the payment did not succeed then returning
	 * code will be a paymentError allowing another payment to be attempted by starting the whole process again by
	 * calling createUnpaidAccountOrder() -> process payment independently -> completeAccountOrder().
	 *
	 * @param string $paymentTrackingReference the tracking reference used or returned from the independent payment
	 *                                         processor. Used as the account payment reference
	 * @param boolean $paymentSuccess           whether the payment processed independently was successful or not.
	 *
	 * @return \Red61\Via\DataObject\ViaApiOrderDetails orderDetails object or NULL if <i>$_basketId</i> is NULL.
	 * @see    \Red61\Via\ViaApiClient::completeAccountOrder
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 *
	 * $orderDetails->code is the transactionId in the form 'serverId:orderId'. Use the entire string as the transactionId.
	 * If the orderId part is < 1 then an error has occurred, see $orderDetails->statusmessage for details. If an error
	 * has occurred start the whole process again by calling createUnpaidAccountOrder() -> process payment
	 * independently -> completeAccountOrder().
	 */
	public function completeAccountOrder($paymentTrackingReference, $paymentSuccess) {
		return $this->api_service->completeAccountOrder(
			\Red61\Via\ApiRequest\apiCompleteAccountOrderRequest::create()
				->setPaymentProcessorTrackingValue($paymentTrackingReference)
				->setPaymentSuccess($paymentSuccess)
		);
	}

	/**
	 * Creates the entire order : order record, line item records. If the resevation is successful this method call
	 * removes the basket thus invalidating the unique basket id so it sets the field <i>$_basketId</i> to NULL.
	 *
	 * @param integer $customerId this is the id of the customer who is making the reservation.
	 *
	 * @return \Red61\Via\DataObject\ViaApiOrderDetails orderDetails object or NULL if <i>$_basketId</i> is NULL.
	 * @see    \Red61\Via\ViaApiClient::createReservation
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 *
	 * $orderDetails->code is the transactionId in the form 'serverId:orderId'. Use the entire string as the transactionId.
	 * If the orderId part is < 1 then an error has occurred, see $orderDetails->statusmessage for details. If an error
	 * has occurred this method can be called again to try again. If orderId part is -2 then some of the tickets in this
	 * order have been set off sale and removed from the basket, the basket should therefore be refreshed.
	 */
	public function createReservation($customerId) {
		return $this->api_service->createReservation(
			\Red61\Via\ApiRequest\apiCreateReservationRequest::create()
				->setCustomerId($customerId)
		);
	}

	/**
	 * Creates an order using the supplied repeat payment token
	 *
	 * @param string $repeatPaymentToken the repeat token
	 * @param string $cv2                the optional CV2 token
	 * @param string $workstationId
	 * @param int    $deliveryOption
	 * @param int    $customerId
	 *
	 * @return \Red61\Via\DataObject\ViaApiOrderDetails orderDetails object or NULL if <i>$_basketId</i> is NULL.
	 * @see    \Red61\Via\ViaApiClient::createRepeatPaymentOrder
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 */
	public function createRepeatPaymentOrder($repeatPaymentToken, $cv2, $workstationId, $deliveryOption, $customerId) {
		$browserType = 0; // Normal PC (1 for wap)

		return $this->api_service->createRepeatPaymentOrder(
		    \Red61\Via\ApiRequest\apiCreateRepeatPaymentOrderRequest::create()
				->setRepeatPaymentToken($repeatPaymentToken)
				->setCv2($cv2)
				->setWorkstationId($workstationId)
				->setDeliveryOption($deliveryOption)
				->setCustomerId($customerId)
				->setBrowserType($browserType)
				->setUserAgent(isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : 'Unknown')
				->setAcceptHeaders(isset($_SERVER['HTTP_ACCEPT']) ? $_SERVER['HTTP_ACCEPT']     : '*')
		);
	}

	/**
	 * Sets the affiliate to be stored with the sale.
	 *
	 * May be called before a basket is created, and will be internally deferred and resent once the basket is created
	 * if required.
	 *
	 * @param string $code the affiliate code
	 * @param string $token the affiliate token
	 *
	 * @return void
	 * @see \Red61\Via\ViaApiService::setAffiliate
	 * @see \Red61\Via\ApiRequest\LazyBasketMetadataSettingRequest
	 *
	 */
	public function setAffiliate($code, $token) {
		$this->api_service->setAffiliate(
		    \Red61\Via\ApiRequest\apiSetAffiliateRequest::create()
		        ->setAffiliateCode($code)
				->setAffiliateToken($token)
		);
	}

	/**
	 * Sets an id which uniquely identifies the referring campaign and the customer the campaign was sent to.
	 *
	 * May be called before a basket is created, and will be internally deferred and resent once the basket is created
	 * if required.
	 *
	 * @param integer $campaignTrackingId
	 * @param string  $campaignTrackingFields
	 *
	 * @return void
	 * @see \Red61\Via\ViaApiService::setAffiliate
	 * @see \Red61\Via\ApiRequest\LazyBasketMetadataSettingRequest
	 */
	public function setCampaignTrackingId($campaignTrackingId, $campaignTrackingFields=null) {
		$this->api_service->setCampaignTrackingId(
		    \Red61\Via\ApiRequest\apiSetCampaignTrackingIdRequest::create()
		        ->setCampaignTrackingId($campaignTrackingId)
				->setCampaignTrackingFields($campaignTrackingFields)
		);
	}

	/**
	 * This function is called after createOrder() if createOrder() came back with a redirect url to complete the 3-D
	 * secure process(Maestro Cards only). The return from the re direct page will supply <i>$md</i> and <i>$paRes</i>
	 * and this call will complete the 3-D secure process. If the payment is declined then $orderDetails->code will
	 * flag this. See return. If the payment was accepted this call completes the order creation process and sets
	 * the <i>$this->_basketId</i> to NULL.
	 *
	 * @param integer $customerId this is the id of the customer who is making the purchase.
	 * @param string  $md         3D secure field - MD Unique reference used by all parties including SECPay and the
	 *                            merchant to identify and track the 3-D Secure transaction.
	 * @param string  $paRes      3D secure field - PARes Payer Authentication Response. 3-D Secure Protocol
	 *                            message type. This is the response returned after submitting a PAReq.
	 *                            It indicates whether or not the cardholder passed 3-D Secure authentication.
	 *
	 * @return \Red61\Via\DataObject\ViaApiOrderDetails orderDetails object or NULL if <i>$_basketId</i> is NULL.
	 * @see    \Red61\Via\ViaApiClient::send3DSecureReturn
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 *
	 * $orderDetails->code is the transactionId in the form 'serverId:orderId'. Use the entire string as the
	 * transactionId. If the orderId part is < 1 then an error has occurred, see $orderDetails->statusmessage for
	 * details. If an error has occurred start the whole process again by calling createOrder() -> process 3D secure
	 * payment independently using return from re direct -> send3DSecureReturn().
	 */
	public function send3DSecureReturn(/** @noinspection PhpUnusedParameterInspection */
		$customerId, $md, $paRes) {
		return $this->api_service->send3DSecureReturn(
		    \Red61\Via\ApiRequest\apiSend3DSecureReturnRequest::create()
		        ->setMd($md)
				->setPaRes($paRes)
		);
	}

	/**
	 * Takes the parameter <i>$orderId</i> and returns a purchaseSummary object. This returns a brief summary of the
	 * order that the VIA system has for this order id. If <i>$orderId</i> is 0 then NULL is returned.
	 *
	 * @param string $orderId    the id of the order for which a summary is required.
	 * @param string $orderTitle the order reference of the order for which a summary is required.
	 *
	 * @throws Red61\Via\Exception\ClientInvalidRequestException
	 * @return \Red61\Via\DataObject\ViaApiPurchaseSummary
	 * @see    \Red61\Via\ViaApiClient::getOrderSummary
	 */
	public function getOrderSummary($orderId,$orderTitle) {
		$request = \Red61\Via\ApiRequest\apiGetOrderSummaryRequest::create()->setOrderId($orderId);
		$summary = $this->api_service->getOrderSummary($request);

		// Add a layer of basic security - to make it harder to accidentally access another customer's order confirmation
		if ($summary->getOrdertitle() !== $orderTitle) {
			throw new ClientInvalidRequestException($request, array('orderTitle' => 'Invalid transaction ID and transaction Reference'));
		}
		return $summary;
	}

	/**
	 * Gets details about the tickets in a given order.
	 *
	 * @param string $orderId 'serverId:orderId' of the order to lookup tickets for.
	 *
	 * @throws Red61\Via\Exception\ClientInvalidRequestException
	 * @return \Red61\Via\DataObject\ViaApiTicketDetails[]
	 * @see    \Red61\Via\ViaApiService::getOrderTicketDetails
	 */
	public function getOrderTicketDetails($orderId)
	{
		return $this->api_service->getOrderTicketDetails(
		    \Red61\Via\ApiRequest\apiGetOrderTicketDetailsRequest::create()
		        ->setOrderId($orderId)
		);
	}

	/**
	 * Given a particular customer returns an array of this customer's order ids
	 *
	 * @param integer $customerId    the id of the customer
	 * @param string  $customerEmail the email address of the customer
	 *
	 * @return array An array of order ids
	 * @see getCustomerTransactions
	 * @deprecated
	 */
	public function getCustomerOrders($customerId,$customerEmail){
		trigger_error("Call to deprecated method getCustomerOrders(), use getCustomerTransactions()", E_USER_NOTICE);
		return $this->getCustomerTransactions($customerId,$customerEmail);
	}

	/**
	 * Given a particular customer returns an array of this customer's order ids
	 *
	 * @param integer $customerId    the id of the customer
	 * @param string  $customerEmail the email address of the customer
	 *
	 * @return \int[]
	 * @see \Red61\Via\ViaApiClient::getCustomerTransactions
	 */
	public function getCustomerTransactions($customerId,$customerEmail){
		return $this->api_service->getCustomerTransactions(
		    \Red61\Via\ApiRequest\apiGetCustomerTransactionsRequest::create()
		        ->setCustomerId($customerId)
				->setEmail($customerEmail)
		);
	}

	/**
	 * Given a particular customer, returns an array of ViaApiTransactionSummary containing each of this customer order
	 * id, order title, and order date.
	 *
	 * @param integer $customerId the id of the customer
	 *
	 * @return \Red61\Via\DataObject\ViaApiTransactionSummary[]
	 * @see getCustomerTransactionsSummary
	 * @deprecated
	 */
	public function getCustomerOrdersSummary($customerId){
		trigger_error("Call to deprecated method getCustomerOrdersSummary(), use getCustomerTransactionsSummary()", E_USER_NOTICE);
		return $this->getCustomerTransactionsSummary($customerId);
	}

	/**
	 * Given a particular customer, returns an array of ViaApiTransactionSummary containing each of this customer order
	 * id, order title, and order date.
	 *
	 * @param integer $customerId the id of the customer
	 *
	 * @return \Red61\Via\DataObject\ViaApiTransactionSummary[]
	 * @see \Red61\Via\ViaApiClient::getCustomerTransactionsSummary
	 */
	public function getCustomerTransactionsSummary($customerId){
		return $this->api_service->getCustomerTransactionsSummary(
		    \Red61\Via\ApiRequest\apiGetCustomerTransactionsSummaryRequest::create()
		        ->setCustomerId($customerId)
		);
	}

	/**
	 * This takes the unique basket id <i>$_basketId</i> and the parameter <i>$cutoff</i> and checks to see that the
	 * earliest performance in the shopping basket is outwith the post cut off time. If so it returns true, if not it
	 * returns false. The parameter <i>$cutoff</i> is not currently stored in the db so is hardcoded and stored within
	 * session management. It is used to decide whether to give the customer the option of having there tickets posted
	 * to them. See /includes/delivery_address.php
	 *
	 * @param integer $cutoff Amount of days before the earliest performance in the basket that post is still an option
	 *
	 * @return boolean boolean indicating whether post is still an option for this order
	 * @see \Red61\Via\ViaApiClient::isPostAllowed
	 * @throws \Red61\Via\Exception\BasketIdNotSetException if called when a basket has not been initialised
	 * @deprecated for getDeliveryOptions
	 */
	public function isPostAllowed ($cutoff) {
		$this->logDepCall("isPostAllowed replaced by getDeliveryOptions");
		return $this->api_service->isPostAllowed(
		    \Red61\Via\ApiRequest\apiIsPostAllowedRequest::create()
		        ->setCutoff($cutoff)
		);
	}

	/**
	 * Sets the friend value of the customer referenced by customerId
	 *
	 * @param integer $customerId
	 * @param boolean $friend The friend value to set
	 *
	 * @deprecated
	 * @see setCustomerFriendStatus
	 */
	public function setFriend($customerId,$friend){
		trigger_error("Call to deprecated method setFriend(), use setCustomerFriendStatus()", E_USER_NOTICE);
		$this->setCustomerFriendStatus($customerId,$friend);
	}

	/**
	 * Sets the friend value of the customer referenced by customerId
	 *
	 * @param integer $customerId
	 * @param boolean $friend The friend value to set
	 *
	 * @return void
	 * @see \Red61\Via\ViaApiClient::setCustomerFriendStatus
	 */
	public function setCustomerFriendStatus($customerId,$friend){
		$this->api_service->setCustomerFriendStatus(
		    \Red61\Via\ApiRequest\apiSetCustomerFriendStatusRequest::create()
		        ->setCustomerId($customerId)
				->setFriend($friend)
		);
	}

	/**
	 * Gets the friend value of the customer referenced by $customerId
	 *
	 * @param integer $customerId
	 *
	 * @return boolean representing if this customer is a friend
	 * @deprecated
	 * @see getCustomerFriendStatus
	 */
	public function getFriend($customerId){
		trigger_error("Call to deprecated method getFriend(), use getCustomerFriendStatus()", E_USER_NOTICE);
		return $this->getCustomerFriendStatus($customerId);
	}

	/**
	 * @param integer $customerId
	 * Gets the friend value of the customer referenced by $customerId
	 *
	 * @return boolean representing if this customer is a friend
	 * @see \Red61\Via\ViaApiClient::getCustomerFriendStatus
	 */
	public function getCustomerFriendStatus($customerId){
		return $this->api_service->getCustomerFriendStatus(
		    \Red61\Via\ApiRequest\apiGetCustomerFriendStatusRequest::create()
		        ->setCustomerId($customerId)
		);
	}

	/**
	 * This method returns a customerDetails object which contains all customer details.
	 *
	 * @param integer $customerId The id of the customer.
	 *
	 * @return \Red61\Via\DataObject\ViaApiCustomerDetails
	 * @see    \Red61\Via\ViaApiService::getCustomerDetails
	 */
	public function getCustomerDetails($customerId) {
		return $this->api_service->getCustomerDetails(
		    \Red61\Via\ApiRequest\apiGetCustomerDetailsRequest::create()
		        ->setCustomerId($customerId)
		);
	}

	/**
	 * This method returns a customerOtherDetails object which contains other information about the customer.
	 *
	 * @param \Red61\Via\DataObject\ViaApiCustomerDetails $customer A customerDetails object.
	 *
	 * @return \Red61\Via\DataObject\ViaApiCustomerOtherDetails
	 * @deprecated in favour of the getter on the object passed in
	 */
	public function getCustomerOtherDetails(\Red61\Via\DataObject\ViaApiCustomerDetails $customer) {
		return $customer->getOtherdetails();
	}

	/**
	 * This method uses the authentication code set in the constructor (<i>$web_key</i>) and  the parameter
	 * <i>$keyword</i> and searches for events. It looks at event titles, event performance dates (using DD/MM/YY),
	 * venue names, sub venue names, company/artists and promoters. If there are no matching events it returns an
	 * empty array.
	 *
	 * @param string $keyword The search term
	 *
	 * @return array array of event objects
	 * @deprecated
	 * @see eventQuickSearch
	 */
	public function getQuickSearchResults($keyword) {
		trigger_error("Call to deprecated method getQuickSearchResults(), use eventQuickSearch()", E_USER_NOTICE);
		return $this->eventQuickSearch($keyword);
	}

	/**
	 * This method uses the authentication code set in the constructor (<i>$web_key</i>) and  the parameter <i>$keyword</i>
	 * and searches for events. It looks at event titles, event performance dates (using DD/MM/YY), venue names, sub
	 * venue names, company/artists and promoters. If there are no matching events it returns an empty array.
	 *
	 * @param string $keyword The search term
	 *
	 * @return \Red61\Via\DataObject\ViaApiEvent[]
	 */
	public function eventQuickSearch($keyword) {
		return $this->api_service->eventQuickSearch(
		    \Red61\Via\ApiRequest\apiEventQuickSearchRequest::create()
		        ->setKeyword($keyword)
				->setQuickSearch(1)
		);
	}

	/**
	 * This method uses the authentication code set in the constructor (<i>$web_key</i>) and the parameters to searches
	 * for events that match any of the submitted non empty parameter fields. If the parameter <i>$typeids</i> is the
	 * only parameter set then no events are returned, this parameter must be set in conjunction with one of the other
	 * parameters. See search.php for an implementation of this. If no events are found it returns an empty array.
	 *
	 * @param string $event     The event name.
	 * @param string $venue     The venue name.
	 * @param string $subvenue  The subvenue name.
	 * @param string $startDate The start of an events performances.
	 * @param string $endDate   The enddate of an events performances (not inclusive).
	 * @param string $typeids   The id pairing of the type in the form 'typeid_subtypeid'.
	 * @param string $artist    The artist name.
	 * @param string $promoter  The promoter name.
	 *
	 * @return \Red61\Via\DataObject\ViaApiEvent[]
	 * @deprecated
	 * @see eventDetailedSearch
	 */
	public function getAdvancedSearchResults($event,$venue,$subvenue,$startDate,$endDate,$typeids,$artist,$promoter) {
		trigger_error("Call to deprecated method getAdvancedSearchResults(), use eventDetailedSearch()", E_USER_NOTICE);
		return $this->eventDetailedSearch($event,$venue,$subvenue,$startDate,$endDate,$typeids,$artist,$promoter);
	}

	/**
	 * This method uses the authentication code set in the constructor (<i>$web_key</i>) and the parameters to searches
	 * for events that match any of the submitted non empty parameter fields. If the parameter <i>$typeids</i> is the
	 * only parameter set then no events are returned, this parameter must be set in conjunction with one of the other
	 * parameters. See search.php for an implementation of this. If no events are found it returns an empty array.
	 *
	 * @param string $event     The event name.
	 * @param string $venue     The venue name.
	 * @param string $subvenue  The subvenue name.
	 * @param string $startDate The start of an events performances.
	 * @param string $endDate   The enddate of an events performances (not inclusive).
	 * @param string $typeids   The id pairing of the type in the form 'typeid_subtypeid'.
	 * @param string $artist    The artist name.
	 * @param string $promoter  The promoter name.
	 * @param string $seasonCode can be used to filter results by season
	 *
	 * @return \Red61\Via\DataObject\ViaApiEvent[]
	 * @see \Red61\Via\ViaApiClient::eventDetailedSearch
	 */
	public function eventDetailedSearch($event,$venue,$subvenue,$startDate,$endDate,$typeids,$artist,$promoter, $seasonCode = NULL) {
		$typeids = $typeids ? : '0_0';
		list($typeid, $subtypeid) = explode('_', $typeids);

		return $this->api_service->eventDetailedSearch(
		    \Red61\Via\ApiRequest\apiEventDetailedSearchRequest::create()
		        ->setEvent($event)
				->setVenue($venue)
				->setSubvenue($subvenue)
				->setCompany($artist)
				->setPromoter($promoter)
				->setStartDate($startDate)
				->setEndingDate($endDate)
				->setTypeId($typeid)
				->setSubTypeId($subtypeid)
				->setSeasonCode($seasonCode)
				->setQuickSearch(0)
		);
	}

	/**
	 * This method returns details on countries and their ISO 3166 codes. A code can then be passed into createAccount.
	 *
	 * @return \Red61\Via\DataObject\ViaApiListElement[] with id being the countries code and title being it's name.
	 *
	 * @see \Red61\Via\ViaApiClient::getCountries
	 */
	public function getCountries(){
		return $this->api_service->getCountries();
	}

	/**
	 * This method returns the default country along with its ISO 3166 code, and an isDefault field to check
	 * whether it is the default country. This code can then be passed into createAccount.
	 *
	 * @return \Red61\Via\DataObject\ViaApiCountryDetails[]
 	 * @see \Red61\Via\ViaApiClient::getCountryDetails
	 */
	public function getCountryDetails(){
		return $this->api_service->getCountryDetails();
	}

	/**
	 * This method searches for a full address given the details. This returns multiple possible addresses as objects
	 * with a summary description and reference.
	 *
	 * The reference can then be used in the call to getAddress to obtain a full address object.
	 *
	 * @param string $address1 The first line of the address
	 * @param string $address2 The second line of the address
	 * @param string $town     The town
	 * @param string $county   The county/state
	 * @param string $country  The country
	 * @param string $postcode The postcode
	 *
	 * @return \Red61\Via\DataObject\ViaApiAddressResult[]
	 * @see    getAddress
	 * @see    \Red61\Via\ViaApiService::searchAddress
	 */
	public function searchAddress($address1,$address2,$town,$county,$country,$postcode)
	{
		$details = new ViaApiAddressDetails($address1, $address2, $town, $county, $country, $postcode);
		return $this->api_service->searchAddress(
		    \Red61\Via\ApiRequest\apiSearchAddressRequest::create()
		        ->setAddressDetails($details)
		);
	}

	/**
	 * This method retrieves an address given a reference.
	 * Used in conjunction with @see searchAddress
	 *
	 * @param string $reference The required addresses reference
	 *
	 * @return \Red61\Via\DataObject\ViaApiAddressDetails
	 * @see    searchAddress
	 * @see    \Red61\Via\ViaApiService::getAddress
	 */
	public function getAddress($reference){
		return $this->api_service->getAddress(
		    \Red61\Via\ApiRequest\apiGetAddressRequest::create()
		        ->setAddressRef($reference)
		);
	}

	/**
	 * This method gives access to the VIA API Client's <i>$_basketId</i> which stores the unique basket id.
	 *
	 * It should be called on every page to allow the basket id to be stored using session management so that when
	 * the red61_via class is instantiated at the beginning of every page the basket id can be passed to the constructor.
	 * See /includes/setup.php
	 *
	 * @return string the unique basket id, NULL if this hasn't been created yet
	 */
	public function getBasketId () {
		return $this->api_client->getBasketId();
	}

	/**
	 * This method will reset a customers password to a new random password.
	 *
	 * @param string $emailAddress The email address of the customer
	 *
	 * @return string
	 * @see \Red61\Via\ViaApiService::customerPasswordReset
	 */
	public function customerPasswordReset($emailAddress) {
		return $this->api_service->customerPasswordReset(
		    \Red61\Via\ApiRequest\apiCustomerPasswordResetRequest::create()
		        ->setEmailAddress($emailAddress)
		);
	}

	/**
	 * This method will return the seating plan describing the seating plan for use with AJAX
	 *
	 * @param string $perfRef    The reference of the performance
	 * @param int    $lastUpdate The unix timestamp of the last copy of the plan received, pass 0 for first load
	 *
	 * @return string an XML document
	 * @see \Red61\Via\ApiClient::getSeatingPlan
	 */
	public function getSeatingPlan($perfRef, $lastUpdate) {
		return $this->api_service->getSeatingPlan(
		    \Red61\Via\ApiRequest\apiGetSeatingPlanRequest::create()
		        ->setPerformanceRef($perfRef)
				->setLastUpdate($lastUpdate)
		);
	}

	/**
	 * Questionnaires are only returned if they are appropriate to display to this customer. The order id is used to
	 * determine if Questionnaires which fire on particular events or performances should be displayed.
	 *  Returns an array of ListElement objects. A listElement object contains two fields, <i>integer id</i> and
	 * <i>string title</i>. In this case the id is the serverId:questionnaireId and the title is the name of the
	 * questionnaire. Returns an empty array if there are no questionnaires.
	 *
	 * @param int $customerId
	 * @param string $orderId
	 *
	 * @return array|\Red61\Via\DataObject\ViaApiListElement[]
	 * @see \Red61\Via\ApiClient::getDisplayableQuestionnaires
	 */
	public function getDisplayableQuestionnaires($customerId,$orderId){
		return $this->api_service->getDisplayableQuestionnaires(
		    \Red61\Via\ApiRequest\apiGetDisplayableQuestionnairesRequest::create()
		        ->setCustomerId($customerId)
				->setOrderId($orderId)
		);
	}

	/**
	 * Return a ViaApiQuestionnaireDetails object containing information on the given questionnaire.
	 *
	 * @param string $questionnaireRef The reference of the questionnaire, of the form serverId:QuestionnaireId
	 *
	 * @return \Red61\Via\DataObject\ViaApiQuestionnaire
 	 * @see \Red61\Via\ApiClient::getDisplayableQuestionnaires
	 */
	public function getQuestionnaireDetails($questionnaireRef){
		return $this->api_service->getQuestionnaireDetails(
		    \Red61\Via\ApiRequest\apiGetQuestionnaireDetailsRequest::create()
		        ->setQuestionnaireRef($questionnaireRef)
		);
	}

	/**
	 * Marks the given customer as haven declined this questionnaire. This questionnaire won't be returned for this
	 * customer again.
	 *
	 * @param string $customerId       The reference of the customer declining
	 * @param string $questionnaireRef The reference of the questionnaire, of the form serverId:QuestionnaireId
	 *
	 * @see \Red61\Via\ApiClient::declineQuestionnaire
	 */
	public function declineQuestionnaire($customerId, $questionnaireRef){
		$this->api_service->declineQuestionnaire(
		    \Red61\Via\ApiRequest\apiDeclineQuestionnaireRequest::create()
		        ->setCustomerId($customerId)
				->setQuestionnaireRef($questionnaireRef)
		);
	}

	/**
	 * Marks the customer as having answered this questionnaire with the given answers. This questionnaire won't be
	 * returned for this customer again.
	 *
	 * @param string                   $customerId       The reference of the customer declining
	 * @param string                   $questionnaireRef The reference of the questionnaire, of the form serverId:QuestionnaireId
	 * @param ViaApiQuestionResponse[] $responses        An array of the customers responses. These are objects with the following properties:
	 *                                                   questionRef - The reference of the question this response is answering, of the form serverId:QuestionId
	 *                                                   text            - If this is a TEXT question the answer the customer supplied. Otherwise null.
	 *                                                   answerRefs    - An array of answer references, of the form serverId:AnswerId
	 *
	 * @see \Red61\Via\ApiClient::answerQuestionnaire
	 */
	public function answerQuestionnaire($customerId, $questionnaireRef, $responses){
		$this->api_service->answerQuestionnaire(
		    \Red61\Via\ApiRequest\apiAnswerQuestionnaireRequest::create()
		        ->setCustomerId($customerId)
				->setQuestionnaireRef($questionnaireRef)
				->setResponses($responses)
		);
	}


	/**
	 * Returns an array of all the custom fields. If customerId is supplied then the returned objects will also contain
	 * the customer's values of these fields.
	 *
	 * @param string $customerId The id of the customer (optional).
	 *
	 * @return \Red61\Via\DataObject\ViaApiCustomCustomerFieldDetails[] If customerId is defined then the entry field
	 * contains the value of this field for non choice fields otherwise the entry field contains the display name of the
	 * choice and the selectedChoice field contains the selected choices id.
	 *
	 * @see \Red61\Via\ApiClient::getCustomCustomerFields
	 */
	public function getCustomCustomerFields($customerId=null){
		return $this->api_service->getCustomCustomerFields(
		    \Red61\Via\ApiRequest\apiGetCustomCustomerFieldsRequest::create()
		        ->setCustomerId($customerId)
		);
	}

	/**
	 * Returns a list of all EventTags
	 *
	 * with id being the event tags id in the form serverId:tagId and title being it's title.
	 *
	 * @return \Red61\Via\DataObject\ViaApiListElement[]
	 * @see \Red61\Via\ApiClient::getEventTags
	 */
	public function getEventTags(){
		return $this->api_service->getEventTags();
	}

	/**
	 * Returns the number of baskets available and in use.
	 *
	 * [!!] value is volitile, but we cache for 10 seconds so that if site is under very high load, any queuing system
	 * can only create a finite load while checking what the server load is
	 *
	 * @return \Red61\Via\DataObject\ViaApiBasketLimits
	 * @see \Red61\Via\ApiClient::getBasketLimits
	 */
	public function	getBasketLimits() {
		return $this->api_service->getBasketLimits();
	}

	/**
	 * Updates a customers custom fields using the ViaAPiCustomCustomerFieldDetails objects supplied
	 *
	 * @param string $customerId The id of the customer to update.
	 * @param \Red61\Via\DataObject\ViaApiCustomCustomerFieldDetails[] The new values of the fields
	 *
	 * @return void
	 * @see \Red61\Via\ViaApiService::updateCustomersCustomFields
	 */
	public function updateCustomersCustomFields($customerId, $entries){
		$this->api_service->updateCustomerCustomFields(
		    \Red61\Via\ApiRequest\apiUpdateCustomerCustomFieldsRequest::create()
		        ->setCustomerId($customerId)
				->setEntries($entries)
		);
	}

	/**
	 * Get the statistics captured by the ViaProfilerPlugin
	 *
	 * For backwards compatibility, only statistics for successful calls that missed the cache will be
	 * returned. To get statistics for all calls including cached and failed, pass an $only_success = FALSE
	 * argument.
	 *
	 * Profiling is only active if the 'profile' => TRUE option is passed in the configuration options
	 * when the class is created.
	 *
	 * @param bool $only_success
	 *
	 * @return array
	 * @see    \Red61\Via\Profiler\ViaProfilerPlugin::getProfilerStats
	 */
	public function getProfileStats($only_success = TRUE)
	{
		if ( ! $this->profiler) {
			return array();
		}

		$full_stats = $this->profiler->getProfilerStats();
		if ($only_success) {
			return $full_stats[ViaProfilerPlugin::CALL_SUCCESS];
		} else {
			return $full_stats;
		}
	}

	/**
	 * Updates a customer's interested events using the eventRef supplied
	 *
	 * @param string  $customerId The id of the customer to update.
	 * @param string  $eventRef   The id of the server/event you wish to register your interest in.
	 * @param boolean $interested is set to true/false based on whether an event should be added/removed from the customer's wishlist.
	 * @param string  $type       The type of the customer's interest.
	 *
	 * @throws InvalidArgumentException
	 * @return void
	 * @see \Red61\Via\ApiClient::registerInterestInEvent
	 */
	public function registerInterestInEvent($customerId, $eventRef, $interested, $type = NULL){
		$this->api_service->registerInterestInEvent(
		    \Red61\Via\ApiRequest\apiRegisterInterestInEventRequest::create()
		        ->setCustomerId($customerId)
				->setEventRef($eventRef)
				->setInterested($interested)
				->setType($type)
		);
	}

	/**
	 *   Updates a customer's interested events using the eventRef supplied
	 *
	 * @param string  $customerId     The id of the customer to update.
	 * @param string  $performanceRef The id of the server/performance you wish to register your interest in.
	 * @param boolean $interested     is set to true/false based on whether a performance should be added/removed from the customer's wishlist.
	 * @param string  $type           The type of the customer's interest.
	 *
	 * @throws InvalidArgumentException if no customer ID
	 * @return void
	 * @see \Red61\Via\ApiClient::registerInterestInEvent
	 */
	public function registerInterestInPerformance($customerId, $performanceRef,$interested, $type = NULL){
		$this->api_service->registerInterestInPerformance(
		    \Red61\Via\ApiRequest\apiRegisterInterestInPerformanceRequest::create()
		        ->setCustomerId($customerId)
				->setPerformanceRef($performanceRef)
				->setInterested($interested)
				->setType($type)
		);
	}

	/**
	 * Retrieves a customer's event wishlist using the customerId
	 *
	 * @param string $customerId The id of the customer to retrieve the event wishlist
	 *
	 * @throws InvalidArgumentException
	 * @return \Red61\Via\DataObject\ViaApiCustomerEventInterest[]
	 * @see \Red61\Via\ApiClient::getCustomerInterestsEvent
	 */
	public function getCustomerInterestsEvent($customerId){
		return $this->api_service->getCustomerInterestsEvent(
		    \Red61\Via\ApiRequest\apiGetCustomerInterestsEventRequest::create()
		        ->setCustomerId($customerId)
		);
	}

	/**
	 * Retrieves a customer's performance wishlist using the customerId
	 *
	 * @param string $customerId The id of the customer to retrieve the performance wishlist
	 *
	 * @throws InvalidArgumentException
	 * @return \Red61\Via\DataObject\ViaApiCustomerPerformanceInterest[]
	 * @see \Red61\Via\ApiClient::getCustomerInterestsPerformance
	 */
	public function getCustomerInterestsPerformance($customerId){
		return $this->api_service->getCustomerInterestsPerformance(
		    \Red61\Via\ApiRequest\apiGetCustomerInterestsPerformanceRequest::create()
		        ->setCustomerId($customerId)
		);
	}

	/**
	 * Retrieves a 10 digit alpha numeric token which can be used to claim the items provided
	 *
	 * @param int[] $items    An int array of the items that you wish to get a token for. This token can be used by a
	 *                        existing customer to claim items in a transaction.
	 *                        Currently tickets, subscriptions and vouchers can have tokens generated.
	 * @param string $orderId the id of the order to which the items belong.
	 *
	 * @return string The 10 digit alpha numeric token to claim for these items in the order identified by $orderId
	 * @see \Red61\Via\ApiClient::sendToAFriend
	 */
	public function sendToAFriend($items, $orderId){
		return $this->api_service->sendToAFriend(
		    \Red61\Via\ApiRequest\apiSendToAFriendRequest::create()
		        ->setItems($items)
				->setOrderId($orderId)
		);
	}

	/**
	 * Claim the items for a customer in the order identified with the token
	 *
	 * @param string $token      The 10 digit alpha numberic token to claim items in the order identified by $orderId
	 * @param string $orderId    the id of the order to which the items belong.
	 * @param int    $customerId the id of the customer who is claiming the items
	 *
	 * @throws \Red61\Via\Exception\InvalidTokenException
	 * @see \Red61\Via\ViaApiService::claimFromFriend
	 */
	public function claimFromFriend($token, $orderId, $customerId) {
		$this->api_service->claimFromFriend(
		    \Red61\Via\ApiRequest\apiClaimFromFriendRequest::create()
		        ->setToken($token)
				->setOrderId($orderId)
				->setCustomerId($customerId)
		);
	}

	/**
	 * Using a token provided by the system, check what transaction items can be claimed by this token
	 *
	 * @param string $token The 10 digit alpha numberic token used to see the details of the items that you about to
	 *                      claim using {@link red61_via::claimFromFriend($token, $orderId, $customerId)}
	 *
	 * @return \Red61\Via\DataObject\ViaApiTicketDetails[] one for each item that can be claimed by this token
	 * @throws \Red61\Via\Exception\InvalidTokenException
	 * @see \Red61\Via\ViaApiService::getTicketsByClaimToken
	 */
	public function getTicketsByClaimToken($token) {
		return $this->api_service->getTicketsByClaimToken(
		    \Red61\Via\ApiRequest\apiGetTicketsByClaimTokenRequest::create()
		        ->setToken($token)
		);
	}

	/**
	 * This function is used to fetch the details of the items that have been claimed by this customer and have not been
	 * refunded
	 *
	 * @param int     $customerId                 The ID of the customer that has claiming the items.
	 * @param boolean $listPastPerformanceTickets Whether you want to see tickets from past performances
	 *
	 * @return \Red61\Via\DataObject\ViaApiTicketDetails[] one for each item that has not been refunded and has been
	 * claimed by this customer
	 * @see \Red61\Via\ViaApiService::getMyClaimedTickets
	 */
	public function getMyClaimedTickets($customerId, $listPastPerformanceTickets) {
		return $this->api_service->getMyClaimedTickets(
		    \Red61\Via\ApiRequest\apiGetMyClaimedTicketsRequest::create()
				->setCustomerId($customerId)
				->setListPastPerformanceTickets($listPastPerformanceTickets)
		);
	}

	/**
	 * This method is used to add a promocode to the current basket.
	 * The purpose of this is to bring promocode specific ticket offers into scope so that they can be
	 * applied to the basket.
	 *
	 * @param string $promocode The promocode to be added to the basket.
	 *
	 * @throws CartNotFoundException if the basket cannot be found, usually because it has expired.
	 * @return void
	 * @see \Red61\Via\ViaApiService::getMyClaimedTickets
	 */
	public function addPromocodeToBasket($promocode) {
		$this->api_service->addPromocodeToBasket(
		    \Red61\Via\ApiRequest\apiAddPromocodeToBasketRequest::create()
		        ->setPromocode($promocode)
		);
	}

	/**
	 * @return \Red61\Via\DataObject\ViaApiSeasonDetails[]
	 * @see \Red61\Via\ViaApiService::getActiveSeasons
	 */
	public function getActiveSeasons()
	{
		return $this->api_service->getActiveSeasons();
	}

	/**
	 * Returns if that email address is present in the customer database. If so will return true. API users are
	 * recommended to call this during the signup process prior to the customer filling in all their details to
	 * guide the customer towards “claiming” their customer record via the password reset method rather than
	 * creating a new customer record.
	 *
	 * @param string $email
	 *
	 * @return bool
	 * @see    \Red61\Via\ViaApiService::customerEmailExists
	 */
	public function customerEmailExists($email)
	{
		return $this->api_service->customerEmailExists(
		    \Red61\Via\ApiRequest\apiCustomerEmailExistsRequest::create()
		        ->setEmailAddress($email)
		);
	}

	/**
	 * VIA supplied the ability to request that card details saved to be able to be used at a later date for purchase,
	 * as the request to save these is passed onto the payment service provider and Red61 have no mechanism to delete
	 * these card details, there can’t be a method that deletes the card details, but this method will disable the
	 * re-use of that payment token for future payments. Care should be taken when using this method that the messaging
	 * to the customer does not say “delete card” or similar as this is not what is happening behind the scenes, so
	 * the legal wording must be correct.
	 *
	 * @param int    $customerId
	 * @param string $token
	 *
	 * @return bool
	 * @see    \Red61\Via\ViaApiService::removeRepeatabilityCustomerSavedCard
	 */
	public function removeRepeatabilityCustomerSavedCard($customerId, $token)
	{
		return $this->api_service->removeRepeatabilityCustomerSavedCard(
		    \Red61\Via\ApiRequest\apiRemoveRepeatabilityCustomerSavedCardRequest::create()
		        ->setCustomerId($customerId)
				->setRepeatPaymentToken($token)
		);
	}

	/**
	 * This new method provides the ability to fetch the list of performances only within the supplied time range. This
	 * method is to be used for cases where there are say performances every 15 minutes for a single event, so you only
	 * wish to fetch the list for the next 24 hours.
	 *
	 * @param string $eventRef
	 * @param string $startDateTime
	 * @param string $endDateTime
	 *
	 * @return \Red61\Via\DataObject\ViaApiPerformance[]
	 * @see    \Red61\Via\ViaApiService::getPerformancesByDate
	 */
	public function getPerformancesByDate($eventRef, $startDateTime, $endDateTime)
	{
		return $this->api_service->getPerformancesByDate(
		    \Red61\Via\ApiRequest\apiGetPerformancesByDateRequest::create()
		        ->setEventRef($eventRef)
				->setStartDateTime($startDateTime)
				->setEndDateTime($endDateTime)
		);
	}

	/**
	 * This method is similar to apiEventDetailedSearch, but rather than returning a list of events which have at least
	 * one performance within the time range specified, it actually returns a list of ViaApiEventPerformance objects
	 * which provide the detail of the performance event name and distance from the supplied LatLng if supplied. This
	 * method is designed to be used for “nearby now” functionality in mobile apps or for “Next Shows” displays
	 *
	 * @param string $eventName
	 * @param string $venueName
	 * @param string $subvenueName
	 * @param string $company
	 * @param string $promoter
	 * @param string $startDate
	 * @param string $endDate
	 * @param int $typeId
	 * @param int $subtypeId
	 * @param string $seasonCode
	 * @param string $latLong
	 * @param string $radius
	 *
	 * @return \Red61\Via\DataObject\ViaApiEventPerformance[]
	 * @see    \Red61\Via\ViaApiService::getPerformancesByTime
	 */
	public function getPerformancesByTime($eventName, $venueName, $subvenueName, $company, $promoter, $startDate, $endDate, $typeId, $subtypeId, $seasonCode, $latLong, $radius)
	{
		return $this->api_service->getPerformancesByTime(
		    \Red61\Via\ApiRequest\apiGetPerformancesByTimeRequest::create()
				->setLatLong($latLong)
				->setRadius($radius)
				->setStartDate($startDate)
				->setEndingDate($endDate)
				->setPromoter($promoter)
				->setCompany($company)
				->setEventName($eventName)
				->setVenueName($venueName)
				->setSubvenueName($subvenueName)
				->setTypeId($typeId)
				->setSubTypeId($subtypeId)
				->setSeasonCode($seasonCode)
		);
	}
	
	
	/**
	 * This method provides a list of the companies from whom the customer could possibly receive email
	 * correspondence. The list returned only contains those companies for whom customer has not explicitly
	 * confirmed or denied permission for contact via email.
	 *
	 * @param int $customerId
	 *
	 * @return \Red61\Via\DataObject\ViaApiCompany[]
	 * 
	 * @deprecated in favour of \Red61\Via\ViaApiService::getCustomerUndecidedMailPermissionCompanies
	 * 
	 * @see    \Red61\Via\ViaApiService::getCustomerUndecidedMailPermissionCompanies
	 */
	public function getCustomerCompanyDetails($customerId) {
		return $this->api_service->getCustomerCompanyDetails(
						\Red61\Via\ApiRequest\apiGetCustomerCompanyDetailsRequest::create()
								->setCustomerId($customerId)
		);
	}

	/**
	 * This new method provides a list of the companies from whom the customer could possibly receive email
	 * correspondence. The list returned only contains those companies for whom customer has not explicitly
	 * confirmed or denied permission for contact via email.
	 *
	 * @param int $customerId
	 *
	 * @return \Red61\Via\DataObject\ViaApiCompany[]
	 * @see    \Red61\Via\ViaApiService::getCustomerUndecidedMailPermissionCompanies
	 */
	public function getCustomerUndecidedMailPermissionCompanies($customerId)
	{
		return $this->api_service->getCustomerUndecidedMailPermissionCompanies(
		    \Red61\Via\ApiRequest\apiGetCustomerUndecidedMailPermissionCompaniesRequest::create()
		        ->setCustomerId($customerId)
		);
	}

	/**
	 * This method is used to update a customer's mailing permissions to include the new responses given, with regards
	 * to what companies can and cannot send them email correspondence.
	 *
	 * @param int                                                   $customerId
	 * @param \Red61\Via\DataObject\ViaApiCustomerCompanyResponse[] $customerCompanyResponses
	 */
	public function updateCustomerCompanyMailPermissions($customerId, $customerCompanyResponses)
	{
		$this->api_service->updateCustomerCompanyMailPermissions(
		    \Red61\Via\ApiRequest\apiUpdateCustomerCompanyMailPermissionsRequest::create()
		        ->setCustomerId($customerId)
				->setCustomerCompanyResponses($customerCompanyResponses)
		);
	}
	
	/**
	 * This method is used to retrieve the Card Type of a given card using its BIN
	 *
	 * @param string $cardBIN
	 *
	 * @return \Red61\Via\DataObject\ViaApiCardType
	 * @see    \Red61\Via\ViaApiService::getCardTypeFromBIN
	 */
	public function getCardTypeFromBIN($cardBIN)
	{
		return $this->api_service->getCardTypeFromBIN(
		    \Red61\Via\ApiRequest\apiGetCardTypeFromBINRequest::create()
		        ->setCardBIN($cardBIN)
		);
	}
	
	
	/**
	 * This method is used to retrieve details about the schemes in a given order.
	 *
	 * @param string $orderId 
	 * @return \Red61\Via\DataObject\ViaApiSchemeTierDetails[]
	 * @see \Red61\Via\ViaApiService::apiGetOrderSchemeDetails
	 */
	
	public function getOrderSchemeDetails($orderId) {
		return $this->api_service->getOrderSchemeDetails(
		    \Red61\Via\ApiRequest\apiGetOrderSchemeDetailsRequest::create()
		        ->setOrderId($orderId)
		);
	}
	
	/**
	 * This method is used to retrieve the instance's offset from the end of the day.
	 *
	 * @return int the number of hours past midnight at which the day ends.
	 * @see    \Red61\Via\ViaApiService::apiGetDayChangeOffset
	 */
	public function getDayChangeOffset()
	{
		if($this->dayOffset == null) {
			$this->dayOffset = $this->api_service->getDayChangeOffset();
		}
		return $this->dayOffset;
	}
	
	
	
	/**
	 * This method is used to extend the life of a customer's basket by a further 30 minutes in the event that they are not finished shopping but have also not
	 * added anything to the basket within the last 30 minutes.
	 *
	 * @throws \Red61\Via\Exception\CartNotFoundException If the basket cannot be found, usually because it has expired.
	 * @see \Red61\Via\ViaApiService::apiBasketKeepAlive
	 */
	public function basketKeepAlive() {
		return $this->api_service->basketKeepAlive();
	}
	
	
	
	/**
	 * This method is used to add a category to the given customer.
	 *
	 * @param customerId The id of the customer
	 * @param categoryId The id of the category to add. This must correspond to web accessible customer categories ids.
	 * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
	 * 
	 * @see \Red61\Via\ViaApiService::apiAddCustomerCategory
	 */
	public function addCustomerCategory($customerId, $categoryId) {
		return $this->api_service->addCustomerCategory(
		    \Red61\Via\ApiRequest\apiAddCustomerCategoryRequest::create()
		        ->setCustomerId($customerId)
				->setCategoryId($categoryId)
		);
	}

	/**
	 * This method is used to remove a category from the given customer.
	 *
	 * @param int $customerId The id of the customer
	 * @param int $categoryId The id of the category to remove. This must correspond to web accessible customer categories ids.
	 *
	 * @return void
	 * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
	 * 
	 * @see \Red61\Via\ViaApiService::apiRemoveCustomerCategory
	 */
	public function removeCustomerCategory($customerId, $categoryId) {
		return $this->api_service->removeCustomerCategory(
		    \Red61\Via\ApiRequest\apiRemoveCustomerCategoryRequest::create()
		        ->setCustomerId($customerId)
				->setCategoryId($categoryId)
		);
	}
	
	/**
	 * This method is used to retrieve the valid and current subscriptions for the given customer.
	 *
	 * @param int $customerId The id of the customer
	 *
	 * @return \Red61\Via\DataObject\ViaApiSubscriptionDetails[] containing information on the customer's subscription as well as the
	 *                                                           scheme the subscription is for.
	 * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
	 * @see \Red61\Via\ViaApiService::apiGetCustomerSubscriptionDetailst
	 * 
	 */
	public function getCustomerSubscriptionDetails($customerId){
		return $this->api_service->getCustomerSubscriptionDetails(
		    \Red61\Via\ApiRequest\apiGetCustomerSubscriptionDetailsRequest::create()
		        ->setCustomerId($customerId)
		);
	}
	
	/**
	 * This method is used to retrieve information on the gift aid declarations that this customer has made.
	 *
	 * @param int $customerId The id of the customer
	 *
	 * @return \Red61\Via\DataObject\ViaApiGiftAidDeclarationDetails[] containing information on the customer's non-recanted gift aid declarations
	 * 
	 * @see \Red61\Via\ViaApiService::apiGetCustomerGiftAidDeclarations
	 */
	public function getCustomerGiftAidDeclarations($customerId) {
		return $this->api_service->getCustomerGiftAidDeclarations(
						\Red61\Via\ApiRequest\apiGetCustomerGiftAidDeclarationsRequest::create()
								->setCustomerId($customerId)
		);
	}

	/**
	 * This method is used to retrieve all information on the gift aid declaration for the given customer and gift aid
	 * declaration Id.
	 *
	 * @param int $customerId The Id of the customer
	 * @param int $giftAidDeclarationId The Id of the gift aid declaration record
	 * @return \Red61\Via\DataObject\ViaApiGiftAidDeclarationFullDetails Full details of the gift aid declaration record for the given customer.
	 */
	public function getCustomerGiftAidDeclaration($customerId, $giftAidDeclarationId)
	{
		return $this->api_service->getCustomerGiftAidDeclaration(
			\Red61\Via\ApiRequest\apiGetCustomerGiftAidDeclarationRequest::create()
				->setCustomerId($customerId)
				->setGiftAidDeclarationId($giftAidDeclarationId)
		);
	}

	/**
	 * This method is used to determine whether or not the underlying payment processor
	 * supports saving the customers card information for ease of future purchases.
	 *
	 *
	 * @return bool
	 */
	public function doesProcessorSupportRepeatPayments() {
		return $this->api_service->doesProcessorSupportRepeatPayments();
	}

	public function getOrderIdForSessionId($orderSessionId)	{
		return $this->api_service->getOrderIdForSessionId(\Red61\Via\ApiRequest\apiGetOrderIdForSessionIdRequest::create()
			->setOrderSessionId($orderSessionId));
	}
	
	public function getUserSimilarEvents($customerId, $numberOfRecommendations) {
		return $this->api_service->getUserSimilarEvents(
				\Red61\Via\ApiRequest\apiGetUserSimilarEventsRequest::create()
		        ->setCustomerId($customerId)
				->setNumberOfEvents($numberOfRecommendations)
				);
	}


	/**
   * This method is used to retrieve a {@link ViaApiSchemeUsageDetails} object containing
   * information on the number and details of discounted tickets for the customer's subscription
   *
   * @param int $customerId The id of the customer
   * 
   * @param String $schemeRef The reference for the scheme, of the form serverId:schemeId
   *
   * @return \Red61\Via\DataObject\ViaApiSchemeUsageDetails containing information on the number and details of discounted tickets for the customer's subscription
   *
   * @see \Red61\Via\ViaApiService::apiGetCustomerSchemeUsages
   */
	public function getCustomerSchemeUsages($customerId, $schemeRef) {
		return $this->api_service->getCustomerSchemeUsages(
		\Red61\Via\ApiRequest\apiGetCustomerSchemeUsagesRequest::create()
		->setCustomerId($customerId)
		->setSchemeRef($schemeRef)
		);
	}
	
	
	/**
	* This method is used to retrieve a {@link ViaApiAvailablePrintOptionReturn} object containing
	* a list of valid delivery options available calculated against the current contents of the basket,
	* also contains a String basketChangedMessage the presence of which indicates that the basket's print option has been changed
	* due to the basket contents not allowing the originally selected print option. 
	* It is advisable to refresh the basket contents if this exception is populated.
	* 
 	* Post is only available if all performance's in the basket are not after the {@code postCutOff} set 
 	* and E-Ticket is only available if all the sub venues in the basket are enabled for E-Ticket.
 	* Therefore these delivery options can change after any basket contents change.
	*
	* @see \Red61\Via\ViaApiService::apiGetBasketPrintOptions
	* @deprecated for getDeliveryOptions
   */
	public function getBasketPrintOptions($postCutOff) {
		$this->logDepCall("getBasketPrintOptions replaced by getDeliveryOptions");
		return $this->api_service->getDeliveryOptions(
		\Red61\Via\ApiRequest\apiGetDeliveryOptionsRequest::create()
		->setPostCutOff($postCutOff)
		);
	}

	public function getDeliveryOptions() {
		return $this->api_service->getDeliveryOptions(
		\Red61\Via\ApiRequest\apiGetDeliveryOptionsRequest::create()
		);
	}

	public function setDeliveryOption($deliveryOptionId, $itemTypeFees) {
		$fees = array();
		foreach($itemTypeFees as $typeId=>$feeId) {
			$fees[] = new Red61\Via\DataObject\ViaApiFeeForItemType($typeId, $feeId);
		}
		return $this->api_service->setDeliveryOption(
		\Red61\Via\ApiRequest\apiSetDeliveryOptionRequest::create()
		->setDeliveryOptionId($deliveryOptionId)
		->setDeliveryFees($fees)
		);
	}
	
	/**
	* This method is used to retrieve details about the tickets and their scanned status for a given order.
	* This differs to {@link#apiGetOrderTicketDetails(java.lang.String, java.lang.String) }
	* due to the returned object {@link ViaApiTicketDetailsWithStatus} containing a {@code List} of
	* {@link ViaApiTicketItemDetailsWithStatus} objects
	* which contain the field {@link ViaApiTicketItemDetailsWithStatus#availableToScan}
	*
	* @param string $orderId
	*
	* @return \Red61\Via\DataObject\ViaApiTicketDetailsWithStatus[]
	*
	* @see \Red61\Via\ViaApiService::apiGetOrderTicketDetailsWithStatus
	*/
	public function getOrderTicketDetailsWithStatus($orderId) {
		return $this->api_service->getOrderTicketDetailsWithStatus(
		\Red61\Via\ApiRequest\apiGetOrderTicketDetailsWithStatusRequest::create()
		->setOrderId($orderId)
		);
	}
	
	
	/**
	* This method is used to retrieve the full transaction details for a given order ID.
	* Identical to {@link #apiGetOrderSummary(java.lang.String, java.lang.String) } but
	* includes ticket scanning information {@link ViaApiTicketItemDetailsWithStatus#availableToScan}
	*
	* @param string $orderId
	*
	* @return \Red61\Via\DataObject\ViaApiTransactionDetails
	*
	* @see \Red61\Via\ViaApiService::apiGetTransactionDetails
	*/
	public function getTransactionDetails($orderId) {
		return $this->api_service->getTransactionDetails(
		\Red61\Via\ApiRequest\apiGetTransactionDetailsRequest::create()
		->setOrderId($orderId)
		);
	}
	
	/**
	* This method is used to list the transaction items that can be claimed using the token generated using
	* {@link #apiSendToAFriend(java.util.Set, java.lang.String, java.lang.String) }
	* for the transaction provided. To claim the items for a customer call
	* {@link #apiClaimFromFriend(java.lang.String, java.lang.String, int, java.lang.String)}
	*
	* @param $token the token that would be used to claim the items
	* @param $orderId The ID of the transaction that the items to claim are from
	*
	* @return \Red61\Via\DataObject\ViaApiTicketDetails[]
	* @throws \Red61\Via\Exception\InvalidTokenException if there are no items for this order matching the token, the transaction is invalid or the token
	* is malformed
	*
	* @see \Red61\Via\ViaApiService::apiListSentByFriend
	*/
	public function listSentByFriend($token, $orderId) {
		return $this->api_service->listSentByFriend(
		\Red61\Via\ApiRequest\apiListSentByFriendRequest::create()
		->setToken($token)
		->setOrderId($orderId)
		);
	}

	public function setBasketOrderNote($type, $note) {
		return $this->api_service->setBasketOrderNote(
		\Red61\Via\ApiRequest\apiSetBasketOrderNoteRequest::create()
		->setType($type)
		->setNote($note)
		);
	}

	public function getBasketOrderNotes() {
		return $this->api_service->getBasketOrderNotes(
		\Red61\Via\ApiRequest\apiGetBasketOrderNotesRequest::create()
		);
	}

	/**
	 * Refunds the selected complimentary items (i.e. free tickets) from the given order.
	 * <p>
	 * Can not be used to refund anything that cost money.
	 *
	 * @param string $orderId Original order to refund items from, in format 'serverId:orderId'
	 * @param int[] $itemIds Items to refund
	 * @param string $refundNote Reasons for the refund
	 * @return \Red61\Via\DataObject\ViaApiOrderDetails
	 */
	public function refundCompItems($orderId, $itemIds, $refundNote)
	{
		return $this->api_service->refundCompItems(
			\Red61\Via\ApiRequest\apiRefundCompItemsRequest::create()
				->setOrderId($orderId)
				->setItemIds($itemIds)
				->setRefundNote($refundNote)
		);
	}

	/**
	 * Fetches basic information for performances that use a given ticket offer (by concession code), between start and
	 * end date/time parameters
	 *
	 * @param string $concessionCode Concession code of the offer to filter by
	 * @param string $startTime Starting datetime of the search window
	 * @param string $endTime Ending datetime of the search window
	 * @return \Red61\Via\DataObject\ViaApiOfferPerformance[]
	 */
	public function getOfferPerformances($concessionCode, $startTime, $endTime)
	{
		return $this->api_service->getOfferPerformances(
			\Red61\Via\ApiRequest\apiGetOfferPerformancesRequest::create()
				->setConcessionCode($concessionCode)
				->setStartTime($startTime)
				->setEndTime($endTime)
		);
	}
	
	/**
	 * 
	 * @param orderId Order to which you wish to add transaction notes, in format 'serverId:orderId'
	 * @param type
	 * @param note
	 */
	public function addOrderNote($orderId, $type, $note) {
		return $this->api_service->addOrderNote(
		\Red61\Via\ApiRequest\apiAddOrderNoteRequest::create()
		->setOrderId($orderId)
		->setType($type)
		->setNote($note)
		);
	}
	


	/**
	 * 
	 * @param orderId Order to which you wish to add transaction notes, in format 'serverId:orderId'
	 * @param voucherLineItemId the Item Id of the voucher you wish to mark as gifted
	 * @param orderNote 
	 */
  public function markVoucherAsGifted($orderId, $voucherLineItemId, $orderNote)
  {
    return $this->api_service->markVoucherAsGifted(
		\Red61\Via\ApiRequest\apiMarkVoucherAsGiftedRequest::create()
		->setOrderId($orderId)
		->setVoucherLineItemId($voucherLineItemId)
		->setOrderNote($orderNote)
		);
  }
  

	
  /**
   * 
   * @param orderId Order id in format 'serverId:orderId'
   * @param orderTitle the title of the order
   * @param customerId The customer of the order
   *
   * @return \Red61\Via\DataObject\ViaApiVoucherGiftDetails[]
   */
  public function getVoucherGiftDetailsForOrder($orderId, $orderTitle, $customerId)
  {
	  return $this->api_service->getVoucherGiftDetailsForOrder(
		\Red61\Via\ApiRequest\apiGetVoucherGiftDetailsForOrderRequest::create()
		->setOrderId($orderId)
		->setOrderTitle($orderTitle)
		->setCustomerId($customerId)
		);  
  }

	/**
	 * Fetch all global data protection consent questions to ask a new customer at customer creation
	 *
	 * @return \Red61\Via\DataObject\ViaApiConsentQuestion[]
	 */
	public function getConsentQuestionsForNewCustomer()
	{
		return $this->api_service->getConsentQuestionsForNewCustomer();
	}

	/**
	 * Fetch any un-answered, global data protection consent questions to ask an existing customer
	 *
	 * @param $customerId
	 * @return \Red61\Via\DataObject\ViaApiConsentQuestion[]
	 */
	public function getConsentQuestionsForExistingCustomer($customerId)
	{
		return $this->api_service->getConsentQuestionsForExistingCustomer(
			\Red61\Via\ApiRequest\apiGetConsentQuestionsForExistingCustomerRequest::create()
				->setCustomerId($customerId)
		);
	}

	/**
	 * Fetch any un-answered data protection consent questions for the basket's customer, including any questions specific to the basket contents
	 *
	 * @return \Red61\Via\DataObject\ViaApiConsentQuestions
	 * @throws \Red61\Via\Exception\CartNotFoundException
	 */
	public function getConsentQuestionsForBasket()
	{
		return $this->api_service->getConsentQuestionsForBasket();
	}

	/**
	 * Fetch all answered data protection consent questions for a customer, with their given answers.
	 * Optionally return any additional un-answered questions for this customer.
	 *
	 * @param int   $customerId         Customer to look up questions and answers for
	 * @param bool  $includeUnanswered  If true, any new or currently un-answered questions for the customer will also be returned
	 * @return \Red61\Via\DataObject\ViaApiCustomerConsentQuestions
	 */
	public function getCustomerConsentQuestions($customerId, $includeUnanswered)
	{
		return $this->api_service->getCustomerConsentQuestions(\Red61\Via\ApiRequest\apiGetCustomerConsentQuestionsRequest::create()
			->setCustomerId($customerId)
			->setIncludeUnanswered($includeUnanswered));
	}

	/**
	 * Save new answers to data protection consent questions, for the given customer.
	 *
	 * @param int $customerId The customer who these answers are for
	 * @param \Red61\Via\DataObject\ViaApiCustomerConsentQuestion[] $questions The questions being answered, with the customer's answers
	 */
	public function setCustomerConsentAnswers($customerId, $questions)
	{
		$this->api_service->setCustomerConsentAnswers(
			\Red61\Via\ApiRequest\apiSetCustomerConsentAnswersRequest::create()
				->setCustomerId($customerId)
				->setQuestions($questions)
		);
	}

	private function logDepCall($message) {
		if(defined("DEBUG_BACKTRACE_IGNORE_ARGS")) {
			$tmp = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 2);
			$call = $tmp[1];
			trigger_error($message . " triggered by " . $call['file'] . " on line " . $call['line'], E_USER_DEPRECATED);
		}
		else {
			$tmp = debug_backtrace(2);
			$call = $tmp[1];
			trigger_error($message . " triggered by " . $call['file'] . " on line " . $call['line'], E_USER_DEPRECATED);
		}
	}
	
	
	
	
	/**
	 * Converts tickets within a reservation. If the {@code itemIds} List is empty or {@code null} the system will convert all remaining active tickets within
	 * the reservation. An active ticket is one that has not been released or already converted. Converting a reservation will take the specified tickets and
	 * add them to the basket defined by {@code sessionId}. Converting <b>IS POSSIBLE</b> if this basket already has items in it. The {@code convertingBehaviour}
	 * controls how the system handles the ticket offers present against the tickets being converted. This has 3 values:
	 * <ol>
	 * <li><b>disregard</b> : all tickets will be added at full price</li>
	 * <li><b>available</b> : all tickets will be added with best ticket offers currently available</li>
	 * <li><b>forced</b> : all tickets will be added with the ticket offer reserved at the time of the reservation, if still possible. If not it will fail</li>
	 * </ol>
	 * Different {@code convertingBehaviour} values will impose different restrictions on the converting process, with {@code forced} being the most restrictive.
	 * <br/><b>WARNING:</b> The active basket will have its customer set to the customer of the reservation.
	 *
	 * @param $orderId String The order ID of the reservation in the form 'serverId:orderId'. See {@link ViaApiPurchaseSummary#orderid} {@link ViaApiTransactionDetails#orderid}
	 * @param $itemIds int[] A list of the item Ids, {@link ViaApiTicketItemDetails#itemId}s or {@link ViaApiTicketItemDetailsWithStatus#itemId}s, of the active tickets within this
	 * reservation to be converted. If the field {@link ViaApiTicketItemDetails#reservationStatus} or {@link ViaApiTicketItemDetailsWithStatus#reservationStatus} is <b>{@code Pending}</b> then the ticket
	 * is active at the time of loading.
	 * See {@link ViaApiPurchaseSummary#products} or {@link ViaApiTransactionDetails#products} for grouped tickets within the reservation, then {@link ViaApiTicketDetails#ticketItems}
	 * or {@link ViaApiTicketDetailsWithStatus#ticketItems} for individual tickets. If {@code null} or empty all remaining active tickets within this reservation will be converted.
	 * @param $convertingBehaviour String either "<b>disregard</b>" or "<b>available</b>" or "<b>forced</b>". See above for details.
	 *
	 *
	 * @throws \Red61\Via\Exception\CartNotFoundException If {@code sessionId} does not reference an active basket.
	 * @throws \Red61\Via\Exception\CustomerMergedException If the customer has been merged
	 * @throws \Red61\Via\Exception\FailedConvertException If there was a problem with the chosen {@code convertingBehaviour} when adding the active items from the reservation to the basket. If this is triggered the original reservation
	 * will be unchanged
	 * @throws \Red61\Via\Exception\AddSeatsException If there was a problem adding the active items from the reservation to the basket. If this is triggered the original reservation
	 * will be unchanged
	 * @throws \Red61\Via\Exception\FailedRevertException If there was any problem with the basket calls required for converting the system will revert the convert of the reservation. This error is thrown in the rare case
	 * that this revert fails
	 * @see \Red61\Via\ViaApiService::apiGetOrderSummary
	 * @see \Red61\Via\ViaApiService::apiGetTransactionDetails
	 * @see \Red61\Via\ViaApiService::apiGetOrderTicketDetails
	 * @see \Red61\Via\ViaApiService::apiGetOrderTicketDetailsWithStatus
	 */
	public function convertReservation($orderId, $itemIds, $convertingBehaviour)
	{
		$this->api_service->convertReservation(
			\Red61\Via\ApiRequest\apiConvertReservationRequest::create()
				->setOrderId($orderId)
				->setItemIds($itemIds)
				->setConvertingBehaviour($convertingBehaviour)
		);
	}
	
	/**
	 * Releases tickets within a reservation. If the {@code itemIds} List is empty or {@code null} the system will release all remaining active tickets within
	 * the reservation. An active ticket is one that has not been released or already converted.
	 *
	 * @param $orderId String The order ID of the reservation in the form 'serverId:orderId'. See {@link ViaApiPurchaseSummary#orderid} {@link ViaApiTransactionDetails#orderid}
	 * @param $itemIds int[] A list of the item Ids, {@link ViaApiTicketItemDetails#itemId}s or {@link ViaApiTicketItemDetailsWithStatus#itemId}s, of the active tickets within this
	 * reservation to be released. If the field {@link ViaApiTicketItemDetails#reservationStatus} or {@link ViaApiTicketItemDetailsWithStatus#reservationStatus} is <b>{@code Pending}</b> then the ticket
	 * is active at the time of loading. See {@link ViaApiPurchaseSummary#products} or {@link ViaApiTransactionDetails#products} for grouped tickets within the reservation,
	 * then {@link ViaApiTicketDetails#ticketItems} or {@link ViaApiTicketDetailsWithStatus#ticketItems} for individual tickets.
	 * If {@code null} or empty all remaining active tickets within this reservation will be released.
	 *
	 * @param $releaseNote String An optional String with the reason for releasing the tickets from the reservation.
	 * 
	 * @throws \Red61\Via\Exception\catalog.exceptions.FailedReleaseException If an error occurred releasing the tickets from the reservation. See the message for details
	 * 
	 */
	public function releaseTicketsFromReservation($orderId, $itemIds, $releaseNote)
	{
		$this->api_service->releaseTicketsFromReservation(
			\Red61\Via\ApiRequest\apiReleaseTicketsFromReservationRequest::create()
				->setOrderId($orderId)
				->setItemIds($itemIds)
				->setReleaseNote($releaseNote)
		);
	}


}

/**
 * Compares titles of two ViaApiListElement objects for use as a sort callback
 *
 * @param \Red61\Via\DataObject\ViaApiListElement $a
 * @param \Red61\Via\DataObject\ViaApiListElement $b
 *
 * @return int
 * @deprecated not recommended for use in third-party code
 */
function sort_ViaApiListElement($a, $b) {
	// @todo move sort_ViaApiListElement into a better location
	return(strcmp($a->getTitle(), $b->getTitle()));
}
