<?php
/**
 * Defines NamespacedMemcachedAdaptorSpec - specifications for Red61\Via\Cache\NamespacedMemcachedAdaptor
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */

namespace spec\Red61\Via\Cache;

use PhpSpec\Exception\Example\FailureException;
use PhpSpec\Exception\Example\SkippingException;
use Prophecy\Argument;
use Red61\Via\Cache\CacheConfigurationException;
use Red61\Via\Cache\MemcachedOperationFailedException;
use Red61\Via\Cache\NamespacedMemcachedAdaptor;
use Red61\Via\Exception\CacheNotAvailableException;
use spec\ObjectBehavior;

if ( ! class_exists('\Memcached')) {
    // Define an empty class to allow us to get past prophecy's mocking and skip the tests
    eval('class Memcached { public static $not_installed = TRUE;}');
}

/**
 *
 * @see Red61\Via\Cache\NamespacedMemcachedAdaptor
 */
class NamespacedMemcachedAdaptorSpec extends ObjectBehavior
{
    /**
     * Use $this->subject to get proper type hinting for the subject class
     *
     * @var \Red61\Via\Cache\NamespacedMemcachedAdaptor
     */
    protected $subject;

    /**
     * @param \Memcached $memcache
     */
    function let($memcache)
    {
        if (isset(\Memcached::$not_installed)) {
            throw new SkippingException("Skipping - memcache extension is not installed");
        }
        $this->beConstructedWith($memcache, []);
        $memcache->getServerList()->willReturn([]);
        $memcache->addServer(Argument::cetera())->willReturn(TRUE);
    }

    function it_is_initializable()
    {
        $this->subject->shouldHaveType(NamespacedMemcachedAdaptor::class);
    }

    /**
     * @param \Memcached $memcache
     */
    function it_does_not_attempt_to_add_a_server_if_no_server_options_passed($memcache)
    {
        $this->beConstructedWith($memcache, []);
        $this->subject->getWrappedObject();
        $memcache->addServer(Argument::any())->shouldNotHaveBeenCalled();
    }


    /**
     * @param \Memcached $memcache
     */
    function it_does_not_attempt_to_add_a_server_if_empty_server_options_passed($memcache)
    {
        $this->newSubjectWith($memcache, ['server' => NULL]);
        $memcache->addServer(Argument::any())->shouldNotHaveBeenCalled();
    }

    /**
     * @param \Memcached $memcache
     */
    function it_adds_server_if_options_passed_and_server_not_in_pool($memcache)
    {
        $this->newSubjectWith($memcache, ['server' => ['host' => '123.4.5.2', 'port' => 1234]]);
        $memcache->addServer('123.4.5.2', 1234)->shouldHaveBeenCalled();
    }

    /**
     * @param \Memcached $memcache
     */
    function it_throws_if_server_specified_without_host($memcache)
    {
        $this->beConstructedWith($memcache, ['server' => ['host' => NULL, 'port' => 1234]]);
        $this->shouldThrow(CacheConfigurationException::missingHost())->duringInstantiation();
    }

    /**
     * @param \Memcached $memcache
     */
    function it_merges_default_port_if_server_passed_as_host_only($memcache)
    {
        $this->newSubjectWith($memcache, ['server' => ['host' => '127.0.0.1']]);
        $memcache->addServer('127.0.0.1', NamespacedMemcachedAdaptor::MEMCACHED_DEFAULT_PORT)->shouldHaveBeenCalled();
    }

    /**
     * @param \Memcached $memcache
     */
    function it_does_not_attempt_to_add_server_if_already_in_pool($memcache)
    {
        $memcache->getServerList()->willReturn([['host' => '123.4.5.6', 'port' => 11238, 'type' => 'TCP']]);

        $this->newSubjectWith($memcache, ['server' => ['host' => '123.4.5.6', 'port' => 11238]]);

        $memcache->addServer(Argument::cetera())->shouldNotHaveBeenCalled();
    }

    /**
     * @param \Memcached $memcache
     */
    function it_does_not_attempt_to_add_server_if_multiple_of_same_server_in_pool($memcache)
    {
        $memcache->getServerList()->willReturn(
            [
                ['host' => '123.4.5.6', 'port' => 11238, 'type' => 'TCP'],
                ['host' => '123.4.5.6', 'port' => 11238, 'type' => 'TCP'],
            ]
        );

        $this->newSubjectWith($memcache, ['server' => ['host' => '123.4.5.6', 'port' => 11238]]);

        $memcache->addServer(Argument::cetera())->shouldNotHaveBeenCalled();
    }

    /**
     * @param \Memcached $memcache
     */
    function it_throws_if_configuring_a_server_and_other_servers_in_pool($memcache)
    {
        $memcache->getServerList()->willReturn(
            [
                ['host' => '123.4.5.6', 'port' => 11238, 'type' => 'TCP'],
                ['host' => '99.9.99.9', 'port' => 1534, 'type' => 'TCP'],
            ]
        );

        $this->beConstructedWith($memcache, ['server' => ['host' => '123.4.5.6', 'port' => 11238]]);

        $this->shouldThrow(CacheConfigurationException::unexpectedServers())->duringInstantiation();
    }

    /**
     * @param \Memcached $memcache
     */
    function it_throws_if_configuring_a_server_and_other_ports_in_pool($memcache)
    {
        $memcache->getServerList()->willReturn(
            [
                ['host' => '123.4.5.6', 'port' => 11230, 'type' => 'TCP'],
            ]
        );

        $this->beConstructedWith($memcache, ['server' => ['host' => '123.4.5.6', 'port' => 123]]);

        $this->shouldThrow(CacheConfigurationException::unexpectedServers())->duringInstantiation();
    }

    /**
     * @param \Memcached $memcache
     */
    function it_does_not_care_about_servers_in_pool_if_not_providing_a_server_to_connect_to($memcache)
    {
        $this->newSubjectWith($memcache, ['server' => NULL]);

        $memcache->getServerList()->shouldNotHaveBeenCalled();
    }

    /**
     * @param \Memcached $memcache
     */
    function it_throws_if_addserver_fails($memcache)
    {
        // Have not yet been able to actually trigger a case where this doesn't return true IRL but the docs say it can
        // it should then only ever really be a validation or generic system error - this call does not in itself
        // attempt to lookup the DNS address or actually connect to the server
        $memcache->getServerList()->willReturn([]);
        $memcache->addServer('all juink here', 12324)->willReturn(FALSE);
        $this->beConstructedWith($memcache, ['server' => ['host' => 'all juink here', 'port' => 12324]]);

        $this->shouldThrow(CacheConfigurationException::addServerFailed('all juink here', 12324))
            ->duringInstantiation();
    }

    /**
     * @param \Memcached $memcache
     */
    function it_does_not_set_any_client_options_by_default($memcache)
    {
        $this->newSubjectWith($memcache, []);

        $memcache->setOption(Argument::cetera())->shouldNotHaveBeenCalled();
    }

    /**
     * @param \Memcached $memcache
     */
    function it_sets_all_configured_client_options($memcache)
    {
        $memcache->setOption(Argument::cetera())->willReturn(TRUE);

        $this->newSubjectWith(
            $memcache,
            ['client_options' => [\Memcached::OPT_SEND_TIMEOUT => 15, \Memcached::OPT_COMPRESSION => TRUE]]
        );

        $memcache->setOption(\Memcached::OPT_SEND_TIMEOUT, 15)->shouldHaveBeenCalled();
        $memcache->setOption(\Memcached::OPT_COMPRESSION, TRUE)->shouldHaveBeenCalled();
    }

    /**
     * @param \Memcached $memcache
     */
    function it_throws_if_setting_client_option_fails($memcache)
    {
        $memcache->setOption('junk', 'badbad')->willReturn(FALSE);

        $this->beConstructedWith($memcache, ['client_options' => ['junk' => 'badbad']]);

        $this->shouldThrow(CacheConfigurationException::setOptionFailed('junk', 'badbad'))
            ->duringInstantiation();
    }

    /**
     * @param \Memcached $memcache
     */
    function its_get_namespaced_key_returns_prefixed_key_with_existing_ns_prefix_where_present($memcache)
    {
        $memcache->get('namespace:my-stuff')->willReturn('abc123');

        $this->subject->getNamespacedKey('my-stuff', 'anykey')->shouldBe('my-stuff:abc123:anykey');
    }

    /**
     * @param \Memcached $memcache
     */
    function it_caches_namespace_prefix_in_class($memcache)
    {
        $memcache->get('namespace:my-stuff')->willReturn('i am ok to cache');
        $this->subject->getNamespacedKey('my-stuff', 'bar');
        $this->subject->getNamespacedKey('my-stuff', 'foo');

        $memcache->get('namespace:my-stuff')->shouldHaveBeenCalledTimes(1);
    }

    /**
     * @param \Memcached $memcache
     */
    function it_adds_and_uses_new_namespace_prefix_if_required($memcache)
    {
        $set_pfx = NULL;
        $memcache->get('namespace:my-stuff')->willReturn(FALSE);
        $memcache->getResultCode()->willReturn(\Memcached::RES_NOTFOUND);
        $memcache->add('namespace:my-stuff', Argument::any())
            ->will(
                function ($args, $memcache) use (&$set_pfx) {
                    $set_pfx = $args[1];
                    $memcache->get('namespace:my-stuff')->willReturn($set_pfx);

                    return TRUE;
                }
            );

        $this->subject->getNamespacedKey('my-stuff', 'anykey')
            ->shouldBe('my-stuff:'.$set_pfx.':anykey');

        expect($set_pfx)->toMatch('/^[a-zA-Z0-9]+$/');
    }

    /**
     * @param \Memcached $memcache
     */
    function it_uses_new_namespace_prefix_set_by_another_instance_if_race_condition_when_adding($memcache)
    {
        $memcache->get('namespace:my-stuff')->willReturn(FALSE);
        $memcache->getResultCode()->willReturn(\Memcached::RES_NOTFOUND);

        $memcache->add('namespace:my-stuff', Argument::any())->will(
            function ($args, $memcache) {
                // The add will fail, the get will return what another instance said
                $memcache->getResultCode()->willReturn(\Memcached::RES_NOTSTORED);
                $memcache->get('namespace:my-stuff')->willReturn('other-inst-pfx');

                return FALSE;
            }
        );

        $this->subject->getNamespacedKey('my-stuff', 'anykey')
            ->shouldBe('my-stuff:other-inst-pfx:anykey');

    }

    /**
     * @param \Memcached $memcache
     */
    function it_throws_on_failure_to_load_namespace_prefix($memcache)
    {
        $memcache->get('namespace:my-stuff')->willReturn(FALSE);
        $memcache->getResultCode()->willReturn(\Memcached::RES_SERVER_ERROR);
        $memcache->getResultMessage()->willReturn('Borked');

        $this->shouldThrowOperationFailure(
            function () { $this->subject->getNamespacedKey('my-stuff', 'any'); }
        );
    }

    /**
     * @param \Memcached $memcache
     */
    function it_throws_on_failure_to_create_namespace_prefix($memcache)
    {
        $memcache->get('namespace:my-stuff')->willReturn(FALSE);
        $memcache->getResultCode()->willReturn(\Memcached::RES_NOTFOUND);

        $memcache->add(Argument::cetera())->will(
            function ($args, $memcache) {
                // Set the result of this call
                $memcache->getResultCode()->willReturn(\Memcached::RES_SERVER_ERROR);
                $memcache->getResultMessage()->willReturn('Borked');

                return FALSE;
            }
        );

        $this->shouldThrowOperationFailure(
            function () { $this->subject->getNamespacedKey('my-stuff', 'any'); }
        );
    }

    /**
     * @param \Memcached $memcache
     */
    function it_throws_on_failure_to_read_created_namespace_prefix($memcache)
    {
        $memcache->get('namespace:my-stuff')->willReturn(FALSE);
        $memcache->getResultCode()->willReturn(\Memcached::RES_NOTFOUND);

        $memcache->add(Argument::cetera())->will(
            function ($args, $memcache) {
                // Make the final call fail
                $memcache->get('namespace:my-stuff')->willReturn(FALSE);
                $memcache->getResultCode()->willReturn(\Memcached::RES_SERVER_ERROR);
                $memcache->getResultMessage()->willReturn('Borked');

                return TRUE;
            }
        );

        $this->shouldThrowOperationFailure(
            function () { $this->subject->getNamespacedKey('my-stuff', 'any'); }
        );
    }

    /**
     * @param \Memcached $memcache
     */
    function its_get_reads_from_namespaced_key($memcache)
    {
        $memcache->get('namespace:foo')->willReturn('abcdefg');
        $memcache->get('foo:abcdefg:mykey')->willReturn('all the things');

        $this->subject->get('foo', 'mykey')->shouldBe('all the things');
    }

    /**
     * @param \Memcached $memcache
     */
    function its_get_returns_false_if_not_found($memcache)
    {
        $memcache->get('namespace:foo')->willReturn('abc123');
        $memcache->get('foo:abc123:mykey')->willReturn(FALSE);
        $memcache->getResultCode()->willReturn(\Memcached::RES_NOTFOUND);

        $this->subject->get('foo', 'mykey')->shouldBe(FALSE);

    }

    /**
     * @param \Memcached $memcache
     */
    function its_get_throws_on_error($memcache)
    {
        $memcache->get('namespace:foo')->willReturn('abc123');
        $memcache->get('foo:abc123:mykey')->willReturn(FALSE);
        $memcache->getResultCode()->willReturn(\Memcached::RES_SERVER_ERROR);
        $memcache->getResultMessage()->willReturn('I broke');

        $this->shouldThrowOperationFailure(
            function () { $this->subject->get('foo', 'mykey'); }
        );
    }


    /**
     * @param \Memcached $memcache
     */
    function its_set_stores_value_in_namespaced_key($memcache)
    {
        $memcache->get('namespace:bar')->willReturn('zxcde');
        $memcache->set('bar:zxcde:mykey', 'myvalue', 600)->willReturn(TRUE);

        $this->subject->set('bar', 'mykey', 'myvalue', 600)->shouldBe(TRUE);
    }

    /**
     * @param \Memcached $memcache
     */
    function its_set_throws_on_error($memcache)
    {
        $memcache->get('namespace:bar')->willReturn('zxcde');
        $memcache->set('bar:zxcde:mykey', 'myvalue', 390)->willReturn(FALSE);
        $memcache->getResultCode()->willReturn(\Memcached::RES_BAD_KEY_PROVIDED);
        $memcache->getResultMessage()->willReturn('I broke');

        $this->shouldThrowOperationFailure(
            function () { $this->subject->set('bar', 'mykey', 'myvalue', 390); }
        );
    }

    /**
     * @param \Memcached $memcache
     */
    function its_flush_everything_flushes_everything($memcache)
    {
        $memcache->flush()->willReturn(TRUE);

        $this->subject->flushEverything();
        $memcache->flush()->shouldHaveBeenCalled();
    }

    /**
     * @param \Memcached $memcache
     */
    function its_flush_everything_throws_on_failure($memcache)
    {
        $memcache->flush()->willReturn(FALSE);
        $memcache->getResultCode()->willReturn(\Memcached::RES_SERVER_ERROR);
        $memcache->getResultMessage()->willReturn('I broke');

        $this->shouldThrow(CacheNotAvailableException::class)
            ->during('flushEverything');
    }

    /**
     * @param \Memcached $memcache
     */
    function its_flush_namespace_generates_new_prefix_key($memcache)
    {
        $new_pfx = NULL;
        $memcache->set('namespace:bar', Argument::any())->will(
            function ($args, $memcached) use (& $new_pfx) {
                $new_pfx = $args[1];
                $memcached->get('namespace:bar')->willReturn($new_pfx);

                return TRUE;
            }
        );
        $this->subject->flushNamespace('bar');
        $this->subject->getNamespacedKey('bar', 'bite')
            ->shouldBe('bar:'.$new_pfx.':bite');

    }


    /**
     * @param \Memcached $memcache
     */
    function its_flush_namespace_clears_local_cache_of_prefix_key($memcache)
    {
        // Trigger it to load into cache
        $memcache->get('namespace:bar')->willReturn('old');
        $this->subject->getNamespacedKey('bar', 'any')
            ->shouldBe('bar:old:any');

        // Flush the NS
        $memcache->set('namespace:bar', Argument::any())->will(
            function ($args, $memcache) {
                $memcache->get('namespace:bar')->willReturn('new');
            }
        );
        $this->subject->flushNamespace('bar');

        $this->subject->getNamespacedKey('bar', 'any')
            ->shouldBe('bar:new:any');
    }

    /**
     * @param \Memcached $memcache
     */
    function its_flush_namespace_throws_on_failure($memcache)
    {
        $memcache->set('namespace:bar', Argument::any())->willReturn(FALSE);
        $memcache->getResultCode()->willReturn(\Memcached::RES_BAD_KEY_PROVIDED);
        $memcache->getResultMessage()->willReturn('I broke');

        $this->shouldThrowOperationFailure(
            function () { $this->subject->flushNamespace('bar'); }
        );

    }

    /**
     * @param \Memcached $memcache
     */
    function its_delete_deletes_item_with_namespaced_key($memcache)
    {
        $memcache->get('namespace:bar')->willReturn('ourprefix');
        $memcache->delete('bar:ourprefix:anykey')->willReturn(TRUE);

        $this->subject->delete('bar', 'anykey');

        $memcache->delete('bar:ourprefix:anykey')->shouldHaveBeenCalled();
    }

    /**
     * @param \Memcached $memcache
     */
    function its_delete_succeeds_if_item_does_not_exist($memcache)
    {
        $memcache->get('namespace:bar')->willReturn('ourprefix');
        $memcache->delete('bar:ourprefix:anykey')->willReturn(FALSE);
        $memcache->getResultCode()->willReturn(\Memcached::RES_NOTFOUND);

        $this->subject->delete('bar', 'anykey');

        $memcache->delete('bar:ourprefix:anykey')->shouldHaveBeenCalled();
    }

    /**
     * @param \Memcached $memcache
     */
    function its_delete_throws_on_error($memcache)
    {
        $memcache->get('namespace:bar')->willReturn('ourprefix');
        $memcache->delete('bar:ourprefix:anykey')->willReturn(FALSE);
        $memcache->getResultCode()->willReturn(\Memcached::RES_SERVER_ERROR);
        $memcache->getResultMessage()->willReturn('Borked');

        $this->shouldThrowOperationFailure(
            function () {
                $this->subject->delete('bar', 'anykey');
            }
        );
    }

    function its_cache_failure_hook_is_called_on_errors(\Memcached $memcache)
    {
        $this->beAnInstanceOf(ErrorSpyingMemcachedAdaptor::class);
        $this->beConstructedWith($memcache, []);
        $subject = $this->subject;
        /** @var ErrorSpyingMemcachedAdaptor $subject */

        $memcache->get('namespace:bar')->willReturn(FALSE);
        $memcache->getResultCode()->willReturn(\Memcached::RES_SERVER_ERROR);
        $memcache->getResultMessage()->willReturn('Borked');

        try {
            $this->subject->get('bar', 'whatever');
            throw new FailureException('Expected exception, none got');
        } catch (MemcachedOperationFailedException $e) {
            $subject->expectHookGotSingleFailure(\Memcached::RES_SERVER_ERROR, $e);
        }
    }

    /**
     * @param       $memcache
     * @param array $options
     */
    protected function newSubjectWith($memcache, array $options)
    {
        $this->beConstructedWith($memcache, $options);
        $this->subject->getWrappedObject();
    }

    protected function shouldThrowOperationFailure($callable)
    {
        try {
            $callable();
        } catch (MemcachedOperationFailedException $e) {
            // expected
            return;
        }

        throw new FailureException('Expected a '.MemcachedOperationFailedException::class.', none got');
    }

}


class ErrorSpyingMemcachedAdaptor extends NamespacedMemcachedAdaptor
{
    protected $fails = [];

    protected function onCacheFailure($result_code, MemcachedOperationFailedException $e, $operation_time_ms)
    {
        $this->fails[] = ['code' => $result_code, 'exception' => $e, 'time' => (string) $operation_time_ms];
    }

    public function expectHookGotSingleFailure($result_code, MemcachedOperationFailedException $e)
    {
        expect($this->fails)->toHaveCount(1);
        expect($this->fails[0]['code'])->toBe($result_code);
        expect($this->fails[0]['exception'])->toBe($e);
        // No easy way to phpspec-expect a float value, plus it'll be real world time so can't easily be strict about it
        // If it looks like a floating point number and less than a millisecond it should be fine
        expect($this->fails[0]['time'])->toMatch('/^0.[0-9]+$/');
    }

}