<?php
/**
* Defines the IllegalArgumentException VIA exception
*
* @author    Red61 Ltd
* @copyright Red61 Ltd
* @licence   proprietary
*/

namespace Red61\Via\Exception;
use Red61\Via\ApiRequest\ViaApiRequest;

/**
 * This exception is thrown when a request is skipped before sending to VIA because the provided data is not valid
 *
 * It is intended for use in cases when we know in advance that a request would result in a SoapFault - usually
 * IllegalArgumentException - if sent to VIA, to save the VIA server and your application from the wasted roundtrip.
 *
 * It is possible that this will be thrown due to invalid user-submitted data - you may wish to integrate the exception
 * with your form validation or request processing code to feed back the errors to the user in a structured way.
 *
 * [!!] This exception extends \SoapFault for legacy compatibility reasons, but may change
 *      in future to change to extend \BadMethodCallException or a similar runtime exception
 *      class.
 *
 */
class ClientInvalidRequestException extends IllegalArgumentException implements ViaException {

	/**
	 * @var \Red61\Via\ApiRequest\ViaApiRequest
	 */
	protected $request;

	/**
	 * @var string[]
	 */
	protected $invalidParams;

	/**
	 * @param ViaApiRequest $request       the request that was invalid
	 * @param string[]      $invalidParams the parameters that failed, as an array of $name => $error_message
	 */
	public function __construct(ViaApiRequest $request, $invalidParams)
	{
		$this->request       = $request;
		$this->invalidParams = $invalidParams;

		$message = "You attempted an invalid VIA ".$request->getSoapMethodName()." request".\PHP_EOL
			.\PHP_EOL
			."Details:".\PHP_EOL;
		foreach ($invalidParams as $name => $error) {
			$message .= " * $name: $error".\PHP_EOL;
		}

		parent::__construct('soap:Client', $message);
	}

	/**
	 * @return \string[] the parameters that failed, as an array of $name => $error_message
	 */
	public function getInvalidParameters()
	{
		return $this->invalidParams;
	}

	/**
	 * @return ViaApiRequest
	 */
	public function getThrowingRequest()
	{
		return $this->request;
	}
}
