<?php
/**
 * Defines SoapClientFactorySpec - specifications for Red61\Via\SoapClientFactory
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */

namespace spec\Red61\Via;

use PhpSpec\Exception\Example\FailureException;
use PhpSpec\Exception\Example\SkippingException;
use Red61\Via\ViaClassmap;
use spec\ObjectBehavior;
use Prophecy\Argument;

/**
 *
 * @see Red61\Via\SoapClientFactory
 */
class SoapClientFactorySpec extends ObjectBehavior
{
	const FAKE_WSDL_URL = 'http://foo.bar.com';
	/**
     * Use $this->subject to get proper type hinting for the subject class
     * @var \Red61\Via\SoapClientFactory
     */
	protected $subject;

	protected $old_xdebug_state;
	protected $has_xdebug;
	protected $old_error_reporting;

	function let()
	{
		$this->has_xdebug = function_exists('xdebug_is_enabled');
		$this->old_xdebug_state = ($this->has_xdebug && xdebug_is_enabled());
		$this->old_error_reporting = error_reporting();
		$this->beConstructedWith(array());
	}

	function letgo()
	{
		if ($this->has_xdebug && $this->old_xdebug_state)
		{
			xdebug_enable();
		}
		else if ($this->has_xdebug)
		{
			xdebug_disable();
		}
		error_reporting($this->old_error_reporting);
	}

	function it_is_initializable()
    {
		$this->subject->shouldHaveType('Red61\Via\SoapClientFactory');
	}

	function it_can_make_real_soap_client_with_valid_wsdl_address()
	{
		$wsdl_url = 'http://qa7.red61.com/f/api/soap?wsdl';
		if ( ! @ file_get_contents($wsdl_url)) {
		    throw new SkippingException($wsdl_url.' does not seem to be a valid WSDL URL');
		}
		$client = $this->subject->make($wsdl_url);
		$client->shouldBeAnInstanceOf('\SoapClient');
	}

	function it_can_make_fake_soap_client_when_class_injected()
	{
		$this->subject->beConstructedWith(array(), __NAMESPACE__.'\FakeSoapClient');
		$this->subject->make(self::FAKE_WSDL_URL)->shouldBeAnInstanceOf(__NAMESPACE__.'\FakeSoapClient');
	}

	function it_creates_client_with_wsdl_url_passed_to_make()
	{
		$client = $this->given_made_for_fake_class_with_default_options();
		$client->wsdl->shouldBe(self::FAKE_WSDL_URL);
	}

	function it_creates_client_with_compression_option_by_default()
	{
		$client = $this->given_made_for_fake_class_with_default_options();
		$client->options['compression']->shouldBe(SOAP_COMPRESSION_GZIP | SOAP_COMPRESSION_ACCEPT);
	}

	function it_creates_client_with_single_element_array_feature_by_default()
	{
		$client = $this->given_made_for_fake_class_with_default_options();
		$client->options['features']->shouldBe(SOAP_SINGLE_ELEMENT_ARRAYS);
	}

	function it_creates_client_with_trace_off_by_default()
	{
		$client = $this->given_made_for_fake_class_with_default_options();
		$client->options['trace']->shouldBe(FALSE);
	}

	function it_creates_client_with_exceptions_on_by_default()
	{
		$client = $this->given_made_for_fake_class_with_default_options();
		$client->options['exceptions']->shouldBe(TRUE);
	}

	function it_creates_client_with_core_classmap_by_default()
	{
		$client = $this->given_made_for_fake_class_with_default_options();
		$client->options['classmap']->shouldBe(ViaClassmap::$dataClassmap);
	}

	function it_overrides_default_client_options_from_constructor_if_set()
	{
		$client = $this->given_made_for_fake_class_with_options(array('features' => SOAP_WAIT_ONE_WAY_CALLS));
		$client->options['features']->shouldBe(SOAP_WAIT_ONE_WAY_CALLS);
	}

	function it_merges_classes_in_classmap_when_overriding()
	{
		$client = $this->given_made_for_fake_class_with_options(array(
			'classmap' => array('viaApiAddressResult' => '\MyRequestClass'))
		);
		$client->options['classmap']['viaApiAddressResult']->shouldBe('\MyRequestClass');
		$client->options['classmap']['viaApiDonationFund']->shouldBe(ViaClassmap::$dataClassmap['viaApiDonationFund']);
	}

	function it_sets_soap_client_remote_address_header_to_remote_address_if_defined()
	{
		$_SERVER['REMOTE_ADDR'] = '10.10.10.10';
		$client = $this->given_made_for_fake_class_with_default_options();

		$client->headers->shouldBeLike(new \SoapHeader('com.red61.via.api', 'REMOTE_ADDR', '10.10.10.10'));
	}

	function it_sets_soap_client_remote_address_header_to_localhost_if_not_defined()
	{
		unset($_SERVER['REMOTE_ADDR']);
		$client = $this->given_made_for_fake_class_with_default_options();

		$client->headers->shouldBeLike(new \SoapHeader('com.red61.via.api', 'REMOTE_ADDR', '127.0.0.1'));
	}

	function it_disables_xdebug_during_soap_client_creation()
	{
		$this->require_xdebug();
		xdebug_enable();
		$client = $this->given_made_for_fake_class_with_default_options();
		$client->xdebug_was_enabled->shouldBe(FALSE);
	}

	function it_reinstates_xdebug_state_after_soap_client_creation()
	{
		$this->require_xdebug();
		xdebug_enable();
		$client = $this->given_made_for_fake_class_with_default_options();
		expect(xdebug_is_enabled())->toBe(TRUE);
	}

	function it_reinstates_xdebug_state_after_exception_during_soap_client_creation()
	{
		$this->require_xdebug();
		xdebug_enable();
		try
		{
			$this->subject->beConstructedWith(array(), __NAMESPACE__ . '\FatalSoapClient');
			$this->subject->make(self::FAKE_WSDL_URL);
			throw new FailureException("Expected exception");
		} catch (\Exception $e) {
			// Expected
		}
		expect(xdebug_is_enabled())->toBe(TRUE);
	}

	function it_does_not_enable_xdebug_if_it_wasnt_already()
	{
		$this->require_xdebug();
		xdebug_disable();
		$client = $this->given_made_for_fake_class_with_default_options();
		expect(xdebug_is_enabled())->toBe(FALSE);
	}

	function it_disables_error_reporting_during_creation_of_soapclient()
	{
		error_reporting(E_ALL);
		$client = $this->given_made_for_fake_class_with_default_options();
		$client->error_reporting_was->shouldBe(0);
	}

	function it_restores_error_reporting_after_creation_of_soapclient()
	{
		error_reporting(E_ALL);
		$this->given_made_for_fake_class_with_default_options();
		expect(error_reporting())->toBe(E_ALL);
	}

	function it_restores_error_reporting_after_exception_during_soapclient_create()
	{
		error_reporting(E_ALL);
		try
		{
			$this->subject->beConstructedWith(array(), __NAMESPACE__ . '\FatalSoapClient');
			$this->subject->make(self::FAKE_WSDL_URL);
			throw new FailureException("Expected exception");
		} catch (\Exception $e) {
			// Expected
		}
		expect(error_reporting())->toBe(E_ALL);
	}

	/**
	 * @return FakeSoapClient
	 */
	protected function given_made_for_fake_class_with_default_options()
	{
		return $this->given_made_for_fake_class_with_options(array());
	}

	/**
	 * @param array $options
	 * @return FakeSoapClient
	 */
	protected function given_made_for_fake_class_with_options($options)
	{
		$this->subject->beConstructedWith($options, __NAMESPACE__ . '\FakeSoapClient');
		$client = $this->subject->make(self::FAKE_WSDL_URL);
		return $client;
	}

	protected function require_xdebug()
	{
		if ( ! $this->has_xdebug) {
			throw new SkippingException('xdebug extension is not installed');
		}
	}

}


class FakeSoapClient
{

	public $options;
	public $wsdl;
	public $headers;
	public $xdebug_was_enabled;
	public $error_reporting_was;

	public function __construct($wsdl, $options)
	{
		$this->options = $options;
		$this->wsdl = $wsdl;
		$this->xdebug_was_enabled  = function_exists('xdebug_is_enabled') && xdebug_is_enabled();
		$this->error_reporting_was = error_reporting();
	}

	public function __setSoapHeaders($headers)
	{
		$this->headers = $headers;
	}
}

class FatalSoapClient
{
	public function __construct()
	{
		throw new \Exception('Created a fatal soap client');
	}
}
