<?php
/**
 * Defines ViaApiClient
 *
 * @author    Andrew Coulton <andrew@ingenerator.com>
 * @copyright 2014 Red61 Ltd
 * @licence   proprietary
 */

namespace Red61\Via;
use Red61\Via\ApiRequest\apiGetBasketSessionIdRequest;
use Red61\Via\ApiRequest\apiClearBasketRequest;
use Red61\Via\ApiRequest\AuthenticatedRequest;
use Red61\Via\ApiRequest\BasketIdOptionalRequest;
use Red61\Via\ApiRequest\BasketIdRequiredRequest;
use Red61\Via\ApiRequest\LazyBasketCreatingRequest;
use Red61\Via\ApiRequest\LazyBasketMetadataSettingRequest;
use Red61\Via\ApiRequest\OrderCompletingRequest;
use Red61\Via\ApiRequest\PreflightFilteringRequest;
use Red61\Via\ApiRequest\ViaApiRequest;
use Red61\Via\DataObject\ViaApiOrderDetails;
use Red61\Via\Exception\BasketIdNotSetException;
use Red61\Via\Exception\CartNotFoundException;
use Red61\Via\Exception\ViaExceptionMapper;
use Red61\Via\Plugin\ViaApiCallNotification;
use Red61\Via\Plugin\ViaPluginManager;
use Red61\Via\SessionStorage\ViaSessionStorage;

/**
 * Controls all communication with the VIA API.
 *
 * Usually you will pass a client instance to an create an instance of the desired API service (eg ViaApiService) - the
 * service class provides method stubs for all available API operations, which are typehinted to show the appropriate
 * request and response types.
 *
 * @package Red61\Via
 * @see     \spec\Red61\Via\ViaApiClientSpec
 */
class ViaApiClient {

	const OBJECT_COMPATIBILITY_SILENT = 'silent';
	const OBJECT_COMPATIBILITY_WARN   = 'warn';
	const OBJECT_COMPATIBILITY_OFF    = 'off';

	/**
	 * Controls backwards-compatibility behaviour of VIA response objects, in particular whether fields are readable.
	 *
	 * OBJECT_COMPATIBILITY_SILENT - fields are readable as if they were public
	 * OBJECT_COMPATIBILITY_WARN   - fields are readable as if they were public, but a php notice will be emitted
	 * OBJECT_COMPATABILITY_OFF    - fields are not accessible, and must be accessed via getters and setters
	 *
	 * @var string
	 */
	public static $object_compatibility_mode = self::OBJECT_COMPATIBILITY_SILENT;

	/**
	 * @var string
	 */
	protected $basket_id;

	/**
	 * @var SoapClientFactory
	 */
	protected  $soap_factory;

	/**
	 * @var \SoapClient
	 */
	protected $soap_client;

	/**
	 * @var Exception\ViaExceptionMapper
	 */
	protected $exception_mapper;

	/**
	 * @var Plugin\ViaPluginManager
	 */
	protected $plugin_manager;

	/**
	 * @var string
	 */
	protected $wsdl;

	/**
	 * @var string
	 */
	protected $webkey;

	/**
	 * @var string
	 */
	protected $language;

	/**
	 * Requests that are waiting to be sent when a basket is created
	 *
	 * @var object[]
	 */
	protected $_on_basket_created_request_queue = array();

	/**
	 * @var ViaSessionStorage optional persistence for a user's basket ID between requests
	 */
	protected $session_storage;

	/**
	 * @param SoapClientFactory            $soap_factory     responsible for delayed creation of the SoapClient
	 * @param Exception\ViaExceptionMapper $exception_mapper converts SoapFault to VIA exception types
	 * @param Plugin\ViaPluginManager      $plugin_manager
	 * @param string                       $wsdl             URL of the VIA WSDL to use
	 * @param string                       $web_key          Your application's VIA API key
	 */
	public function __construct(
		SoapClientFactory  $soap_factory,
		ViaExceptionMapper $exception_mapper,
		ViaPluginManager   $plugin_manager,
		$wsdl,
		$web_key)
	{
		$this->soap_factory     = $soap_factory;
		$this->exception_mapper = $exception_mapper;
		$this->wsdl             = $wsdl;
		$this->webkey           = $web_key;
		$this->plugin_manager   = $plugin_manager;

		$this->language = isset($_SERVER['HTTP_ACCEPT_LANGUAGE']) && $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?
			filter_var( $_SERVER['HTTP_ACCEPT_LANGUAGE'], FILTER_SANITIZE_STRING ) :
			null;
	}

	/**
	 * Send the request and handle the response
	 *
	 * @param ViaApiRequest $request
	 *
	 * @throws Exception\ViaException on any failure to connect or if an API call fails.
	 * @return mixed
	 */
	public function send(ViaApiRequest $request)
	{
		if ($this->shouldDeferLazyBasketRequest($request))
		{
			$this->deferTillBasketCreated($request);
			return NULL;
		}

		$this->createBasketIfRequiredForLazyRequest($request);
		$this->assignWebKeyAndBasketIfRequired($request);
		$request->setLanguage($this->language);

		$call_notification = new ViaApiCallNotification($request);
		if ($request instanceof PreflightFilteringRequest)
		{
			$request->onBeforeViaCall($call_notification);
		}

		if ($call_notification->shouldSkipCall()) {
			return $call_notification->getResponse();
		} else {
			return $this->notifyPluginsAndTrySend($request, $call_notification);
		}
	}

	/**
	 * @param ViaApiRequest          $request
	 * @param ViaApiCallNotification $call_notification
	 *
	 * @return mixed
	 * @throws BasketIdNotSetException if there is no basket ID by this point for a request that requires it
	 */
	protected function notifyPluginsAndTrySend(ViaApiRequest $request, ViaApiCallNotification $call_notification)
	{
		$this->throwIfBasketIdRequiredAndNotSet($request);

		$this->plugin_manager->notify($call_notification);

		if ($call_notification->shouldSkipCall()) {
			$result = $call_notification->getResponse();
		} else {
			$result = $this->trySendOrThrow($request, $call_notification);
		}

		$this->updateBasketStateFromResponse($request, $result);

		$call_notification->_setSuccessfulResponse($result);
		$this->plugin_manager->notify($call_notification);
		return $result;
	}


	/**
	 * Set the current user's active basket ID
	 *
	 * A basket is created for the user the first time they add tickets (or other items) through the API. Once created
	 * its unique ID is available through ViaApiClient::getBasketId - your application code is responsible for storing
	 * this value for example in the user's session.
	 *
	 * On subsequent requests, you should retrieve the value from the user's session and set it here immediately after
	 * creating an instance of ViaApiClient.
	 *
	 * [!!] This method may trigger API requests, if a basket ID is set after you have attempted to send any
	 *      LazyBasketMetadataSettingRequests - like setAffiliate, setCampaignTracking, etc.
	 *
	 * @param string $id
	 *
	 * @return $this
	 */
	public function setBasketId($id)
	{
		$is_new_basket_creation = ($id AND ! $this->basket_id);
		$this->basket_id = $id;

		if ($this->soap_client) {
			$this->soap_client->__setCookie('basketId', $this->basket_id);
		}

		if ($this->session_storage) {
			$this->session_storage->saveBasketId($this->basket_id);
		}

		if ($is_new_basket_creation) {
			$this->sendDeferredBasketCreationRequests();
		}

		return $this;
	}

	/**
	 * Sets the language you are currently requesting data from the API in.
	 *
	 * Note, unlike when taking the default from the HTTP headers, this does not perform any validation or sanitising
	 * of the input - calling code should filter/validate the value as required.
	 *
	 * @param string $language
	 *
	 * @return $this
	 */
	public function setLanguage($language)
	{
		$this->language = $language ? : NULL;
		return $this;
	}

	/**
	 * Get the current user's active basket ID - this must be persisted between requests by your application
	 *
	 * @return string
	 * @see ViaApiClient::setBasketId
	 */
	public function getBasketId()
	{
		return $this->basket_id;
	}

	/**
	 * Creates an instance of the SoapClient the first time it is required for an API call.
	 *
	 * @return \SoapClient
	 */
	protected function createOrGetSoapClient()
	{
		if ($this->soap_client) {
			return $this->soap_client;
		}

		$this->soap_client = $this->soap_factory->make($this->wsdl);
		if ($id = $this->getBasketId())
		{
			$this->soap_client->__setCookie('basketId', $this->basket_id);
		}

		return $this->soap_client;
	}

	/**
	 * @param $result
	 *
	 * @return null
	 */
	protected function extractResultValue($result)
	{
		if ($result AND isset($result->value)) {
			return $result->value;
		} else {
			return NULL;
		}
	}

	/**
	 * Check if this request and result should cause the basketId to be reset to NULL - for example, because the
	 * user requested to clear their basket or because an order was completed
	 *
	 * @param ViaApiRequest $request
	 * @param mixed $result
	 *
	 * @return bool
	 */
	protected function isBasketClearingRequest(ViaApiRequest $request, $result)
	{
		if ($request instanceof apiClearBasketRequest) {
			return ($request->getRemoveBasket() AND $result);

		} elseif ($request instanceof OrderCompletingRequest) {
			return ($result instanceof ViaApiOrderDetails AND $result->wasSuccessful());

		} else {
			return FALSE;
		}
	}

	/**
	 * Throws if a request requires a basket ID property but one has not yet been set
	 *
	 * @param \Red61\Via\ApiRequest\ViaApiRequest $request
	 *
	 * @throws Exception\BasketIdNotSetException
	 */
	protected function throwIfBasketIdRequiredAndNotSet(ViaApiRequest $request)
	{
		if (
			($request instanceof BasketIdRequiredRequest)
			AND ! $request->getSessionId()
		) {
			throw new BasketIdNotSetException($request);
		}
	}

	/**
	 * Check if a request should be deferred till after a basket is created.
	 *
	 * @param ViaApiRequest $request
	 *
	 * @return bool
	 */
	protected function shouldDeferLazyBasketRequest(ViaApiRequest $request)
	{
		return (
			($request instanceof LazyBasketMetadataSettingRequest)
			AND ! $this->getBasketId()
		);
	}

	/**
	 * Add this request to the queue to call if a basket is eventually created
	 *
	 * @param ViaApiRequest $request
	 *
	 * @return void
	 */
	protected function deferTillBasketCreated(ViaApiRequest $request)
	{
		$this->_on_basket_created_request_queue[] = $request;
	}

	/**
	 * Work off the queue of requests that have been deferred until a basket is created
	 *
	 * @return void
	 */
	protected function sendDeferredBasketCreationRequests()
	{
		while ($request = array_shift($this->_on_basket_created_request_queue)) {
			$this->send($request);
		}
		return;
	}

	/**
	 * If the request is a LazyBasketCreatingRequest and a basket has not already been created, get one now
	 *
	 * @param ViaApiRequest $request
	 *
	 * @return void
	 */
	protected function createBasketIfRequiredForLazyRequest(ViaApiRequest $request)
	{
		$should_create = (($request instanceof LazyBasketCreatingRequest) AND ! $this->getBasketId());

		if ($should_create) {
			$this->send(apiGetBasketSessionIdRequest::create());
		}
	}

	/**
	 * Actually send a request to the VIA API, creating a SoapClient if required
	 *
	 * @param ViaApiRequest          $request
	 * @param ViaApiCallNotification $call_notification
	 *
	 * @return mixed
	 * @throws Exception\ViaException
	 */
	protected function trySendOrThrow(ViaApiRequest $request, $call_notification)
	{
		try {
			$headers = array();
			if ($request->getLanguage()) {
				$headers[] = new \SoapHeader('com.red61.via.api', 'Accept-Language', $request->getLanguage());
			}

			return $this->extractResultValue(
				$this->createOrGetSoapClient()
					->__soapCall($request->getSoapMethodName(), array($request), null, $headers)
			);
		} catch (\SoapFault $fault) {
			$fault = $this->exception_mapper->map($fault);

			if ($fault instanceof CartNotFoundException) {
				$this->setBasketId(NULL);
			}

			$call_notification->_setFailureException($fault);
			$this->plugin_manager->notify($call_notification);

			throw $fault;
		}
	}

	/**
	 * Assign the webkey and (if required) basket id on the request, or throw if state is not valid
	 *
	 * @param ViaApiRequest $request
	 *
	 * @return void
	 */
	protected function assignWebKeyAndBasketIfRequired(ViaApiRequest $request)
	{
		if ($request instanceof AuthenticatedRequest) {
			$request->setWebKey($this->webkey);
		}

		if ($request instanceof BasketIdRequiredRequest) {
			$request->setSessionId($this->getBasketId());
		} elseif ($request instanceof BasketIdOptionalRequest) {
			$request->setSessionId($this->getBasketId());
		}
	}

	/**
	 * Sets or clears the basket ID if required
	 *
	 * @param ViaApiRequest $request
	 * @param mixed $response
	 *
	 * @return void
	 */
	protected function updateBasketStateFromResponse(ViaApiRequest $request, $response)
	{
		if ($request instanceof apiGetBasketSessionIdRequest) {
			$this->setBasketId($response);
		}

		if ($this->isBasketClearingRequest($request, $response)) {
			$this->setBasketId(NULL);
		}
	}


	/**
	 * Assigns a ViaSessionStorage implementation to persist the user's basket ID between requests.
	 *
	 * This will trigger the API client to load the current basket ID from the session. The session storage
	 * will then receive a saveBasketId call whenever a basket is created or invalidated.
	 *
	 * [!!] Providing SessionStorage is optional - by default no session storage will be set and you will need
	 *      to manage persistence of the basket ID in your application code.
	 *
	 * @param ViaSessionStorage $storage
	 *
	 * @return $this
	 */
	public function connectSessionStorage(ViaSessionStorage $storage)
	{
		$this->session_storage = $storage;
		$this->setBasketId($storage->loadBasketId());
		return $this;
	}
}
