<?php
/**
 * Defines ViaApcCircuitBreakerStorageSpec - specifications for Red61\Via\CircuitBreaker\ViaApcCircuitBreakerStorage
 *
 * @author    Andrew Coulton <andrew@ingenerator.com>
 * @copyright 2014 Edinburgh International Book Festival Ltd
 * @licence   http://opensource.org/licenses/BSD-3-Clause
 */
namespace spec\Red61\Via\CircuitBreaker;

use PhpSpec\Exception\Example\SkippingException;
use Red61\Via\CircuitBreaker\ViaApcCircuitBreakerStorage;
use spec\ObjectBehavior;
use Prophecy\Argument;

/**
 *
 * @see Red61\Via\CircuitBreaker\ViaApcCircuitBreakerStorage
 */
class ViaApcCircuitBreakerStorageSpec extends ObjectBehavior
{
    /**
     * Use $this->subject to get proper type hinting for the subject class
     * @var \Red61\Via\CircuitBreaker\ViaApcCircuitBreakerStorage
     */
	protected $subject;

	function let()
	{
		$this->beAnInstanceOf('\spec\Red61\Via\CircuitBreaker\ViaApcCircuitBreakerStorageWithFakeClock');

		if ( ! apc_store('test', 'test'))
		{
			throw new SkippingException("Skipped because APC is not available - perhaps you need to configure apc.enable_cli = 1?");
		}
		apc_delete(array(
			ViaApcCircuitBreakerStorage::APCKEY_TRIPPED_AT,
			ViaApcCircuitBreakerStorage::APCKEY_RETRIED_AT,
			ViaApcCircuitBreakerStorage::APCKEY_RETRY_LOCK,
		));
	}

	function it_is_an_initializable_circuit_breaker_storage_implementation()
    {
		$this->subject->shouldHaveType('Red61\Via\CircuitBreaker\ViaApcCircuitBreakerStorage');
		$this->subject->shouldBeAnInstanceOf('Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage');
	}

	function it_can_be_tripped_and_reset()
	{
		$this->subject->isTripped()->shouldBe(FALSE);
		$this->subject->trip();
		$this->subject->isTripped()->shouldBe(TRUE);
		$this->subject->reset();
		$this->subject->isTripped()->shouldBe(FALSE);
	}

	function it_is_still_tripped_after_retry()
	{
		$this->subject->trip();
		$this->subject->markRetried();
		$this->subject->isTripped()->shouldBe(TRUE);
	}

	function it_was_first_to_trip_from_clean_state()
	{
		$this->subject->trip();
		$this->subject->wasFirstToTrip()->shouldBe(TRUE);
	}

	function it_was_not_first_to_trip_after_two_trips_on_first_instance()
	{
		$this->subject->trip();
		$this->subject->trip();
		$this->subject->wasFirstToTrip()->shouldBe(FALSE);
	}

	function it_was_not_first_to_trip_after_trip_on_another_instance()
	{
		$other = new ViaApcCircuitBreakerStorage;
		$other->trip();
		$this->subject->trip();
		$this->subject->wasFirstToTrip()->shouldBe(FALSE);
	}

	function it_was_not_retried_within_x_seconds_if_not_tripped()
	{
		$this->subject->wasRetriedWithinSeconds(9000)->shouldBe(FALSE);
	}

	function it_was_retried_within_x_seconds_after_trip()
	{
		$this->subject->trip();
		$this->waitSeconds(5);
		$this->subject->wasRetriedWithinSeconds(6)->shouldBe(TRUE);
	}

	function it_was_retried_within_x_seconds_after_retry()
	{
		$this->subject->trip();
		$this->waitSeconds(2);
		$this->subject->markRetried();
		$this->subject->wasRetriedWithinSeconds(1)->shouldBe(TRUE);

	}

	function it_was_not_retried_once_x_seconds_elapsed_after_trip()
	{
		$this->subject->trip();
		$this->waitSeconds(10);
		$this->subject->wasRetriedWithinSeconds(9)->shouldBe(FALSE);
	}

	function it_was_not_retried_once_x_seconds_elapsed_after_retry()
	{
		$this->subject->trip();
		$this->waitSeconds(2);
		$this->subject->markRetried();
		$this->waitSeconds(2);
		$this->subject->wasRetriedWithinSeconds(1)->shouldBe(FALSE);
	}

	function it_was_not_retried_after_reset()
	{
		$this->subject->trip();
		$this->subject->reset();
		$this->subject->wasRetriedWithinSeconds(1)->shouldBe(FALSE);
	}

	function it_grants_retry_lock_on_first_request()
	{
		$this->subject->tryLockForRetry(10)->shouldBe(TRUE);
	}

	function it_does_not_grant_retry_lock_if_already_held()
	{
		$this->subject->tryLockForRetry(10)->shouldBe(TRUE);
		$this->subject->tryLockForRetry(10)->shouldBe(FALSE);
		$this->waitSeconds(9);
		$this->subject->tryLockForRetry(10)->shouldBe(FALSE);
	}

	function it_grants_retry_lock_after_lifetime_expires()
	{
		$this->subject->tryLockForRetry(10)->shouldBe(TRUE);
		$this->waitSeconds(10);
		$this->subject->tryLockForRetry(10)->shouldBe(TRUE);
	}

	function its_errors_per_minute_is_zero_initially()
	{
		$this->subject->getErrorsPerMinute(uniqid())->shouldBe(0);
	}

	function it_increments_errors_per_minute_with_each_call()
	{
		$severity = uniqid();
		$this->shouldIncrementToRate($severity, 1);
		$this->shouldIncrementToRate($severity, 2);
		$this->shouldIncrementToRate($severity, 3);
	}

	function it_increments_errors_per_minute_for_errors_within_60_second_window()
	{
		$severity = uniqid();
		$this->shouldIncrementToRate($severity, 1);
		$this->waitSeconds(10);
		$this->shouldIncrementToRate($severity, 2);
	}

	function its_errors_per_minute_is_zero_after_minute_with_no_errors()
	{
		$severity = uniqid();
		$this->shouldIncrementToRate($severity, 1);
		$this->waitSeconds(60);
		$this->subject->getErrorsPerMinute($severity)->shouldBe(0);
	}

	function its_errors_per_minute_decreases_as_older_errors_drop_out_of_time_window()
	{
		$severity = uniqid();
		$this->shouldIncrementToRate($severity, 1);
		$this->shouldIncrementToRate($severity, 2);
		$this->waitSeconds(10);
		$this->shouldIncrementToRate($severity, 3);
		$this->waitSeconds(40);
		$this->shouldIncrementToRate($severity, 4);
		$this->waitSeconds(10);
		$this->subject->getErrorsPerMinute($severity)->shouldBe(2); // drops oldest two
	}

	function it_separates_error_counts_by_severity_level()
	{
		$this->shouldIncrementToRate(uniqid(), 1);
		$this->shouldIncrementToRate(uniqid(), 1);
	}

	protected function waitSeconds($seconds)
	{
		$this->subject->tickTock($seconds);
	}

	/**
	 * @param string $severity
	 * @param int    $i
	 */
	protected function shouldIncrementToRate($severity, $i)
	{
		$this->subject->incrementErrorRate($severity);
		$this->subject->getErrorsPerMinute($severity)->shouldBe($i);
	}

}

/**
 * Stub to allow testing without waiting for real clock ticks
 *
 * @package spec\Red61\Via\CircuitBreaker
 */
class ViaApcCircuitBreakerStorageWithFakeClock extends ViaApcCircuitBreakerStorage
{

	/**
	 * @var int
	 */
	protected $time = 1414779355;

	/**
	 * @param int $seconds
	 */
	public function tickTock($seconds)
	{
		$this->time += $seconds;
	}

	/**
	 * {@inheritdoc}
	 */
	protected function time()
	{
		return $this->time;
	}
}
