<?php
/**
 * Defines ViaApiClientSpec - specifications for Red61\Via\ViaApiClient
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */

namespace spec\Red61\Via;

use PhpSpec\Exception\Example\FailureException;
use Prophecy\Doubler\ClassPatch\ClassPatchInterface;
use Prophecy\Doubler\Generator\Node\ClassNode;
use Red61\Via\ApiRequest\apiAddSchemeToBasketRequest;
use Red61\Via\ApiRequest\apiGetBasketItemsRequest;
use Red61\Via\ApiRequest\apiGetBasketSessionIdRequest;
use Red61\Via\ApiRequest\apiGetBasketSummaryRequest;
use Red61\Via\ApiRequest\apiGetCustomerDetailsRequest;
use Red61\Via\ApiRequest\apiGetCustomerNameRequest;
use Red61\Via\ApiRequest\apiGetEventDetailsRequest;
use Red61\Via\ApiRequest\apiGetEventsRequest;
use Red61\Via\ApiRequest\apiClearBasketRequest;
use Red61\Via\ApiRequest\apiGetEventTypesRequest;
use Red61\Via\ApiRequest\apiGetPostageChargesRequest;
use Red61\Via\ApiRequest\apiGetVersionRequest;
use Red61\Via\ApiRequest\apiSetAffiliateRequest;
use Red61\Via\ApiRequest\OrderCompletingRequest;
use Red61\Via\ApiRequest\PreflightFilteringRequest;
use Red61\Via\ApiRequest\ViaApiRequest;
use Red61\Via\DataObject\ViaApiBasketSummary;
use Red61\Via\DataObject\ViaApiEvent;
use Red61\Via\Exception\CartNotFoundException;
use Red61\Via\Exception\CustomerMergedException;
use Red61\Via\Exception\ViaSoapFault;
use Red61\Via\Plugin\ViaApiCallNotification;
use Red61\Via\Plugin\ViaPluginInterface;
use Red61\Via\Plugin\ViaPluginManager;
use Red61\Via\Plugin\ViaPluginNotification;
use spec\ObjectBehavior;
use Prophecy\Argument;

/**
 *
 * @see Red61\Via\ViaApiClient
 */
class ViaApiClientSpec extends ObjectBehavior
{
	const WSDL_URL = 'https://some.via.server/?wsdl';
	const WEBKEY   = '5:2:32322313';

	/**
     * Use $this->subject to get proper type hinting for the subject class
     * @var \Red61\Via\ViaApiClient
     */
	protected $subject;

	/**
	 * @var FakePluginManager
	 */
	protected $plugin_manager;

	/**
	 * @param \Red61\Via\SoapClientFactory            $soap_factory
	 * @param \spec\Red61\Via\SoapClientInterface     $soap_client
	 * @param \Red61\Via\Exception\ViaExceptionMapper $exception_mapper
	 */
	function let($soap_factory, $soap_client, $exception_mapper)
	{
		$this->plugin_manager = new FakePluginManager;
		$this->beConstructedWith($soap_factory, $exception_mapper, $this->plugin_manager, self::WSDL_URL, self::WEBKEY);

		$soap_factory->make(Argument::cetera())->willReturn($soap_client);
		$exception_mapper->map(Argument::type('SoapFault'))->willReturnArgument();

		$soap_client->__setCookie('basketId', Argument::any())->willReturn(NULL);
		$soap_client->__soapCall(Argument::cetera())->willReturn(NULL);
	}

	function it_is_initializable()
    {
		$this->subject->shouldHaveType('Red61\Via\ViaApiClient');
	}

	function it_exposes_basket_id_property()
	{
		$id = '5926:1234:234567:23812323';
		$this->subject
			->setBasketId($id)
			->getBasketId()
			->shouldBe($id);
	}

	/**
	 * @param \Red61\Via\SoapClientFactory $soap_factory
	 */
	function it_does_not_create_soap_client_on_creation($soap_factory)
	{
		$this->subject->getWrappedObject();
		$soap_factory->make(Argument::any())->shouldNotHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\SoapClientFactory              $soap_factory
	 */
	function it_creates_soap_client_to_send_first_request($soap_factory)
	{
		$this->subject->send(new apiGetEventsRequest);
		$soap_factory->make(Argument::cetera())->shouldHaveBeenCalledTimes(1);
		$this->subject->send(new apiGetEventsRequest);
		$soap_factory->make(Argument::cetera())->shouldHaveBeenCalledTimes(1);
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface $soap_client
	 */
	function it_does_not_set_basket_id_cookie_on_soap_client_create_when_no_basket_id($soap_client)
	{
		$this->subject->send(new apiGetEventsRequest);
		$soap_client->__setCookie(Argument::cetera())->shouldNotHaveBeenCalled();
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface $soap_client
	 */
	function it_sets_basket_id_cookie_on_soap_client_create_when_basket_exists($soap_client)
	{
		$basketId = '5926:123:456';
		$this->subject->setBasketId($basketId)
			->send(new apiGetEventsRequest);

		$soap_client->__setCookie('basketId', $basketId)->shouldHaveBeenCalled();
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface $soap_client
	 */
	function it_sends_requests_with_the_soap_client($soap_client)
	{
		$request = new apiGetEventsRequest;
		$this->subject->send($request);
		$soap_client->__soapCall(apiGetEventsRequest::SOAP_METHOD_NAME, array($request))
			->shouldHaveBeenCalled();
		$soap_client->__soapCall(Argument::cetera())->shouldHaveBeenCalledTimes(1);
	}

	/**
	 * @param \Red61\Via\ApiRequest\apiGetEventsRequest $request
	 */
	function it_sets_webkey_on_outgoing_authenticated_requests($request)
	{
		$this->subject->send($request);
		$request->setWebKey(self::WEBKEY)->shouldHaveBeenCalled();
	}

	function it_does_not_set_webkey_on_outgoing_non_authenticated_requests()
	{
		$request = apiGetEventTypesRequest::create();
		expect($request)->notToBeAnInstanceOf('\Red61\Via\ApiRequest\AuthenticatedRequest');
		$this->subject->send($request);
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface $soap_client
	 */
	function it_returns_the_value_from_the_soap_response($soap_client)
	{
		$this->given_soap_method_will_return_response_value($soap_client, apiGetEventsRequest::SOAP_METHOD_NAME, 'string:foo');
		$this->subject->send(new apiGetEventsRequest)->shouldBe('string:foo');
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface $soap_client
	 */
	function it_returns_null_if_soap_response_has_no_value($soap_client)
	{
		$this->given_soap_method_will_return_response_value($soap_client, apiGetEventsRequest::SOAP_METHOD_NAME, NULL);
		$this->subject->send(new apiGetEventsRequest)->shouldBe(NULL);
	}

	/**
	 * @param \Red61\Via\SoapClientFactory            $soap_factory
	 * @param \Red61\Via\Exception\ViaExceptionMapper $exception_mapper
	 *
	 * @throws \PhpSpec\Exception\Example\FailureException
	 */
	function it_maps_and_rethrows_soapfaults_from_soapclient_creation($soap_factory, $exception_mapper)
	{
		$fault    = new \SoapFault('soap:Server', 'Bad Gateway');
		$newfault = new ViaSoapFault('soap:Server', 'Bad Gateway');
		$soap_factory->make(Argument::any())->willThrow($fault);
		$exception_mapper->map($fault)->willReturn($newfault);

		try {
			$this->subject->send(new apiGetVersionRequest);
			throw new FailureException('Expected exception not thrown');
		} catch (ViaSoapFault $e) {
			expect($e)->toBe($newfault);
		}
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface     $soap_client
	 * @param \Red61\Via\Exception\ViaExceptionMapper $exception_mapper
	 *
	 * @throws \PhpSpec\Exception\Example\FailureException
	 */
	function it_maps_and_rethrows_soapfaults_from_api_call($soap_client, $exception_mapper)
	{
		$fault    = new \SoapFault('soap:Server', 'Customer Merged');
		$newfault = new CustomerMergedException('soap:Server', 'Customer Merged');

		$soap_client->__soapCall(Argument::any(), Argument::any())->willThrow($fault);
		$exception_mapper->map($fault)->willReturn($newfault);

		try {
			$this->subject->send(new apiGetCustomerNameRequest);
			throw new FailureException('Expected exception not thrown');
		} catch (CustomerMergedException $e) {
			expect($e)->toBe($newfault);
		}
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface     $soap_client
	 * @param \Red61\Via\Exception\ViaExceptionMapper $exception_mapper
	 *
	 * @throws \PhpSpec\Exception\Example\FailureException
	 */
	function it_clears_the_basket_id_property_on_a_cart_not_found_exception_from_any_call($soap_client, $exception_mapper)
	{
		$fault    = new \SoapFault('soap:Server', 'Basket Expired');
		$newfault = new CartNotFoundException('soap:Server', 'Basket Expired');

		$soap_client->__setCookie(Argument::any(), Argument::any())->willReturn(NULL);
		$soap_client->__soapCall(Argument::any(), Argument::any())->willThrow($fault);
		$exception_mapper->map($fault)->willReturn($newfault);

		$this->subject->setBasketId('512323232');
		try {
			$this->subject->send(new apiGetBasketSummaryRequest);
			throw new FailureException("Expected exception not thrown");
		} catch (CartNotFoundException $e) {
			$this->subject->getBasketId()->shouldBe(NULL);
		}
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface                $soap_client
	 */
	function it_sets_the_basket_id_property_on_successfully_creating_a_basket($soap_client)
	{
		$basketid = '5926:2323:2323232';
		$this->given_api_will_return_basket_id_for_get_basket_session_id($soap_client, $basketid);

		$this->subject->send(new apiGetBasketSessionIdRequest)->shouldBe($basketid);
		$this->subject->getBasketId()->shouldBe($basketid);
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface         $soap_client
	 */
	function it_sets_the_soap_client_basket_id_cookie_on_successfully_creating_a_basket($soap_client)
	{
		$basketid = '5926:2323:2323232';
		$this->given_api_will_return_basket_id_for_get_basket_session_id($soap_client, $basketid);

		$this->subject->send(new apiGetBasketSessionIdRequest)->shouldBe($basketid);
		$soap_client->__setCookie('basketId', $basketid)->shouldHaveBeenCalled();
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface         $soap_client
	 */
	function it_clears_the_basket_id_property_on_successfully_clearing_a_basket_with_remove_true($soap_client)
	{
		$this->subject->setBasketId('5926:2323:23232323');
		$this->given_soap_method_will_return_response_value($soap_client, apiClearBasketRequest::SOAP_METHOD_NAME, TRUE);
		$soap_client->__setCookie(Argument::cetera())->willReturn(NULL);

		$request = apiClearBasketRequest::create()->setRemoveBasket(TRUE);
		$this->subject->send($request)->shouldBe(TRUE);
		$this->subject->getBasketId()->shouldBe(NULL);
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface         $soap_client
	 */
	function it_does_not_clear_the_basket_id_property_on_successfully_clearing_basket_with_remove_false($soap_client)
	{
		$this->subject->setBasketId('5926:2323:23232323');
		$this->given_soap_method_will_return_response_value($soap_client, apiClearBasketRequest::SOAP_METHOD_NAME, TRUE);
		$soap_client->__setCookie(Argument::cetera())->willReturn(NULL);

		$request = apiClearBasketRequest::create()->setRemoveBasket(FALSE);
		$this->subject->send($request)->shouldBe(TRUE);
		$this->subject->getBasketId()->shouldBe('5926:2323:23232323');
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface         $soap_client
	 */
	function it_does_not_clear_the_basket_id_property_on_failing_to_clear_basket_with_remove_true($soap_client)
	{
		$this->subject->setBasketId('5926:2323:23232323');
		$this->given_soap_method_will_return_response_value($soap_client, apiClearBasketRequest::SOAP_METHOD_NAME, FALSE);
		$soap_client->__setCookie(Argument::cetera())->willReturn(NULL);

		$request = apiClearBasketRequest::create()->setRemoveBasket(TRUE);
		$this->subject->send($request)->shouldBe(FALSE);
		$this->subject->getBasketId()->shouldBe('5926:2323:23232323');
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface          $soap_client
	 * @param \Red61\Via\DataObject\ViaApiOrderDetails     $order_details
	 */
	function it_clears_the_basket_id_after_successful_order_completing_request($soap_client, $order_details)
	{
		$this->subject->setBasketId('5926:2323:23232323');
		$order_details->wasSuccessful()->willReturn(TRUE);

		$this->given_soap_method_will_return_response_value(
			$soap_client, DummyOrderCompletingRequest::SOAP_METHOD_NAME, $order_details->getWrappedObject()
		);
		$soap_client->__setCookie(Argument::cetera())->willReturn(NULL);

		$this->subject->send(new DummyOrderCompletingRequest)->shouldBe($order_details);
		$this->subject->getBasketId()->shouldBe(NULL);
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface          $soap_client
	 * @param \Red61\Via\DataObject\ViaApiOrderDetails     $order_details
	 */
	function it_does_not_clear_the_basket_id_on_failed_order_creation($soap_client, $order_details)
	{
		$this->subject->setBasketId('5926:2323:23232323');
		$order_details->wasSuccessful()->willReturn(FALSE);

		$this->given_soap_method_will_return_response_value(
			$soap_client, DummyOrderCompletingRequest::SOAP_METHOD_NAME, $order_details->getWrappedObject()
		);
		$soap_client->__setCookie(Argument::cetera())->willReturn(NULL);

		$this->subject->send(new DummyOrderCompletingRequest)->shouldBe($order_details);
		$this->subject->getBasketId()->shouldBe('5926:2323:23232323');
	}

	function it_sets_basket_id_on_outgoing_requests_that_require_it()
	{
		$request = new apiGetBasketItemsRequest;

		$this->subject->setBasketId('5926:4:321');
		$this->subject->send($request);

		expect($request->getSessionId())->toBe('5926:4:321');
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface          $soap_client
	 */
	function it_creates_a_basket_before_a_lazy_basket_creating_request_if_required($soap_client)
	{
		$request = new apiAddSchemeToBasketRequest;
		expect($request)->toBeAnInstanceOf('\Red61\Via\ApiRequest\LazyBasketCreatingRequest');

		$this->given_api_will_return_basket_id_for_get_basket_session_id($soap_client);
		$this->given_soap_method_will_return_response_value($soap_client, apiAddSchemeToBasketRequest::SOAP_METHOD_NAME, TRUE);

		$this->subject->send($request)->shouldBe(TRUE);

		$soap_client->__soapCall(apiGetBasketSessionIdRequest::SOAP_METHOD_NAME, Argument::any())
			->shouldHaveBeenCalledTimes(1);
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface          $soap_client
	 */
	function it_does_not_create_a_basket_before_lazy_basket_creating_request_if_basket_already_created($soap_client)
	{
		$this->subject->setBasketId('5926:43:2323232');

		$request = new apiAddSchemeToBasketRequest;
		expect($request)->toBeAnInstanceOf('\Red61\Via\ApiRequest\LazyBasketCreatingRequest');
		$soap_client->__setCookie(Argument::cetera())->willReturn(NULL);

		$this->given_soap_method_will_return_response_value($soap_client, apiAddSchemeToBasketRequest::SOAP_METHOD_NAME, TRUE);

		$this->subject->send($request)->shouldBe(TRUE);

		$soap_client->__soapCall(apiGetBasketSessionIdRequest::SOAP_METHOD_NAME, Argument::any())
			->shouldNotHaveBeenCalled();
	}


	function it_throws_if_basket_not_created_before_a_requires_basket_request()
	{
		$request = new apiClearBasketRequest;
		expect($request)->toBeAnInstanceOf('\Red61\Via\ApiRequest\BasketIdRequiredRequest');
		try {
			$this->subject->send($request);
			throw new FailureException("Expected a BasketIdNotSetException");
		} catch (\Red61\Via\Exception\BasketIdNotSetException $e) {
			// OK
		}
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface          $soap_client
     */
	function it_sends_lazy_basket_metadata_setting_requests_immediately_if_basket_created($soap_client)
	{
		$this->subject->setBasketId('5926:1:123456');

		$request = new apiSetAffiliateRequest;
		expect($request)->toBeAnInstanceOf('Red61\Via\ApiRequest\LazyBasketMetadataSettingRequest');

		$this->subject->send($request);

		$soap_client->__soapCall($request::SOAP_METHOD_NAME, array($request))->shouldHaveBeenCalledTimes(1);

		// Verify that requests aren't queued and resent if the basket was already existing the first time
		$this->subject->setBasketId(NULL);

		$this->given_api_will_return_basket_id_for_get_basket_session_id($soap_client);
		$this->subject->send(new apiGetBasketSessionIdRequest());
		$soap_client->__soapCall($request::SOAP_METHOD_NAME, array($request))->shouldHaveBeenCalledTimes(1);
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface          $soap_client
	 */
	function it_does_not_send_lazy_basket_metadata_setting_requests_if_no_basket($soap_client)
	{
		$request = new apiSetAffiliateRequest;
		expect($request)->toBeAnInstanceOf('Red61\Via\ApiRequest\LazyBasketMetadataSettingRequest');

		$this->subject->send($request);
		$soap_client->__soapCall(Argument::cetera())->shouldNotHaveBeenCalled();
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface          $soap_client
	 */
	function it_sends_lazy_basket_metadata_setting_requests_immediately_basket_id_is_set_with_setter($soap_client)
	{
		$request = new apiSetAffiliateRequest;
		expect($request)->toBeAnInstanceOf('Red61\Via\ApiRequest\LazyBasketMetadataSettingRequest');

		$this->subject->send($request);
		$this->subject->setBasketId('5926:2:2323');
		$soap_client->__soapCall($request::SOAP_METHOD_NAME, array($request))->shouldHaveBeenCalledTimes(1);
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface          $soap_client
	 */
	function it_sends_lazy_basket_metadata_setting_requests_immediately_basket_id_is_set_with_api_request($soap_client)
	{
		$request = new apiSetAffiliateRequest;
		expect($request)->toBeAnInstanceOf('Red61\Via\ApiRequest\LazyBasketMetadataSettingRequest');

		$this->given_api_will_return_basket_id_for_get_basket_session_id($soap_client);
		$soap_client->__soapCall($request::SOAP_METHOD_NAME, Argument::cetera())->willReturn(NULL);

		$this->subject->send($request);
		$this->subject->send(new apiGetBasketSessionIdRequest);
		$soap_client->__soapCall($request::SOAP_METHOD_NAME, array($request))->shouldHaveBeenCalledTimes(1);
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface          $soap_client
	 */
	function it_does_not_send_lazy_basket_metadata_setting_requests_if_basket_creation_fails($soap_client)
	{
		$request = new apiSetAffiliateRequest;
		expect($request)->toBeAnInstanceOf('Red61\Via\ApiRequest\LazyBasketMetadataSettingRequest');

		$this->given_api_will_return_basket_id_for_get_basket_session_id($soap_client, NULL);

		$this->subject->send($request);
		$this->subject->send(new apiGetBasketSessionIdRequest);
		$soap_client->__soapCall($request::SOAP_METHOD_NAME, array($request))->shouldNotHaveBeenCalled();
	}

	function it_notifies_plugins_before_a_call_is_sent()
	{
		$request = new apiGetVersionRequest;

		$this->subject->send($request);

		$notification_0 = $this->plugin_manager->notifications[0];
		expect($notification_0->getCallStatus())->toBe(ViaApiCallNotification::ON_BEFORE_CALL);
		expect($notification_0->getRequest())->toBe($request);
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface $soap_client
	 */
	function it_notifies_plugins_with_response_when_a_call_is_successful($soap_client)
	{
		$response = '5.2.0';
		$this->given_soap_method_will_return_response_value($soap_client, apiGetVersionRequest::SOAP_METHOD_NAME, $response);
		$request = new apiGetVersionRequest;

		$this->subject->send($request);
		$notification_1 = $this->plugin_manager->notifications[1];
		expect($notification_1->getCallStatus())->toBe(ViaApiCallNotification::ON_CALL_SUCCESS);
		expect($notification_1->getRequest())->toBe($request);
		expect($notification_1->getResponse())->toBe($response);
	}

	/**
	 * @param \spec\Red61\Via\SoapClientInterface             $soap_client
	 * @param \Red61\Via\Exception\ViaExceptionMapper         $exception_mapper
	 * @param \Red61\Via\ApiRequest\apiGetCustomerNameRequest $request
	 *
	 * @throws \PhpSpec\Exception\Example\FailureException
	 */
	function it_notifies_plugins_with_response_when_a_call_fails($soap_client, $exception_mapper, $request)
	{
		$fault    = new \SoapFault('soap:Server', 'Customer Merged');
		$newfault = new CustomerMergedException('soap:Server', 'Customer Merged');

		$soap_client->__soapCall(Argument::any(), Argument::any())->willThrow($fault);
		$exception_mapper->map($fault)->willReturn($newfault);

		try {
			$this->subject->send($request);
			throw new FailureException('Expected exception not thrown');
		} catch (CustomerMergedException $e) {
			$notification_1 = $this->plugin_manager->notifications[1];
			expect($notification_1->getCallStatus())->toBe(ViaApiCallNotification::ON_CALL_FAILED);
			expect($notification_1->getRequest())->toBe($request);
			expect($notification_1->getResponse())->toBe(NULL);
			expect($notification_1->getException())->toBe($e);
		}
	}

	/**
	 * @param \Red61\Via\SoapClientFactory $soap_factory
	 * @param \Red61\Via\ApiRequest\apiGetEventDetailsRequest $request
	 */
	function it_skips_soap_client_and_api_call_and_returns_response_from_plugin_if_set_during_before_call($soap_factory, $request)
	{
		$event = new ViaApiEvent;
		$this->plugin_manager->trigger_skipping_response = TRUE;
		$this->plugin_manager->skipping_response = $event;

		$this->subject->send($request)->shouldBe($event);
		$soap_factory->make(Argument::any())->shouldNotHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\ApiRequest\apiGetEventDetailsRequest $request
	 */
	function it_notifies_plugins_with_response_when_a_call_is_skipped($request)
	{
		$event = new ViaApiEvent;
		$this->plugin_manager->trigger_skipping_response = TRUE;
		$this->plugin_manager->skipping_response = $event;

		$this->subject->send($request)->shouldBe($event);
		$notification_1 = $this->plugin_manager->notifications[1];
		expect($notification_1->getCallStatus())->toBe(ViaApiCallNotification::ON_CALL_SUCCESS);
		expect($notification_1->getResponse())->toBe($event);
	}

	/**
	 * @param \spec\Red61\Via\DummyPreflightFilteringRequest $request
	 */
	function it_calls_preflight_filtering_request_handlers($request)
	{
		$this->subject->send($request);

		$request->onBeforeViaCall(Argument::any())->shouldHaveBeenCalled();
	}

	/**
	 * @param \spec\Red61\Via\DummyPreflightFilteringRequest $request
	 */
	function it_returns_response_from_preflight_filter_if_skip_call_set($request)
	{
		$response = new ViaApiBasketSummary;
		$this->given_request_will_set_skipping_response_from_preflight($request, $response);

		$this->subject->send($request)->shouldBe($response);
	}

	/**
	 * @param \Red61\Via\SoapClientFactory $soap_factory
	 * @param \spec\Red61\Via\DummyPreflightFilteringRequest $request
	 */
	function it_does_not_trigger_plugins_or_send_call_if_preflight_request_skips_call($soap_factory, $request) {
		$response = new ViaApiBasketSummary;
		$this->given_request_will_set_skipping_response_from_preflight($request, $response);

		$this->subject->send($request);

		$soap_factory->make(Argument::any())->shouldNotHaveBeenCalled();
		expect($this->plugin_manager->notifications)->toBe(array());
	}


	/**
	 * @param SoapClientInterface $soap_client
	 * @param string              $basketid
	 */
	protected function given_api_will_return_basket_id_for_get_basket_session_id($soap_client, $basketid = '5926:2:23')
	{
		$response        = new \stdClass();
		$response->value = $basketid;
		$soap_client->__soapCall(apiGetBasketSessionIdRequest::SOAP_METHOD_NAME, Argument::any())->willReturn($response);
	}

	/**
	 * @param SoapClientInterface $soap_client
	 * @param string              $method
	 * @param mixed               $value
	 */
	protected function given_soap_method_will_return_response_value($soap_client, $method, $value)
	{
		if ($value !== NULL) {
			$response        = new \stdClass();
			$response->value = $value;
		} else {
			$response        = NULL;
		}

		$soap_client->__soapCall($method, Argument::any())->willReturn($response);
	}/**
 * @param $request
 * @param $response
 */protected function given_request_will_set_skipping_response_from_preflight($request, $response)
{
	$request->onBeforeViaCall(Argument::any())->will(function ($args) use ($response) {
		$notification = $args[0];
		/** @var ViaApiCallNotification $notification */
		$notification->skipCallAndReturn($response);
	});
}

	// how to handle:
	// - requests that have canned responses or don't want to be sent (invalid, or no basket ID)
}

interface SoapClientInterface {

	public function __construct($wsdl, $options);
	public function __setSoapHeaders($headers);
	public function __setCookie($name, $value);
	public function __soapCall($method, $params);
}


/**
 * Dummy request class to test behaviour of the OrderCompletingRequest functionality
 *
 * @package spec\Red61\Via
 */
class DummyOrderCompletingRequest extends ViaApiRequest implements OrderCompletingRequest {
	const SOAP_METHOD_NAME = 'apiAnything';
	public function setWebKey() {}
}


/**
 * Dummy request class to test behaviour of the PreflightFilteringRequest functionality
 *
 * @package spec\Red61\Via
 */
class DummyPreflightFilteringRequest extends ViaApiRequest implements PreflightFilteringRequest {

	public function onBeforeViaCall(ViaApiCallNotification $notification) {}
}

/**
 * Fake plugin manager to allow easier catching and inspecting of the events dispatched
 *
 * @package spec\Red61\Via
 */
class FakePluginManager implements ViaPluginManager {

	/**
	 * @var ViaApiCallNotification[]
	 */
	public $notifications = array();

	public $trigger_skipping_response = FALSE;
	public $skipping_response         = NULL;

	public function notify(ViaPluginNotification $notification)
	{
		/** @var ViaApiCallNotification $notification */
		$this->notifications[] = clone($notification);
		if ($this->trigger_skipping_response AND ($notification->getNotificationName() === ViaApiCallNotification::ON_BEFORE_CALL)) {
			$notification->skipCallAndReturn($this->skipping_response);
		}
	}

	public function registerPlugin(ViaPluginInterface $plugin, $only_notifications = NULL)
	{
	}
}
