VIA PHP Wrapper
===============

This package provides an object oriented wrapper for interacting with the VIA SOAP API from a
a PHP application.

It is primarily designed for public-facing web ticketing applications but can be used to
support a wider range of systems built against the VIA API.

**This package is copyright &copy; 2014 Red61 Ltd and is made available on a proprietary licence
and under a non-disclosure agreement. Contact Red61 for further details of the licence terms.**

# Installation

## Composer (recommended)

Where possible, we recommend installing the wrapper with [composer](https://getcomposer.org/) as
the easiest way to keep up to date with new releases and get early access to pre-release versions
for testing and review.

Because this is a private package, there are a couple of extra configuration steps required so
that composer can access it.

#### Request access to the github project

If you do not yet have access to https://github.com/red61/via-wrapper-php first contact [support@red61.com](mailto:support@red61.com)
and provide:

 * the github usernames of any developers who will be working on the project. If you have a large or
   changing team and/or run continuous integration or deployment scripts you may prefer to register a
   new [machine user account on github](https://developer.github.com/guides/managing-deploy-keys/#machine-users).
 * details of the Red61 client you are working for and the site you are developing.

Once your application has been reviewed you will be granted read-only access to the project repository.

#### Install composer (if you don't already have it)

On a *nix system, you can install composer globally with the following shell commands:

```bash
curl -sS https://getcomposer.org/installer | php
sudo mv composer.phar /usr/local/bin/composer
```

Test this worked by running `composer` - you should see a list of available commands. If you have any problems,
check the [composer install documentation](https://getcomposer.org/doc/00-intro.md) for more options.

#### Add the wrapper and private package repository to your composer.json

Your `composer.json` file will need to contain at least the following. Note that the Red61 packages are not
listed on the public Packagist server so you need the extra "repositories" entry for the php-packages.red61.com
metadata server. Traditionally you would place the composer.json in the root directory of your project.

```json
{
  "require": {
	"red61/via-wrapper-php": "x.x.x" // set desired version
  },
  "repositories": [
    {"type": "composer", "url": "https://php-packages.red61.com"}
  ]
}
```

This configuration asks composer to install the most recent 5.2.0 beta version of the wrapper. Once
we have released 5.2.0 you will likely want to instead use `"red61/via-wrapper-php" : "5.2.*"` which
tells composer only to install stable minor version updates.

The possible version constraints are fully described [in the composer documentation](https://getcomposer.org/doc/01-basic-usage.md#the-require-key).

#### Install your composer dependencies

Open a shell in the same directory that you placed the composer.json file and run

```bash
composer install
```

Composer will prompt you for your github username and password (or you can enter the details for your
machine user). These details are not stored, but are used to create a personal access token which will
be saved to disk in ~/.composer/auth.json. Anyone with this file can use the details to access your
github account - another reason why setting up a machine user with limited access may be worthwhile.

You can revoke this token [on your github settings page](https://github.com/settings/applications) at
any time.

Once authenticated, composer will download the most recent wrapper satisfying your version constraint
and place it (by default) in *{composer_path}*/vendor/red61/via-wrapper-php.

It will also create a composer.lock file alongside the composer.json - this file tracks the exact
version you installed. From now on, running `composer install` will only ever reinstall this exact
version - to get a newer version you instead need to run `composer update`.

#### Commit your composer.json and composer.lock

You should add composer.json and composer.lock to your source code repository. You don't need to
commit the actual package.

#### Update your application if required

If your application already includes the composer autoloader, you should now be ready to get started
with the wrapper.

If not, then early in your application you will need to add:

```php
require_once(PROJECT_BASE_DIR.'/vendor/autoload.php');
```

If you have a legacy application that includes red61_via.php or red61_utils.php you will need to update
these paths to point to the package in your /vendor directory.

#### Update your deployment scripts if required

Your deployment script may also need to run `composer install`. If so, it will also need credentials -
you can either provision the auth.json file onto your remote machines, or set them up with an SSH
key that is linked to your github account. See the composer and github documentation for more.

If your deploy scripts upload files from your local machine (eg with rsync) you may not need
to run composer install remotely.

#### Update your wrapper version when it changes

In future, to update to the most recent version of the wrapper you will simply need to run `composer update`
and commit your composer.json and composer.lock files.

## From distribution zip files

Installing from distribution zip files is probably quicker than setting up composer the first time, but
will be slower than `composer update` for future updates.

First, unzip the distribution file to a suitable place in your project source tree and commit it to
your source code repository.

You will need to register an autoloader that can load the relevant VIA classes as they are required.
The red61_via.php file will do this for you if required, just include it early in your application.

Alternatively if your project already uses composer's autoloader, you can configure it to find the VIA
classes with the following in your composer.json:

```json
{
  "autoload" : {
    "psr-0" : {
      "Red61" : "relative/path/to/the/src/folder/in/the/zip/file"
    }
  }
}
```

# Using the wrapper

## The object oriented approach

Version 5.2.0 introduces a new fully object-oriented interface to the API. We recommend using this
interface for new applications, as it provides more options for customisation and extension and will
also reduce your exposure to breaking API changes in future VIA versions.

You can send a request like this:

```php
// Get the configuration
$wsdl    = $config['via_wsdl'];
$webkey = $config['via_webkey'];

// Set up the relevant objects - this would normally be done in a service container or similar
//
// Responsible for creating a SoapClient just before the first request is sent
$soap_factory     = new SoapClientFactory($arr);
// Converts generic SoapFault to ViaException classes
$exception_mapper = new ViaExceptionMapper();
// Manages the list of plugins and dispatches event notifications
$plugin_manager   = new ViaCorePluginManager;
// Caches VIA responses on disk for subsequent reuse
$cache_driver     = new ViaFilecacheDriver(sys_get_temp_dir().'/via');
// Manages the caching, using the $cache_driver backen interface
$cache_plugin     = new ViaCachePlugin($cache_driver, new ViaCacheKeyGenerator);
$plugin_manager->registerPlugin($cache_plugin);
// Actually sends requests to VIA and processes the responses
$api_client       = new ViaApiClient($soap_factory, $exception_mapper, $plugin_manager, $wsdl, $webkey);
// Thin wrapper around $api_client with a method for each API method to allow typehinting of response types
$api_service      = new ViaApiService($api_client);

// Build the request object - using setters means we can protect your code from some renamed, removed or new
// request parameters in future.

$login_request  = \Red61\Via\ApiRequest\apiLoginRequest::create()
	->setEmail($_POST['email'])
	->setPassword($_POST['password']);

try {
	$cust_id = $api_service->login($login_request);

	$details = $api_service->getCustomerDetails(
	    \Red61\Via\ApiRequest\apiGetCustomerDetailsRequest::create()
	        ->setCustomerId($cust_id);
	);

	print "Hello ".$details->getFirstname().' '.$details->getSurname();
} catch (\Red61\Via\Exception\DuplicateRecordException $e) {
    print "Sorry, the email ".$_POST['email']." is linked to multiple customer accounts, so we can't identify you";

} catch (\Red61\Via\Exception\ViaException $e) {
	print_r($e->getMessage());
}

```

## The legacy Red61_Via class

The RPC-style wrapper class Red61_Via is provided for clients with legacy sites and those who prefer to use it.
Internally, it simply sends all calls on to the new ViaApiService class.

To use this version of the wrapper, just create an instance of `Red61_Via`, passing in the wsdl URL, webkey and
the customer's basket Id if known.

# Basket/session management

A user's basket is identified by a unique ID which must be passed with all basket-related API calls throughout
the booking process. Basket IDs are assigned when the user first attempts to add any item, and will be cleared
during successful order completion or if the basket expires.

**By default, the wrapper does not do anything to persist this ID between requests.** The simplest way to do
this is to pass a class implementing [\Red61\Via\SessionStorage\ViaSessionStorage](src/Red61/Via/SessionStorage/ViaSessionStorage.php).
The provided [PhpNativeSessionStorage](src/Red61/Via/SessionStorage/PhpNativeSessionStorage.php) simply sets
and gets a value from the $_SESSION superglobal array and will cover most common cases with PHP native sessions.
You can obviously implement your own to integrate with any custom application session storage. For example:

```php
// New-style
session_start();
$client = new \Red61\Via\ViaApiClient($soap_factory, $exception_mapper, $plugin_manager, $wsdl, $webkey);
$client->connectSessionStorage(new \Red61\Via\SessionStorage\PhpNativeSessionStorage);

// Old-style
session_start();
$via = new Red61_Via($wsdl, $webkey);
$via->connectSessionStorage(new \Red61\Via\SessionStorage\PhpNativeSessionStorage);
```

Alternatively, implement code elsewhere in your application to assign the basket ID at the start of a request
and save it on shutdown.

## Basket management and concurrency

Ensure that your session management handles concurrent requests appropriately, particularly if your application
uses AJAX or dynamic image thumbnailing/asset compiling which makes concurrent requests from a single user much
more likely. You should protect against the case that a basket ID is allocated or cleared during a short-running
request but then overwritten with the previous value when a concurrent slower request completes.

PHP's native file sessions handle this by taking a filesystem lock on the session for each request, queuing
requests from each session to run in sequence. This obviously has performance implications. Many
database session implementations do not provide locking and simply write the full session content to the database
at the end of every request which is very likely to cause race conditions.

For the best balance of safety and concurrency you may want to consider using a cookie (signed to prevent tampering)
or a dedicated database table which is only updated when the session ID is known to have changed.

# Caching

You should always activate caching to ensure that your site performs acceptably. The wrapper ships with cache
plugins that store VIA responses on disk (the default if using Red61_Via) and in memcache. See the drivers for the
available configuration options.

You may wish to use memcache to store data from other sources (e.g. database, other APIs, etc). If so, we recommend
using the NamespacedMemcachedAdaptor provided by this package to manage a single connection to the memcache server.
This adaptor allows you to virtually 'namespace' items in the cache, so e.g. database data can be flushed separately
from VIA data. It also protects upstream data sources by throwing exceptions if the cache is unreachable, to avoid
a situation where a failure on your cache server triggers extreme load on VIA / mysql / etc.

You can implement storage drivers for other cache backends by defining a class that implements the
\Red61\Via\Cache\ViaCacheDriver interface.

If your application operates across multiple nodes, you **MUST use a central cache backend** - for example the memcached
driver with a common memcache server/cluster. The use of per-instance caching is not supported and will produce
unexpected results.

By default, the cache layer respects HTTP cache-control headers and will disable cache lookups on a request when these
are present. This is intended to make it easy for Box Office staff to refresh cached data for individual events etc
by visiting and refreshing a relevant page. In some circumstances you may wish to disable this behaviour, which you can
do by passing an option when you create the \Red61\Via\Cache\ViaCachePlugin class.

You can also - only after discussion with Red61 - configure the default cache TTL that the ViaCachePlugin applies to
each method.

# Circuit Breaker plugin

Not enabled by default, but you may find this plugin useful. It protects your site by blocking VIA connections when
the error rate goes above configured limits. See the specific plugin documentation for details.

# Customising

If you want to customise the way the wrapper behaves in your project, you have a number of options:

* Most of the classes take options that customise some of their behaviour - check the constructors for details.

* For service classes like ViaApiClient, extend the raw service class (or implement the interface, if any) and pass an
  instance of your custom class in to any others that use it.

* To change how individual requests are sent, extend the ApiRequest class and implement methods, setters, etc as
  required. Create instances of your custom class to send - if you are using the Red61_Via RPC wrapper you will also
  need to extend that class to have the relevant methods create your custom request classes. You can use custom request
  classes if you want to process the input arguments to an API call (for example, perhaps, to always trim a certain
  text field or to set a default value for a required field that is not relevant for your application).

* To customise the individual responses, for example to add helper methods like those in ViaOrderDetails that process
  the raw VIA response to simplify usage in your application, you can extend the response class and pass a customised
  type mapping into the SoapFactory's 'classmap' option.

* To intercept multiple requests and send back canned responses, or to take some action following a call success or call failure,
  implement the ViaPluginInterface and subscribe to event notifications. You might do this to show a user a flash message
  when their basket has expired.

* To intercept individual request types before they are sent, extend the request class and implement the
  PreflightFilteringRequest interface. Your class will receive a callback just before the request is sent into the plugin
  chain and then on to VIA. You can use this to skip the call and return a canned response (see apiGetBasketSummaryRequest),
  to perform simple validation (see apiCreateAccount) or to set or update some request properties just before it is sent.
  You might also use this extension route to block particular requests. Note that calls that are skipped (or throw) at
  the preflight stage are never passed into the plugin chain so will not appear in caching, profiling and global error
  collectors.

# Disabling legacy compatibility mode

In a number of places the wrapper defaults to behaving in a way that is compatible with earlier wrapper versions - and
some of these backwards compatible tweaks will be removed in future versions. For new projects, there are a number of
options you may want to change immediately. This will ensure your code has the longest possible lifespan.

* Set `ViaApiClient::$object_compatibility_mode = ViaApiClient::OBJECT_COMPATIBILITY_OFF` - this will cause data objects
  to throw exceptions if you attempt to access values as properties instead of with their getter methods.
* Pass an array including `array('always_rethrow' => TRUE)` in the options to Red61_Via - this will ensure that the wrapper
  never swallows any exceptions thrown during requests, but always rethrows them for you to handle.


# Exceptions

(almost) all of the exceptions thrown from the wrapper will implement the \Red61\Via\Exception\ViaException interface to
allow you to catch them with a single generic catch block if required.

Many of these exceptions extend SoapFault for compatibility reasons, but are likely to change to other types in future
to better reflect the type of exception.

You should avoid catching SoapFault anywhere in your project, and instead catch either a specific exception class
or the ViaException as a fallback.

# Translations

Version 7.2.0 introduces the ability to request content from the server in a specific language.

By default, this language is taken from the `HTTP_ACCEPT_LANGUAGE` value in the `$_SERVER` superglobal.

You can also explicitly specify a language for the wrapper to use.

## The object oriented approach

In order to explicitly set a language using the object-oriented approach, simply call the `setLanguage` method of your `ViaApiClient` object with your desired language code.

```php
// Get the configuration
$wsdl    = $config['via_wsdl'];
$webkey  = $config['via_webkey'];

// Select a language that you wish to pass to the wrapper.
$language = 'fr_FR';

// Set up the relevant objects - this would normally be done in a service container or similar
//
// Responsible for creating a SoapClient just before the first request is sent
$soap_factory     = new SoapClientFactory($arr);
// Converts generic SoapFault to ViaException classes
$exception_mapper = new ViaExceptionMapper();
// Manages the list of plugins and dispatches event notifications
$plugin_manager   = new ViaCorePluginManager;
// Caches VIA responses on disk for subsequent reuse
$cache_driver     = new ViaFilecacheDriver(sys_get_temp_dir().'/via');

// Manages the caching, using the $cache_driver backen interface
$cache_plugin = new ViaCachePlugin($cache_driver, new ViaCacheKeyGenerator);
$plugin_manager->registerPlugin($cache_plugin);

// Actually sends requests to VIA and processes the responses
$api_client = new ViaApiClient($soap_factory, $exception_mapper, $plugin_manager, $wsdl, $webkey);
// Sets the language of the API client
$api_client->setLanguage($language);

// Thin wrapper around $api_client with a method for each API method to allow typehinting of response types
$api_service      = new ViaApiService($api_client);
```
## The legacy Red61_Via class

It is also possible to specify a translation language with the legacy RPC-style `Red61_Via` class. In order to do this, simply call the `setLanguage` method of your `Red61_Via` class with your desired language code.

```php
// Get the configuration
$wsdl    = $config['via_wsdl'];
$webkey  = $config['via_webkey'];


// Select a language that you wish to pass to the wrapper.
$language = 'fr_FR';

// Initialise your wrapper
$via = new Red61_Via($wsdl, $webkey);
// Sets the language of the wrapper
$via->setLanguage($language);
```

# Contributing

We welcome contributed bugfixes, features and improvements from end-users. The best way to send them is with a github
pull request. If you have installed the wrapper with composer, it's easy to get hold of a git repository checked out
with exactly the right commit for you to work from.

```bash
cd my/project/root # where your composer.json is
rm -rf vendor/red61/via-wrapper-php
composer install --prefer-source
cd vendor/red61/via-wrapper-php
git checkout -b bug/5.2/the-bug-i-found
```

The wrapper project is unit tested with [phpspec](http://phpspec.net/). Before a contribution can be merged it will need
to have passing specifications covering the changes. The ideal workflow would be:

* create a bugfix branch
* write a failing specification that proves the bug, and commit it
* fix the bug to make the specification pass, and commit that as a second commit
* send a pull request for your branch. A single pull should ideally cover just one single bug/feature

To run the specifications, you'll need to install the wrapper project's own dependencies. You can do that by:

```bash
cd my/project/root/vendor/red61/via-wrapper-php
composer install
bin/phpspec run
```

All proposed contributions (including patches by email, etc) will be considered. Please don't be put off
contributing by nervousness about sending pull requests and/or writing specs.
