<?php
/**
 * Defines ViaCircuitBreakerPluginSpec - specifications for Red61\Via\CircuitBreaker\ViaCircuitBreakerPlugin
 *
 * @author    Andrew Coulton <andrew@ingenerator.com>
 * @copyright 2014 Edinburgh International Book Festival Ltd
 * @licence   http://opensource.org/licenses/BSD-3-Clause
 */

namespace spec\Red61\Via\CircuitBreaker;

use PhpSpec\Exception\Example\FailureException;
use Prophecy\Argument;
use Red61\Via\ApiRequest\apiGetVersionRequest;
use Red61\Via\CircuitBreaker\ViaCircuitBreaker;
use Red61\Via\Exception\TooManyCartsException;
use Red61\Via\Exception\ViaCircuitBreakerTrippedException;
use Red61\Via\Exception\ViaException;
use Red61\Via\Exception\ViaHTTPException;
use Red61\Via\Exception\ViaInternalServerErrorException;
use Red61\Via\Exception\ViaSoapFault;
use Red61\Via\Plugin\ViaApiCallNotification;
use spec\ObjectBehavior;

/**
 *
 * @see Red61\Via\CircuitBreaker\ViaCircuitBreakerPlugin
 */
class ViaCircuitBreakerPluginSpec extends ObjectBehavior {

	/**
	 * Use $this->subject to get proper type hinting for the subject class
	 *
	 * @var \Red61\Via\CircuitBreaker\ViaCircuitBreakerPlugin
	 */
	protected $subject;

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreaker $breaker
	 */
	function let($breaker)
	{
		$this->beConstructedWith($breaker);
		$breaker->requestPermissionToCall()->willReturn(TRUE);
	}

	function it_is_initializable()
	{
		$this->subject->shouldHaveType('Red61\Via\CircuitBreaker\ViaCircuitBreakerPlugin');
	}

	function it_is_a_via_plugin()
	{
		$this->subject->shouldHaveType('Red61\Via\Plugin\ViaPluginInterface');
	}

	/**
	 * @param \Red61\Via\Plugin\ViaPluginManager $plugin_manager
	 */
	function it_can_register_with_a_plugin_manager_for_all_notifications($plugin_manager)
	{
		$this->subject->registerWithManager($plugin_manager);

		$plugin_manager->registerPlugin(
			$this->subject,
			NULL
		);
	}

	/**
	 * @param \Red61\Via\Plugin\ViaPluginNotification $notification
	 */
	function it_ignores_notifications_that_are_not_call_notifications($notification)
	{
		$this->subject->onViaPluginNotification($notification);
	}

	function it_allows_api_calls_by_default()
	{
		$this->subject->onViaPluginNotification(new DummyBeforeCallNotification);
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreaker $breaker
	 *
	 * @throws FailureException
	 */
	function it_blocks_api_calls_by_throwing_if_breaker_refuses_permission($breaker)
	{
		$breaker->requestPermissionToCall()->willReturn(FALSE);
		try
		{
			$this->subject->onViaPluginNotification(new DummyBeforeCallNotification);
			throw new FailureException('Expected exception, none got');
		}
		catch (ViaCircuitBreakerTrippedException $e)
		{
			// Expected
		}
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreaker $breaker
	 */
	function it_notifies_breaker_of_critical_failure_on_http_error($breaker)
	{
		$this->shouldNotifyFailureOn(
			new ViaHTTPException('HTTP', 'Not Found'),
			$breaker,
			ViaCircuitBreaker::SEVERITY_CRITICAL
		);
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreaker $breaker
	 */
	function  it_notifies_circuit_breaker_of_critical_failure_on_internal_server_error($breaker)
	{
		$this->shouldNotifyFailureOn(
			new ViaInternalServerErrorException('soap:Server', 'java.lang.NullPointerException'),
			$breaker,
			ViaCircuitBreaker::SEVERITY_CRITICAL
		);
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreaker $breaker
	 */
	function it_notifies_circuit_breaker_of_warning_failure_on_generic_via_error($breaker)
	{
		$this->shouldNotifyFailureOn(
			new ViaSoapFault('soap:Server', 'Customer cannot renew Scheme [foo] too soon to renew'),
			$breaker,
			ViaCircuitBreaker::SEVERITY_WARNING
		);
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreaker $breaker
	 */
	function it_never_notifies_circuit_breaker_of_failure_from_too_many_carts_exception($breaker)
	{
		$this->subject->onViaPluginNotification(
			new DummyCallFailedNotification(
				new TooManyCartsException('soap:Server', 'The system is currently very busy, you may browse but not buy currently.  Please try later')
			)
		);
		$breaker->notifyFailure(Argument::cetera())->shouldNotHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreaker $breaker
	 */
	function it_does_not_reset_circuit_breaker_on_successful_request_from_cache($breaker)
	{
		$this->subject->onViaPluginNotification(new DummyCallCachedSuccessNotification);
		$breaker->reset()->shouldNotHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreaker $breaker
	 */
	function it_resets_circuit_breaker_on_successful_API_request($breaker)
	{
		$breaker->reset()->shouldBeCalled();
		$this->subject->onViaPluginNotification(new DummyCallSuccessNotification);
	}

	/**
	 * @param \Exception        $exception
	 * @param ViaCircuitBreaker $breaker
	 * @param string            $expect_severity
	 */
	protected function shouldNotifyFailureOn(\Exception $exception, ViaCircuitBreaker $breaker, $expect_severity)
	{
		$breaker->notifyFailure($expect_severity, $exception)->willReturn(NULL);
		$this->subject->onViaPluginNotification(new DummyCallFailedNotification($exception));
		$breaker->notifyFailure($expect_severity, $exception)->shouldHaveBeenCalled();
	}

}

class DummyBeforeCallNotification extends ViaApiCallNotification
{
	public function __construct()
	{
		parent::__construct(new apiGetVersionRequest);
	}
}

class DummyCallFailedNotification extends DummyBeforeCallNotification
{
	public function __construct(ViaException $failure)
	{
		parent::__construct();
		$this->_setFailureException($failure);
	}
}

class DummyCallCachedSuccessNotification extends DummyBeforeCallNotification
{
	public function __construct()
	{
		parent::__construct();
		$this->skipCallAndReturn('5.2.2');
	}
}

class DummyCallSuccessNotification extends DummyBeforeCallNotification
{
	public function __construct()
	{
		parent::__construct();
		$this->_setSuccessfulResponse('5.2.2');
	}
}
