<?php
/**
 * Defines ViaApiServiceSpec - specifications for Red61\Via\ViaApiService
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */

namespace spec\Red61\Via;

use PhpSpec\Exception\Example\FailureException;
use Prophecy\Prophet;
use Red61\Via\ApiRequest\ViaApiRequest;
use Red61\Via\ViaApiClient;
use Red61\Via\ViaApiService;
use spec\ObjectBehavior;
use Prophecy\Argument;

/** @noinspection PhpMultipleClassesDeclarationsInOneFile */
/**
 *
 * @see Red61\Via\ViaApiService
 */
class ViaApiServiceSpec extends ObjectBehavior
{

	/**
     * Use $this->subject to get proper type hinting for the subject class
     * @var \Red61\Via\ViaApiService
     */
	protected $subject;

	/**
	 * @param \Red61\Via\ViaApiClient $client
	 */
	function let($client)
	{
		$this->beConstructedWith($client);
	}

	function it_is_initializable()
    {
		$this->subject->shouldHaveType('Red61\Via\ViaApiService');
	}

	function its_methods_send_requests_and_return_client_response()
	{
		$blacklisted_methods = array('__construct');

		foreach ($this->find_api_service_methods() as $method) {
			if (in_array($method->getName(), $blacklisted_methods))
				continue;

			try {
				$this->verify_request_and_response_behaviour($method);
			} catch (FailureException $e) {
				throw new FailureException($method->getName().": ".$e->getMessage(), 0, $e);
			}
		}
	}

	/**
	 * @return \ReflectionMethod[]
	 */
	protected function find_api_service_methods()
	{
		$reflection = new \ReflectionClass('\Red61\Via\ViaApiService');
		$methods    = $reflection->getMethods(\ReflectionMethod::IS_PUBLIC);
		return $methods;
	}

	/**
	 * @param \ReflectionMethod $method
	 *
	 * @throws FailureException
	 */
	protected function verify_request_and_response_behaviour(\ReflectionMethod $method)
	{
		$method_name = $method->getName();
		$client      = new SpyingApiClientStub;
		$service     = new ViaApiService($client);
		$params      = array();

		if ( ! $this->is_parameterless_call($method))
		{
			$params[] = $this->fake_request_for_method($method);
		}

		$result = call_user_func_array(array($service, $method_name), $params);
		$client->assertOneRequestSent(array_shift($params));
		expect($result)->toBe($this->is_void_method($method) ? NULL : $client->getFakedResult());
	}

	/**
	 * @param \ReflectionMethod $method
	 *
	 * @return bool
	 */
	protected function is_parameterless_call(\ReflectionMethod $method)
	{
		return ($method->getNumberOfParameters() === 0);
	}

	/**
	 * @param \ReflectionMethod $method
	 *
	 * @return ViaApiRequest $request
	 */
	protected function fake_request_for_method(\ReflectionMethod $method)
	{
		if ($method->getNumberOfParameters() !== 1)
			throw new \UnexpectedValueException($method->getName().' should have 0 or 1 parameters, not '.$method->getNumberOfParameters());

		$params = $method->getParameters();
		$param  = array_shift($params);
		/** @var \ReflectionParameter $param */

		if ($param->getName() !== 'request')
			throw new \UnexpectedValueException('Did not expect parameter '.$param->getName().' of ViaApiService::'.$method->getName());

		$request_class = $param->getClass()->getName();
		return new $request_class;
	}

	/**
	 * @param \ReflectionMethod $method
	 *
	 * @return bool
	 */
	protected function is_void_method(\ReflectionMethod $method)
	{
		return (bool) preg_match('/\* @return void/', $method->getDocComment());
	}

}

/** @noinspection PhpMultipleClassesDeclarationsInOneFile */
/**
 * Captures the requests sent through the ViaApiClient by the ViaApiService class and allows assertion that
 * expected request types were sent.
 *
 * @package spec\Red61\Via
 */
class SpyingApiClientStub extends ViaApiClient {

	/**
	 * @var ViaApiRequest[]
	 */
	protected $requests = array();

	/**
	 * @var string
	 */
	protected $fake_result = 'fake-client-result';

	/**
	 * Empty constructor to override dependencies
	 */
	public function __construct() {}

	/**
	 * @param ViaApiRequest $request
	 *
	 * @return mixed
	 */
	public function send(ViaApiRequest $request)
	{
		$this->requests[] = $request;
		return $this->fake_result;
	}

	/**
	 * @param ViaApiRequest $request
	 *
	 * @throws FailureException
	 */
	public function assertOneRequestSent(ViaApiRequest $request = NULL)
	{
		if (empty($this->requests))
			throw new FailureException("No VIA API request was sent");

		$sent_request = $this->requests[0];
		if ($request AND $request !== $sent_request)
			throw new FailureException(sprintf(
					'Client expected the provided %s request, but got a different instance of %s',
					$request->getSoapMethodName(),
					get_class($sent_request)
				)
			);
	}

	/**
	 * @return string
	 */
	public function getFakedResult()
	{
		return $this->fake_result;
	}

}
