<?php

namespace Red61\Via\Cache;

use Memcached;


/**
 * Generic adaptor to the php-memcached extension that supports namespacing results and fast-failure on error
 *
 * This adaptor provides a low-level wrapper around the native php-memcached extension to achieve three things:
 *
 *   * Simple de-duped configuration of a single server's host and port when using persistent connections
 *   * Namespacing results within the cache - e.g. to allow flushing VIA cache without affecting database or other
 *     application-level cached data
 *   * Protecting upstream services (VIA, database, etc) from excessive load if the cache fails by throwing an exception
 *     on any failed cache operations rather than silently returning an empty value / ignoring failure to store.
 *
 * It is used by the ViaMemcachedDriver, but can also be used directly within your application - usually as a shared
 * service provided by your service container or otherwise shared in global state.
 *
 * The connection logic within this driver only supports connecting to a single memcached server.
 *
 * Alternatively, you can configure the `Memcached` instance before you pass it into the adaptor's constructor. You may
 * also then want to pass an option of `'server' => NULL` to bypass the connection code here. This will let you achieve
 * more complex setups including a sharded cache with multiple servers.
 *
 * For example:
 *
 *    $mc = new \Memcached;
 *    $mc->addServerList($all_my_servers_and_weights_and_the_like);
 *    $mc->setOption(Memcached::SOME_OPTION, 'some_value');
 *    $driver = new NamespacedMemcachedAdaptor($mc, ['server' => NULL]);
 *
 * If you ask this adaptor to manage your server connection, but there are any other servers in the pool, we will throw
 * an exception. If you want to use multiple servers your code must take full responsibility for managing that.
 *
 * ### Persistent connections
 *
 * To use a persistent connection with the new `memcached` extension, you need to pass a 'persistent_id' to the
 * constructor of the `Memcached` class. This can only be set when the class instance is created. All instances,
 * across all requests, with the same persistent_id will share the same global connection object in the underlying PHP
 * native code.
 *
 * This also means the list of servers is cached across all instances with the same persistent_id. Importantly, the
 * `addServer` call is not de-duplicated - it is your responsibility to call getServerList and only call `addServer`
 * if the server is not already present.
 *
 * If you pass connection details to this adaptor, we will automatically de-dupe for you. If you are calling addServer
 * directly you are responsible for ensuring you only add servers that aren't already in the pool.
 *
 * The other gotcha with persistent connections comes if you need to change the memcached host or port. If you just
 * call `addServer` directly, the new server will be added to the pool but the old one will still be there. You should
 * either use a persistent_id that changes based on the server connections you require, or ensure that PHP is restarted
 * (e.g. with an `apache reload`) on every deployment.
 *
 * Using a persistent connection with the driver is pretty straightforward:
 *
 *    $mc = new \Memcached('my-persistent-id');
 *
 * ### Lazy connection
 *
 * Unlike the old `Memcache` extension and driver, `Memcached` does not attempt to connect until the first actual call
 * to the memcache server. Calling addServer etc just updates the internal list of server(s) that will eventually be
 * contacted. It does not do any validation or e.g. DNS lookup at that stage.
 *
 * We therefore don't know whether the server is actually available / connection details are correct until you attempt
 * a request - the constructor will always succeed regardless of the configuration you pass through.
 *
 * ### Error handling and monitoring
 *
 * As well as improving application performance, the VIA cache is intended to protect VIA itself from excessive load.
 *
 * Therefore if a cache lookup encounters any errors we will throw a MemcachedOperationFailedException, rather than
 * following the default php-memcached behaviour of failing silently. When used with the VIA driver, this will cause the
 * VIA request to fail before it is sent.
 *
 * You may wish to monitor or log cache failures centrally. The `onCacheFailure` method in this class provides a hook
 * point for you to add this. Extend this class and implement that method as required.
 *
 * ### Client options
 *
 * For convenience, you can pass an array of `'client_options' => [\Memcached::COMPRESSION => TRUE]' in the constructor
 * options for this class. They will be set on your Memcached client instance during instantiation. Bear in mind that
 * if you are using persistent connections, client options also persist to all clients with the same persistent_id
 * across all requests.
 *
 * ### Cross compatibility
 *
 * Note that because of differences in the underlying data storage - particularly the serializer - values stored by
 * the memcached class are *not* cross-compatible with values stored by the memcache class. You should pick one of the
 * extensions and use that consistently across your application (hint: pick this extension, it is still supported).
 *
 */
class NamespacedMemcachedAdaptor
{
    const MEMCACHED_DEFAULT_PORT = 11211;

    /**
     * @var Memcached The memcache instance
     */
    protected $memcached = NULL;

    /**
     * @var string[] Local cache of the known namespace prefixes used to support per-namespace flush
     */
    protected $namespace_prefixes = array();

    /**
     * Creates an instance
     *
     * For the simple case, to have the driver connect to a single server without persistent connections:
     *
     *    $d = new ViaMemcachedDriver(new Memcached, ['server' => ['host' => '127.0.0.1', 'port' => 11211]]);
     *    // port defaults to 11211 if omitted
     *
     * For more complex setups:
     *
     *    $mc = new \Memcached
     *    // do whatever you need to $mc including adding servers
     *
     *    new ViaMemcachedDriver($mc);
     *    // or if you want to pass an options array
     *    new ViaMemcachedDriver($mc, ['server' => NULL, 'client_options' => [\Memcached::OPT_COMPRESSION => FALSE]]);
     *
     * See class documentation for full details
     *
     * @param \Memcached $memcache
     * @param array      $options
     *
     * @throws CacheConfigurationException if configuration provided is not valid
     */
    public function __construct(\Memcached $memcached, array $options = array())
    {
        $this->memcached = $memcached;
        $options         = \array_merge(array('server' => NULL, 'client_options' => array()), $options);

        if ($options['server'] !== NULL) {
            $this->validateAndAddServer($options['server']);
        }

        foreach ($options['client_options'] as $opt_key => $opt_value) {
            if ( ! $memcached->setOption($opt_key, $opt_value)) {
                throw CacheConfigurationException::setOptionFailed($opt_key, $opt_value);
            }
        }
    }


    /**
     * @param $server
     */
    protected function validateAndAddServer($server)
    {
        $server = array_merge(
            ['host' => NULL, 'port' => static::MEMCACHED_DEFAULT_PORT],
            $server
        );

        if (empty($server['host'])) {
            throw CacheConfigurationException::missingHost();
        }

        // If the user has passed in a memcached instance with a persistent_id, the server list is maintained for the
        // life of the underlying persistent connection in native PHP. addServer is not internally de-duplicated, so
        // we need to do that here.
        //
        // We also check there are no other servers in the pool, as this would imply the user is doing more advanced
        // configuration of their connections and should not be passing any server details to this driver. See docs
        //
        // However we will tolerate multiple existing connections to the server we've been asked to use - e.g. if the
        // user is separately calling addServer somewhere else with the same connection details. Although non-optimal,
        // this is not fundamentally broken so we should grudgingly tolerate it.

        foreach ($this->memcached->getServerList() as $existing_connection) {
            if ( ! $this->isMatchingServer($existing_connection, $server['host'], $server['port'])) {
                throw CacheConfigurationException::unexpectedServers();
            }
        }

        if ( ! $this->addServerUnlessDuplicate($server['host'], $server['port'])) {
            throw CacheConfigurationException::addServerFailed($server['host'], $server['port']);
        }

    }

    /**
     * Adds a server to a pool of memcached servers if it is not already present and returns the result
     *
     * @param \Memcached $memcached
     * @param string     $host
     * @param int        $port
     *
     * @return bool
     */
    protected function addServerUnlessDuplicate($host, $port = self::MEMCACHED_DEFAULT_PORT)
    {
        foreach ($this->memcached->getServerList() as $existing_server) {
            if ($this->isMatchingServer($existing_server, $host, $port)) {
                return TRUE;
            }
        }

        return $this->memcached->addServer($host, $port);
    }

    /**
     * Simple conditional helper to check if a server array from getServerList matches the given host and port
     *
     * @param array      $existing_server
     * @param string     $host
     * @param string|int $port
     *
     * @return bool
     */
    protected function isMatchingServer(array $existing_server, $host, $port)
    {
        return (
            ($existing_server['host'] === $host)
            &&
            ((string) $existing_server['port'] === (string) $port)
        );
    }

    /**
     * Wrapper method that provides the result checking, hook and error throwing behaviour for any memcached call
     *
     * This method should be called internally for any memcached request. It checks the result of the operation - if
     * false then it checks the underlying memcached result code. Some requests are expected to fail with certain codes
     * - for example, a get can fail with NOT_FOUND. Any expected codes apart from SUCCESS should be passed in the
     * $success_codes parameter.
     *
     * If the operation returned false and the result code was not expected, then a MemcachedOperationFailedException
     * will be created, passed to the onCacheFailure hook method, then thrown.
     *
     * For examples of the callable structure, see the get/set/etc methods in this class
     *
     * @param callable $callable      The actual code block to run
     * @param int[]    $success_codes Expected memcached result codes for this operation
     *
     * @return bool
     * @throws MemcachedOperationFailedException
     */
    protected function ensureCacheSuccess($callable, array $success_codes = array())
    {
        $start  = \microtime(TRUE);
        $result = $callable();
        $end    = \microtime(TRUE);
        if ($result !== FALSE) {
            return $result;
        }


        $result_code = $this->memcached->getResultCode();
        if (($result_code === \Memcached::RES_SUCCESS) || \in_array($result_code, $success_codes)) {
            // Fine, this was expected - e.g. it was a `get` for a value that is actually cached as FALSE, or that is
            // not in cache.
            return FALSE;
        }

        // This was an unexpected result - e.g. the server is down, key is not valid, etc.
        $exception = new MemcachedOperationFailedException(
            $this->memcached->getResultCode(),
            $this->memcached->getResultMessage(),
            $this->memcached->getServerList()
        );
        $this->onCacheFailure($result_code, $exception, 1000 * ($end - $start));
        throw $exception;
    }

    /**
     * Extension hook point for capturing details when memcached operations fail
     *
     * Implement this in a child class if required - it is called just before the exception is thrown
     *
     * @param int                               $result_code       The actual memcached result code returned
     * @param MemcachedOperationFailedException $e                 The exception about to be thrown
     * @param float                             $operation_time_ms The time the operation took (useful for performance metrics)
     */
    protected function onCacheFailure($result_code, MemcachedOperationFailedException $e, $operation_time_ms)
    {
        // Extension point for logging, metrics collection etc
    }

    /**
     * Get a value from memcached
     *
     * @param string $namespace The namespace of the item to retrieve
     * @param string $key       The key of the item to retrieve
     *
     * @return mixed The cached value, or FALSE if there is no value cached.
     * @throws MemcachedOperationFailedException If the operation fails for any reason
     */
    public function get($namespace, $key)
    {
        $ns_key = $this->getNamespacedKey($namespace, $key);

        return $this->ensureCacheSuccess(
            function () use ($ns_key) { return $this->memcached->get($ns_key); },
            [Memcached::RES_NOTFOUND]
        );
    }


    /**
     * Gets the full namespaced key for a given item
     *
     * This should not generally be used at application level. One exception might be if you wish to store certain
     * data locally but still be able to have it flushed when the shared memcached namespace is flushed.
     *
     * For example, caching larger objects in apcu on your instance may be faster than sending to and from memcached.
     * If you use this method to calculate the key to use in apcu, those objects will be invalidated when you flush
     * the memcached namespace.
     *
     * You will not normally need to call this method if you are only caching items through this memcached adaptor.
     *
     * @param string $namespace The namespace of the item to set/get
     * @param string $key       The key of the item to set/get
     *
     * @return string The current value of the composite key for this item
     * @throws MemcachedOperationFailedException If the operation fails for any reason
     */
    public function getNamespacedKey($namespace, $key)
    {
        if ( ! isset($this->namespace_prefixes[$namespace])) {
            $this->loadNamespacePrefix($namespace);
        }
        $prefix = $this->namespace_prefixes[$namespace];

        return "$namespace:$prefix:$key";
    }

    /**
     * Load or create the current prefix for the given key namespace
     *
     * Items are namespaced using the method outlined at
     * https://github.com/memcached/memcached/wiki/ProgrammingTricks#simulating-namespaces-with-key-prefixes
     *
     * Each namespace has a dynamic prefix value which is used to calculate keys for all items within that namespace.
     * By resetting / changing the prefix for the namespace, all the calculated keys will change and therefore any
     * previously stored data will no longer return a cache hit. Although older values are not flushed as such,
     * memcached will gradually shift them out of memory as required as part of its usual garbage collection and
     * least-recently-used cleanup.
     *
     * @param string $namespace The cache namespace to get the prefix for
     *
     * @return string The prefix
     * @throws MemcachedOperationFailedException If the operation fails for any reason
     */
    protected function loadNamespacePrefix($namespace)
    {
        $prefixkey = "namespace:$namespace";

        // Simple case - we have a 'live' prefix value at the moment
        $prefix = $this->ensureCacheSuccess(
            function () use ($prefixkey) { return $this->memcached->get($prefixkey); },
            [\Memcached::RES_NOTFOUND]
        );

        if ($prefix === FALSE) {
            // There is no prefix for this namespace so we need to create one.
            // To protect against race conditions we attempt to add the value (which will fail if it already exists)
            // then read the value again to be sure we are using the one set by the 'winner'.
            $this->ensureCacheSuccess(
                function () use ($prefixkey) { return $this->memcached->add($prefixkey, uniqid()); },
                [\Memcached::RES_NOTSTORED]
            );
            $prefix = $this->ensureCacheSuccess(
                function () use ($prefixkey) { return $this->memcached->get($prefixkey); }
            );
        }

        $this->namespace_prefixes[$namespace] = $prefix;

        return $prefix;
    }

    /**
     * Stores a value
     *
     * @param string $namespace The namespace in which to store the item
     * @param string $key       The key under which to store the item
     * @param mixed  $var       The value to store - can be anything that is serializable
     * @param int    $expire    The TTL in seconds for the cache value - or 0 to keep indefinitely.
     *
     * @return bool Whether the set was successful - in practice it is not possible for this to return FALSE
     * @throws MemcachedOperationFailedException If the operation fails for any reason
     */
    public function set($namespace, $key, $var, $expire = 0)
    {
        $ns_key = $this->getNamespacedKey($namespace, $key);

        return $this->ensureCacheSuccess(
            function () use ($ns_key, $var, $expire) { return $this->memcached->set($ns_key, $var, $expire); }
        );
    }

    /**
     * FLushes the entire memcached server - use with caution
     *
     * @return void
     * @throws MemcachedOperationFailedException If the operation fails for any reason
     */
    public function flushEverything()
    {
        $this->ensureCacheSuccess(function () { return $this->memcached->flush(); });
    }

    /**
     * Flushes a single namespace (by rotating the item key prefix)
     *
     * @param string $namespace The namespace name
     *
     * @return void
     * @throws MemcachedOperationFailedException If the operation fails for any reason
     */
    public function flushNamespace($namespace)
    {
        $this->ensureCacheSuccess(
            function () use ($namespace) { return $this->memcached->set('namespace:'.$namespace, uniqid()); }
        );
        // Clear the local cached prefix for the next call
        unset($this->namespace_prefixes[$namespace]);
    }

    /**
     * Delete an item from the cache
     *
     * @param string $namespace The namespace of the item to delete
     * @param string $key       The key of the item to delete
     *
     * @return void
     * @throws MemcachedOperationFailedException If the operation fails for any reason
     */
    public function delete($namespace, $key)
    {
        $ns_key = $this->getNamespacedKey($namespace, $key);
        $this->ensureCacheSuccess(
            function () use ($ns_key) { return $this->memcached->delete($ns_key); },
            [\Memcached::RES_NOTFOUND]
        );
    }
}