<?php
/**
 * Defines Red61\Via\Cache\ViaMemcacheDriver
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */



namespace Red61\Via\Cache;
use Red61\Via\Exception\CacheNotAvailableException;

/**
 * Caches VIA responses in a memcache server
 *
 * This is an initial implementation based on the legacy cache. It does not currently support namespacing either within
 * the memcache server as a whole or for groups of VIA methods - these will be added in a future release.
 *
 * @package Red61\Via\Cache
 * @see     spec\Red61\Via\Cache\ViaMemcacheDriverSpec
 */
class ViaMemcacheDriver implements ViaCacheDriver {

	/**
	 * @var \Memcache
	 */
	protected $memcache;

	/**
	 * Create an instance
	 *
	 * Options:
	 * * already_connected - by default, the driver will attempt to open a connection - if you are passing in a
	 *                        shared memcache instance set this option to avoid a second connect attempt.
	 * * host              - the server to connect to - defaults to 127.0.0.1
	 * * port              - the port to connect to - default to 11211
	 * * persistent        - whether to open a persistent connection - defaults off.
	 *
	 * @param \Memcache $memcache
	 * @param array     $options
	 *
	 * @throws \Red61\Via\Exception\CacheNotAvailableException if a connection cannot be opened
	 */
	public function __construct(\Memcache $memcache, $options)
	{
		$this->memcache = $memcache;
		$defaults = array(
			'already_connected' => FALSE,
			'host'              => '127.0.0.1',
			'port'              => 11211,
			'persistent'        => FALSE
		);
		$options = array_merge($defaults, $options);
		$this->connectOrThrow($options);
	}

	/**
	 * {@inheritdoc}
	 */
	public function lookup($namespace, $key, $ttl)
	{
		return $this->memcache->get($key);
	}

	/**
	 * {@inheritdoc}
	 */
	public function store($namespace, $key, $value, $ttl)
	{
		$this->memcache->set($key, $value, 0, $ttl);
	}

	/**
	 * [!!] This will flush all data on the memcache server. A future API release will implement flushing of
	 *      just the VIA data.
	 *
	 * {@inheritdoc}
	 */
	public function flush()
	{
		$this->memcache->flush();
	}

	/**
	 * @param array $options
	 *
	 * @throws \Red61\Via\Exception\CacheNotAvailableException
	 */
	protected function connectOrThrow($options)
	{
		if (!$options['already_connected']) {
			if ($options['persistent']) {
				$connected = $this->memcache->pconnect($options['host'], $options['port']);
			} else {
				$connected = $this->memcache->connect($options['host'], $options['port']);
			}
			if (!$connected) {
				throw new CacheNotAvailableException('Could not open memcache connection to ' . $options['host'] . ':' . $options['port']);
			}
		}
	}

}
