<?php
/**
 * Defines ViaFilecacheDriverSpec - specifications for Red61\Via\Cache\ViaFilecacheDriver
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */

namespace spec\Red61\Via\Cache;

use PhpSpec\Exception\Example\FailureException;
use Red61\Via\ApiRequest\apiSearchAddressRequest;
use Red61\Via\DataObject\ViaApiAddressDetails;
use Red61\Via\Exception\CacheNotAvailableException;
use spec\ObjectBehavior;
use Prophecy\Argument;

/**
 *
 * @see Red61\Via\Cache\ViaFilecacheDriver
 */
class ViaFilecacheDriverSpec extends ObjectBehavior
{
    /**
     * Use $this->subject to get proper type hinting for the subject class
     * @var \Red61\Via\Cache\ViaFilecacheDriver
     */
	protected $subject;

	protected $working_dir;

	protected $old_error_reporting;

	function let()
	{
		$this->working_dir = sys_get_temp_dir().'/via/'.uniqid();
		$this->beConstructedWith($this->working_dir);
		$this->old_error_reporting = error_reporting();
	}

	function letgo()
	{
		`rm -rf $this->working_dir`;
		error_reporting($this->old_error_reporting);
	}

	function it_is_initializable()
    {
		$this->subject->shouldHaveType('Red61\Via\Cache\ViaFilecacheDriver');
	}

	function it_is_a_via_cache_driver()
	{
		$this->subject->shouldHaveType('Red61\Via\Cache\ViaCacheDriver');
	}

	function it_creates_cache_dir_if_not_existing()
	{
		$this->subject->getWrappedObject();

		expect(is_dir($this->working_dir))->toBe(TRUE);
	}

	function it_throws_if_working_dir_cannot_be_created()
	{
		expect(is_writable('/bad-dir'))->toBe(FALSE);
		error_reporting(0);
		$this->beConstructedWith('/bad-dir');

		try {
			$this->subject->getWrappedObject();
			throw new FailureException('Expected exception not thrown');
		} catch (CacheNotAvailableException $e)
		{
			expect($e)->toBeAnInstanceOf('\Red61\Via\Exception\ViaException');
		}
	}

	function it_creates_file_for_store()
	{
		$this->shouldHaveFilesInWorkingDir(0);
		$this->subject->store('apiGetEvents', 'key', 'value', 50);
		$this->shouldHaveFilesInWorkingDir(1);
	}

	function it_can_store_and_retrieve_string()
	{
		$this->shouldHaveFilesInWorkingDir(0);
		$this->subject->store('apiGetEvents', 'key', 'value', 50);
		$this->subject->lookup('apiGetEvents', 'key', 50)->shouldBe('value');
	}

	function it_can_store_and_retrieve_object()
	{
		$addresses = array(new ViaApiAddressDetails('8 Craigleith Hill Row', 'Edinburgh', '', '', '', 'EH4 2JX'));
		$this->subject->store('apiSearchAddress', 'key', $addresses, 50);
		$this->subject->lookup('apiSearchAddress', 'key', 50)->shouldBeLike($addresses);
	}

	function it_returns_false_if_lookup_not_found_by_key()
	{
		$this->subject->store('apiSearchAddress', 'otherkey', 'value', 50);

		$this->subject->lookup('apiSearchAddress', 'key', 50)->shouldBe(FALSE);
	}

	function it_returns_false_from_lookup_and_deletes_file_if_ttl_expired_based_on_mtime()
	{
		$addresses = array(new ViaApiAddressDetails('8 Craigleith Hill Row', 'Edinburgh', '', '', '', 'EH4 2JX'));
		$this->subject->store('apiSearchAddress', 'key', $addresses, 1);

		// Something of a hack....
		$this->shouldHaveFilesInWorkingDir(1);
		$files = $this->getWorkingDirFiles();
		touch(array_pop($files), time() - 2);

		$this->subject->lookup('apiSearchAddress', 'key', 1)->shouldBe(FALSE);
		$this->shouldHaveFilesInWorkingDir(0);
	}

	function it_can_flush_the_cache()
	{
		$this->subject->store('apiSomething', 'key', 'value', 90);
		$this->shouldHaveFilesInWorkingDir(1);
		$this->subject->flush();
		$this->shouldHaveFilesInWorkingDir(0);
	}

	protected function shouldHaveFilesInWorkingDir($expect_count)
	{
		$files = $this->getWorkingDirFiles();
		$actual_count = count($files);
		if ($expect_count !== $actual_count) {
			throw new FailureException(
				"Got $actual_count instead of $expect_count files in $this->working_dir. Got:".\PHP_EOL.\PHP_EOL
				.' * '.implode(\PHP_EOL.' * ', $files)
			);
		}
	}

	protected function getWorkingDirFiles()
	{
		return glob($this->working_dir.'/**/*');
	}



}
