<?php
/**
 * Defines ViaCachePluginSpec - specifications for Red61\Via\Cache\ViaCachePlugin
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */

namespace spec\Red61\Via\Cache;

use Red61\Via\ApiRequest\apiGetEventsRequest;
use Red61\Via\ApiRequest\apiGetVersionRequest;
use Red61\Via\Cache\ViaCacheTTLConfig;
use Red61\Via\DataObject\ViaApiEvent;
use Red61\Via\Plugin\ViaApiCallNotification;
use Red61\Via\ApiRequest\apiAddTicketsToBasketRequest;
use spec\ObjectBehavior;
use Prophecy\Argument;

/**
 *
 * @see Red61\Via\Cache\ViaCachePlugin
 */
class ViaCachePluginSpec extends ObjectBehavior
{
    /**
     * Use $this->subject to get proper type hinting for the subject class
     * @var \Red61\Via\Cache\ViaCachePlugin
     */
	protected $subject;

	const FAKE_CACHE_KEY = 'aASDasds4232231231245';

	/**
	 * @param \Red61\Via\Cache\ViaCacheDriver       $driver
	 * @param \Red61\Via\Cache\ViaCacheKeyGenerator $key_generator
	 */
	function let($driver, $key_generator)
	{
		$this->beConstructedWith($driver, $key_generator);

		$key_generator->get_key(Argument::any())->willReturn(self::FAKE_CACHE_KEY);
	}

	function it_is_initializable()
    {
		$this->subject->shouldHaveType('Red61\Via\Cache\ViaCachePlugin');
	}

	function it_is_a_via_plugin()
	{
		$this->subject->shouldHaveType('Red61\Via\Plugin\ViaPluginInterface');
	}

	/**
	 * @param \Red61\Via\Cache\ViaCacheDriver $driver
	 */
	function it_performs_cache_lookup_on_before_call_event($driver)
	{
		$notification = new ViaApiCallNotification(new apiGetEventsRequest);
		$this->subject->onViaPluginNotification($notification);

		$driver->lookup(apiGetEventsRequest::SOAP_METHOD_NAME, self::FAKE_CACHE_KEY, Argument::type('int'))->shouldHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\Cache\ViaCacheDriver $driver
	 */
	function it_does_not_set_via_to_skip_call_if_cache_returns_null($driver)
	{
		$driver->lookup(Argument::cetera())->willReturn(NULL);

		$notification = new ViaApiCallNotification(new apiGetEventsRequest);
		$this->subject->onViaPluginNotification($notification);

		expect($notification->shouldSkipCall())->toBe(FALSE);
	}

	/**
	 * @param \Red61\Via\Cache\ViaCacheDriver $driver
	 */
	function it_sets_via_to_skip_call_if_cache_returns_value($driver)
	{
		$events = array(new ViaApiEvent);
		$driver->lookup(Argument::cetera())->willReturn($events);

		$notification = new ViaApiCallNotification(new apiGetEventsRequest);
		$this->subject->onViaPluginNotification($notification);

		expect($notification->shouldSkipCall())->toBe(TRUE);
		expect($notification->getResponse())->toBe($events);
	}

	/**
	 * @param \Red61\Via\Cache\ViaCacheDriver $driver
	 */
	function it_performs_cache_store_on_call_success_event($driver)
	{
		$events = array(new ViaApiEvent);

		$notification = new ViaApiCallNotification(new apiGetEventsRequest());
		$notification->_setSuccessfulResponse($events);

		$this->subject->onViaPluginNotification($notification);

		$driver->store(apiGetEventsRequest::SOAP_METHOD_NAME, self::FAKE_CACHE_KEY, $events, Argument::type('int'))->shouldHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\Cache\ViaCacheDriver $driver
	 */
	function it_does_not_perform_cache_store_if_response_was_from_skipped_call($driver)
	{
		$notification = new ViaApiCallNotification(new apiGetEventsRequest);
		$notification->skipCallAndReturn('response');
		$notification->_setSuccessfulResponse('response');
		$this->subject->onViaPluginNotification($notification);

		$driver->store(Argument::cetera())->shouldNotHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\Cache\ViaCacheDriver $driver
	 */
	function it_performs_lookup_with_configured_ttl($driver)
	{
		$notification = new ViaApiCallNotification(new apiGetEventsRequest);
		$this->subject->onViaPluginNotification($notification);

		$driver->lookup(
			Argument::any(),
			Argument::any(),
			ViaCacheTTLConfig::$default_ttl[apiGetEventsRequest::SOAP_METHOD_NAME]
		)->shouldHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\Cache\ViaCacheDriver $driver
	 */
	function it_performs_store_with_configured_ttl($driver)
	{
		$events = array(new ViaApiEvent);

		$notification = new ViaApiCallNotification(new apiGetEventsRequest());
		$notification->_setSuccessfulResponse($events);

		$this->subject->onViaPluginNotification($notification);

		$driver->store(
			Argument::any(),
			Argument::any(),
			Argument::any(),
			ViaCacheTTLConfig::$default_ttl[apiGetEventsRequest::SOAP_METHOD_NAME]
		)->shouldHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\Cache\ViaCacheDriver $driver
	 */
	function it_does_not_perform_lookup_or_store_if_ttl_for_method_is_zero($driver)
	{
		expect(ViaCacheTTLConfig::$default_ttl[apiAddTicketsToBasketRequest::SOAP_METHOD_NAME])->toBe(0);
		$this->given_notified_with_request_and_success_response(new apiAddTicketsToBasketRequest, TRUE);

		$driver->lookup(Argument::cetera())->shouldNotHaveBeenCalled();
		$driver->store(Argument::cetera())->shouldNotHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\Cache\ViaCacheDriver $driver
	 */
	function its_ttl_can_be_overridden_for_lookup_and_store($driver)
	{
		$this->subject->overrideMethodTtls(array(apiGetEventsRequest::SOAP_METHOD_NAME => 92000));
		$this->given_notified_with_request_and_success_response(new apiGetEventsRequest, TRUE);

		$driver->lookup(
			Argument::any(),
			Argument::any(),
			92000
		)->shouldHaveBeenCalled();

		$driver->store(
			Argument::any(),
			Argument::any(),
			Argument::any(),
			92000
		)->shouldHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\Cache\ViaCacheDriver                 $driver
	 * @param \Red61\Via\ApiRequest\apiGetSeatingPlanRequest  $request
	 */
	function it_uses_ttl_provided_by_cache_ttl_overriding_request_object_if_any($driver, $request)
	{
		$request->getSoapMethodName()->willReturn('someRequest');
		$request->getCacheTtl()->willReturn(13042);
		$this->given_notified_with_request_and_success_response($request->getWrappedObject(), TRUE);

		$driver->lookup(
			Argument::any(),
			Argument::any(),
			13042
		)->shouldHaveBeenCalled();

		$driver->store(
			Argument::any(),
			Argument::any(),
			Argument::any(),
			13042
		)->shouldHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\Cache\ViaCacheDriver $driver
	 */
	function it_performs_store_but_not_lookup_if_http_cache_control_headers_are_set($driver)
	{
		$_SERVER['HTTP_CACHE_CONTROL'] = 'no-cache';

		$this->given_notified_with_request_and_success_response(new apiGetEventsRequest, TRUE);

		$driver->lookup(Argument::cetera())->shouldNotHaveBeenCalled();
		$driver->store(Argument::cetera())->shouldHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\Cache\ViaCacheDriver $driver
	 * @param \Red61\Via\Cache\ViaCacheKeyGenerator $key_generator
	 */
	function it_ignores_http_cache_headers_if_configured_with_ignore_http_cache($driver, $key_generator)
	{
		$this->beConstructedWith($driver, $key_generator, array('ignore_http_headers' => TRUE));
		$_SERVER['HTTP_CACHE_CONTROL'] = 'no-cache';

		$this->given_notified_with_request_and_success_response(new apiGetEventsRequest, TRUE);

		$driver->lookup(Argument::cetera())->shouldHaveBeenCalled();
		$driver->store(Argument::cetera())->shouldHaveBeenCalled();
	}

	/**
	 * @param $request
	 * @param $response
	 */
	protected function given_notified_with_request_and_success_response($request, $response)
	{
		$notification = new ViaApiCallNotification($request);
		$this->subject->onViaPluginNotification($notification);
		$notification->_setSuccessfulResponse($response);
		$this->subject->onViaPluginNotification($notification);
	}

}
