<?php
/**
 * Defines Red61\Via\Exception\ViaExceptionMapper
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */

namespace Red61\Via\Exception;
use Red61\Via\ViaClassmap;

/**
 * Responsible for converting SoapFault to the expected exception types based on the defined exception classmap.
 *
 * @package Red61\Via\Exception
 * @see     spec\Red61\Via\Exception\ViaExceptionMapperSpec
 */
class ViaExceptionMapper {

	/**
	 * @var string[]
	 */
	protected $exception_classmap;

	/**
	 * @param string[] $exception_classmap optional, will be merged with shipped definition in ViaClassmap::$exceptionClassmap
	 *
	 * @see ViaClassmap::$exceptionClassmap
	 */
	public function __construct($exception_classmap = array())
	{
		$this->exception_classmap = array_merge(ViaClassmap::$exceptionClassmap, $exception_classmap);
	}

	/**
	 * Takes a SoapFault and returns a new exception instance of the underlying VIA exception type, with the details
	 * copied over.
	 *
	 * @param \SoapFault $fault
	 *
	 * @return ViaException
	 */
	public function map(\SoapFault $fault)
	{
    	$this->coalesceSoapFaultProperties($fault);
		$new_type = $this->getCustomExceptionTypeForSoapFault($fault);
		if ( ! class_exists($new_type)) {
			$new_type = '\Red61\Via\Exception\InvalidExceptionMappingException';
		}
		return $this->copyFaultToNewInstanceOf($fault, $new_type);
	}

	/**
	 * @param \SoapFault $fault
	 */
	protected function coalesceSoapFaultProperties(\SoapFault $fault)
	{
		foreach(array('faultcode', 'faultstring', 'faultactor', 'detail') as $property)
		{
			if ( ! isset($fault->$property))
			{
				$fault->$property = NULL;
			}
		}
	}

	/**
	 * Identifies what class a SoapFault should be converted to, based on the exception classmap
	 *
	 * @param \SoapFault $fault
	 *
	 * @return string
	 */
	protected function getCustomExceptionTypeForSoapFault(\SoapFault $fault)
	{
		$detail_type  = $this->getSingleObjectFieldnameFromDetail($fault);

		if ($detail_type AND isset($this->exception_classmap[$detail_type]))
			return $this->exception_classmap[$detail_type];

		if ($this->isHttpFault($fault))
			return '\Red61\Via\Exception\ViaHTTPException';

		if ($this->isViaInternalError($fault))
			return '\Red61\Via\Exception\ViaInternalServerErrorException';

		return $this->exception_classmap['_generic'];
	}

	/**
	 * Gets the detail property from a SoapFault (which may not always be set) and if it is a class with a single field
	 * returns the name of that field. Otherwise returns NULL.
	 *
	 * For example, a CartNotFoundException comes back as a SoapFault with the following structure
	 *
	 *     object(SoapFault) {
	 *        ...
	 *        ["detail"] => object(stdClass) (1) {
	 *          ["CartNotFoundException"] => string(0) ""
	 *        }
	 *     }
	 *
	 * But it is also possible for the detail property to be missing entirely, and presumably for it to have multiple
	 * keys or not even be an object depending on what XML comes back from the remote server.
	 *
	 * @param \SoapFault $fault
	 *
	 * @return string
	 */
	protected function getSingleObjectFieldnameFromDetail(\SoapFault $fault)
	{
		if ( ! is_object($fault->detail))
		{
			return NULL;
		}

		$vars = get_object_vars($fault->detail);
		if (count($vars) === 1) {
			return key($vars);
		}

		return NULL;
	}

	/**
	 * @param \SoapFault $fault
	 *
	 * @return bool
	 */
	protected function isHttpFault(\SoapFault $fault)
	{
		return in_array($fault->faultcode, array('HTTP', 'WSDL'));
	}

	/**
	 * @param \SoapFault $fault
	 *
	 * @return bool
	 */
	protected function isViaInternalError(\SoapFault $fault)
	{
		if ($fault->faultcode !== 'soap:Server')
			return FALSE;

		return (
			($fault->faultstring === 'java.lang.NullPointerException')
		);
	}

	/**
	 * @param \SoapFault $fault
	 * @param string     $new_class
	 *
	 * @return mixed
	*/
	protected function copyFaultToNewInstanceOf(\SoapFault $fault, $new_class)
	{
		return new $new_class(
			$fault->faultcode,
			$fault->faultstring,
			$fault->faultactor,
			$fault->detail
		);
	}

}
