<?php
/**
 * Defines Red61\Via\CircuitBreaker\ViaCoreCircuitBreaker
 *
 * @author    Andrew Coulton <andrew@ingenerator.com>
 * @copyright 2014 Edinburgh International Book Festival Ltd
 * @licence   http://opensource.org/licenses/BSD-3-Clause
 */

namespace Red61\Via\CircuitBreaker;

/**
 * Provides pluggable state storage for the ViaCircuitBreaker, which should be implemented
 * in shared storage - eg a cache - so that trips and error rates are tracked across all
 * processes. See the package README.md for full documentation.
 *
 * @package Red61\Via\CircuitBreaker
 */
interface ViaCircuitBreakerStorage {

	/**
	 * @return void
	 */
	public function trip();

	/**
	 * @return void
	 */
	public function markRetried();

	/**
	 * @return void
	 */
	public function reset();

	/**
	 * @return bool
	 */
	public function isTripped();

	/**
	 * Indicate whether this instance was the first to call trip() for a given incident
	 *
	 * @return bool
	 */
	public function wasFirstToTrip();

	/**
	 * Acquire a global lock for stampede protection when the retry interval expires.
	 *
	 * @param int $lifetime seconds after which the lock should expire
	 *
	 * @return bool
	 */
	public function tryLockForRetry($lifetime);

	/**
	 * Check whether a retry was attempted (and failed) within the last X seconds
	 *
	 * @param int $interval seconds
	 *
	 * @return bool
	 */
	public function wasRetriedWithinSeconds($interval);

	/**
	 * @param string $severity
	 *
	 * @return void
	 */
	public function incrementErrorRate($severity);

	/**
	 * @param string $severity
	 *
	 * @return int number of errors at this severity level in last minute
	 */
	public function getErrorsPerMinute($severity);
}
