<?php
/**
 * Defines Red61\Via\CircuitBreaker\ViaCoreCircuitBreaker
 *
 * @author    Andrew Coulton <andrew@ingenerator.com>
 * @copyright 2014 Edinburgh International Book Festival Ltd
 * @licence   http://opensource.org/licenses/BSD-3-Clause
 */

namespace Red61\Via\CircuitBreaker;

/**
 * This is the official implementation of the ViaCircuitBreaker interface. See the package
 * README.md for complete documentation.
 *
 * @package Red61\Via\CircuitBreaker
 * @see     spec\Red61\Via\CircuitBreaker\ViaCoreCircuitBreakerSpec
 */
class ViaCoreCircuitBreaker implements ViaCircuitBreaker {

	/**
	 * @var ViaCircuitBreakerStorage
	 */
	protected $storage;

	/**
	 * @var bool
	 */
	protected $stay_tripped;

	/**
	 * @var int[]
	 */
	protected $rate_limits;

	/**
	 * @var int
	 */
	protected $retry_interval;

	/**
	 * @var int
	 */
	protected $retry_lock_lifetime;

	/**
	 * @var ViaCircuitBreakerListener
	 */
	protected $listener;

	/**
	 * @param ViaCircuitBreakerStorage $storage
	 * @param array                    $options
	 */
	public function __construct(
		ViaCircuitBreakerStorage $storage,
		$options
	) {
		$this->storage = $storage;

		$options = array_merge(
			array(
				'retry_interval'       => 60,
				'retry_lock_lifetime'  => 10,
				'severity_rate_limits' => array()
			),
			$options
		);
		$this->rate_limits         = $options['severity_rate_limits'];
		$this->retry_interval      = $options['retry_interval'];
		$this->retry_lock_lifetime = $options['retry_lock_lifetime'];
	}

	/**
	 * @param string     $severity  One of the severity constants
	 * @param \Exception $exception The error thrown by the VIA client
	 *
	 * @return void
	 */
	public function notifyFailure($severity, \Exception $exception)
	{
		$this->storage->incrementErrorRate($severity);
		$rate_limit = $this->getRateLimit($severity);
		if ($rate_limit === NULL)
			return;

		if ($this->storage->isTripped())
		{
			$this->storage->markRetried();
		}
		else
		{
			$error_rate = $this->storage->getErrorsPerMinute($severity);
			if ($error_rate > $rate_limit)
			{
				$this->storage->trip();
				$this->notifyIfFirstToTrip($severity, $exception, $error_rate);
			}
		}
	}

	/**
	 * @param string $severity
	 *
	 * @return int or null if this severity is to be ignored
	 */
	protected function getRateLimit($severity)
	{
		return isset($this->rate_limits[$severity]) ? $this->rate_limits[$severity] : NULL;
	}

	/**
	 * @return bool
	 */
	public function requestPermissionToCall()
	{
		if ($this->stay_tripped)
		{
			$permission = FALSE;
		}
		elseif ( ! $this->isTripped())
		{
			$permission = TRUE;
		}
		elseif ($this->storage->wasRetriedWithinSeconds($this->retry_interval))
		{
			$permission = FALSE;
		}
		else
		{
			$permission = $this->storage->tryLockForRetry($this->retry_lock_lifetime);
		}

		if ($this->listener AND ! $permission)
		{
			$this->listener->onCallBlocked();
		}

		return $permission;
	}

	/**
	 * @return bool
	 */
	public function isTripped()
	{
		if ( ! $this->stay_tripped)
		{
			$this->stay_tripped = $this->storage->isTripped();
		}

		return $this->stay_tripped;
	}

	/**
	 * @return void
	 */
	public function reset()
	{
		if ($this->storage->isTripped())
		{
			$this->storage->reset();
			$this->notifyReset();
		}
		$this->stay_tripped = NULL;
	}

	/**
	 * @param ViaCircuitBreakerListener $listener
	 */
	public function setListener(ViaCircuitBreakerListener $listener)
	{
		$this->listener = $listener;
	}

	/**
	 * @return ViaCircuitBreakerListener
	 */
	public function getListener()
	{
		return $this->listener;
	}

	protected function notifyReset()
	{
		if ($this->listener)
		{
			$this->listener->onReset();
		}
	}

	/**
	 * @param string     $severity
	 * @param \Exception $exception
	 * @param int        $error_rate
	 */
	protected function notifyIfFirstToTrip($severity, \Exception $exception, $error_rate)
	{
		if ( ! $this->storage->wasFirstToTrip())
			return;

		if ($this->listener)
		{
			$this->listener->onTripped($severity, $exception, $error_rate);
		}
	}

}
