<?php
/**
 * Defines Red61\Via\CircuitBreaker\ViaApcCircuitBreakerStorage
 *
 * @author    Andrew Coulton <andrew@ingenerator.com>
 * @copyright 2014 Edinburgh International Book Festival Ltd
 * @licence   http://opensource.org/licenses/BSD-3-Clause
 */

namespace Red61\Via\CircuitBreaker;

/**
 * Provides shared state storage for the circuit breaker using the PHP APC cache. This will provide a
 * separate breaker per PHP process - rate limits should be configured accordingly. For full
 * documentation see the package documentation.
 *
 * @deprecated the apc extension is no longer supported in php - use apcu and the ViaApcuCircuitBreakerStorage
 *
 * @package Red61\Via\CircuitBreaker
 * @see     spec\Red61\Via\CircuitBreaker\ViaApcCircuitBreakerStorageSpec
 */
class ViaApcCircuitBreakerStorage implements ViaCircuitBreakerStorage {

	const APCKEY_TRIPPED_AT = 'via-breaker.tripped_at';
	const APCKEY_RETRIED_AT = 'via-breaker.retried_at';
	const APCKEY_RETRY_LOCK = 'via-breaker.retry_lock';
	const APCKEY_COUNT_NS   = 'via-breaker.rate';

	/**
	 * @var bool
	 */
	protected $was_first_to_trip;

	/**
	 * {@inheritdoc}
	 */
	public function trip()
	{
		$now = $this->time();
		$this->was_first_to_trip = apc_add(self::APCKEY_TRIPPED_AT, $now);
		apc_store(self::APCKEY_RETRIED_AT, $now);
	}

	/**
	 * {@inheritdoc}
	 */
	public function markRetried()
	{
		apc_store(self::APCKEY_RETRIED_AT, $this->time());
	}

	/**
	 * {@inheritdoc}
	 */
	public function reset()
	{
		apc_delete(self::APCKEY_TRIPPED_AT);
		apc_delete(self::APCKEY_RETRIED_AT);
	}

	/**
	 * {@inheritdoc}
	 */
	public function isTripped()
	{
		return (bool) apc_fetch(self::APCKEY_TRIPPED_AT);
	}

	/**
	 * {@inheritdoc}
	 */
	public function wasFirstToTrip()
	{
		return $this->was_first_to_trip;
	}

	/**
	 * {@inheritdoc}
	 */
	public function tryLockForRetry($lifetime)
	{
		$now = $this->time();

		$expire = $now + $lifetime;
		if (apc_add(self::APCKEY_RETRY_LOCK, $expire, $lifetime))
			return TRUE;

		// Couldn't add - a lock exists and may have expired, check for testability and APC TTL bugs
		$locked_till = apc_fetch(self::APCKEY_RETRY_LOCK);
		if ($locked_till <= $now)
		{
			// Lock has expired, claim it with CAS so that only one process gets the new lock
			return apc_cas(self::APCKEY_RETRY_LOCK, $locked_till, $expire);
		}

		return FALSE;
	}

	/**
	 * {@inheritdoc}
	 */
	public function wasRetriedWithinSeconds($interval)
	{
		$last_try = apc_fetch(self::APCKEY_RETRIED_AT);
		return ($this->time() < ($last_try + $interval));
	}

	/**
	 * {@inheritdoc}
	 */
	public function incrementErrorRate($severity)
	{
		$key = $this->getCountKeyPrefix($severity).$this->time();

		if ( ! apc_inc($key, 1))
		{
			// Key does not exist, attempt to create it. Set TTL just over 1 minute as value irrelevant after that
			if ( ! apc_add($key, 1, 61))
			{
				// Race condition - another process has added key, increment it
				apc_inc($key, 1);
			}
		}
	}

	/**
	 * {@inheritdoc}
	 */
	protected function getCountKeyPrefix($severity)
	{
		return self::APCKEY_COUNT_NS.'.'.$severity.'.';
	}

	/**
	 * [!!] This method is slightly expensive so be cautious about usage.
	 *
	 * {@inheritdoc}
	 */
	public function getErrorsPerMinute($severity)
	{
		$base_key = $this->getCountKeyPrefix($severity);
		$count    = 0;
		$now      = $this->time();
		for ($second = ($now - 59); $second <= $now; $second++)
		{
			$count += apc_fetch($base_key.$second, $success);
		}
		return $count;
	}

	/**
	 * @return int current time - in method for testability
	 */
	protected function time()
	{
		return time();
	}

}
