<?php
/**
 * Defines apiCreateAccountRequestSpec - specifications for Red61\Via\ApiRequest\apiCreateAccountRequest
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */

namespace spec\Red61\Via\ApiRequest;

use PhpSpec\Exception\Example\FailureException;
use Red61\Via\Exception\ClientInvalidRequestException;
use Red61\Via\Plugin\ViaApiCallNotification;
use spec\ObjectBehavior;
use Prophecy\Argument;

/**
 *
 * @see Red61\Via\ApiRequest\apiCreateAccountRequest
 */
class apiCreateAccountRequestSpec extends ObjectBehavior
{
    /**
     * Use $this->subject to get proper type hinting for the subject class
     * @var \Red61\Via\ApiRequest\apiCreateAccountRequest
     */
	protected $subject;

	function it_is_initializable()
    {
		$this->subject->shouldHaveType('Red61\Via\ApiRequest\apiCreateAccountRequest');
	}

	function it_is_a_preflight_filtering_request()
	{
		$this->subject->shouldHaveType('Red61\Via\ApiRequest\PreflightFilteringRequest');
	}

	function it_trims_preferred_contact_method()
	{
		$this->subject->setPreferredContactMethod('any ')
			->getPreferredContactMethod()->shouldBe('any');
	}

	function it_trims_forename()
	{
		$this->subject->setForename(' name ')
			->getForename()->shouldBe('name');
	}

	function it_trims_surname()
	{
		$this->subject->setSurname(' name ')
			->getSurname()->shouldBe('name');
	}

	function it_trims_email()
	{
		$this->subject->setSurname(' email@server.com ')
			->getSurname()->shouldBe('email@server.com');
	}

	/**
	 * @param \Red61\Via\Plugin\ViaApiCallNotification $notification
	 */
	function it_does_not_throw_preflight_with_valid_data($notification)
	{
		$this->given_set_with_valid_data_and(array());
		$this->subject->onBeforeViaCall($notification);
	}

	function it_throws_preflight_if_customer_id_not_set_and_create_false()
	{
		$this->given_set_with_valid_data_and(array('customerId' => NULL, 'create' => FALSE));
		$this->shouldThrowOnPreflightWithInvalidParameters(array('customerId'));
	}

	function it_throws_preflight_if_customer_id_set_and_create_true()
	{
		$this->given_set_with_valid_data_and(array('customerId' => 12, 'create' => TRUE));
		$this->shouldThrowOnPreflightWithInvalidParameters(array('customerId'));
	}

	function it_throws_preflight_if_customer_id_null_and_create_true()
	{
		$this->given_set_with_valid_data_and(array('customerId' => NULL, 'create' => TRUE));
		$this->shouldThrowOnPreflightWithInvalidParameters(array('customerId'));
	}

	function it_throws_preflight_if_forename_not_set()
	{
		$this->given_set_with_valid_data_and(array('forename' => NULL));
		$this->shouldThrowOnPreflightWithInvalidParameters(array('forename'));
	}

	function it_throws_preflight_if_surname_less_than_2_characters()
	{
		$this->given_set_with_valid_data_and(array('surname' => 'A'));
		$this->shouldThrowOnPreflightWithInvalidParameters(array('surname'));
	}

	function it_throws_preflight_if_email_not_set()
	{
		$this->given_set_with_valid_data_and(array('email' => ''));
		$this->shouldThrowOnPreflightWithInvalidParameters(array('email'));
	}

	/**
	 * Set up the object state
	 *
	 * @param mixed[] $properties
	 */
	protected function given_set_with_valid_data_and($properties)
	{
		$defaults = array(
			'create'     => TRUE,
			'customerId' => 0,
			'forename'   => 'first',
			'surname'    => 'last',
			'email'      => 'email'
		);
		$values = array_merge($defaults, $properties);
		$this->subject
			->setCreate($values['create'])
			->setCustomerId($values['customerId'])
			->setForename($values['forename'])
			->setSurname($values['surname'])
			->setEmail($values['email']);
	}

	/**
	 * @param string[] $expect_params
	 *
	 * @throws \PhpSpec\Exception\Example\FailureException
	 */
	protected function shouldThrowOnPreflightWithInvalidParameters($expect_params)
	{
		$notification = new ViaApiCallNotification($this->subject->getWrappedObject());
		try {
			$this->subject->onBeforeViaCall($notification);
			throw new FailureException("Expected exception was not thrown");
		} catch (ClientInvalidRequestException $e) {
			$invalid_params = $e->getInvalidParameters();
			foreach ($expect_params as $param_name) {
				expect($invalid_params)->toHaveKey($param_name);
			}
		}
	}

}
