<?php
/**
* Defines the ViaApiService service wrapper class
*
* @author    Red61 Ltd
* @copyright 2022 Red61 Ltd
* @licence   proprietary
* @version   7.2.x
*/

namespace Red61\Via;

/**
 * A SOAP endpoint that defines an API for interacting with an instance of VIA Server. This API provides operations to:
 * 
 * - list events;
 * - list performances for events;
 * - search events and performances;
 * - add tickets for a performance to a shopping basket;
 * - list non ticket items such as schemes, merchandise and vouchers;
 * - manage a shopping basket;
 * - create orders;
 * - create customers;
 * - update customers' details;
 * - customer log in.
 * 
 * An interaction with this endpoint with the server taking payment would follow the following workflow:
 * 
 * - search or list things using the get/list/search methods
 * - create a new basket using `apiGetBasketSessionId`
 * - add tickets to the basket using `apiAddTicketsToBasket`
 * - create a customer record using `apiCreateAccount`
 * - checkout and create order using `apiCreateOrder` If this call returns code 'serverId:-302' then you must redirect
 * to the acs URL provided in statusmessage for 3D secure authorisation. The statusmessage contains the MD and paReq
 * required for authorisation and is in the form '`acsUrl+"?MD="+MD+"&paReq="+paReq`'. Once the authorisation is
 * completed pass the paRes returned from the acs URL and MD to the call `apiSend3DSecureReturn` for final 3D secure
 * settlement. If this succeeds then this call will complete the order and remove the basket. If this call fails then
 * the process can be attempted again by calling `apiCreateOrder` which will add another payment to the current order
 * linked to the basket.
 * 
 * An interaction with this endpoint where payment processing is done independently from the server would follow the
 * following workflow:
 * 
 * - search or list things using the get/list/search methods
 * - create a new basket using `apiGetBasketSessionId`
 * - add tickets to the basket using `apiAddTicketsToBasket`
 * - create a customer record using `apiCreateAccount`
 * - create initial 'Unpaid' order using `apiCreateInitialUnpaidOrder`
 * - process payment independently
 * - complete order using `apiCompleteOrder`
 * 
 * If the payment processed independently was declined you can optionally call `apiCompleteOrder`. This persists to the
 * database the declined payment information. In order to attempt payment again you must repeat the process 5 - 7 again.
 * This does not created a new order but will add another unpaid payment to the existing order. Complete order will only
 * 'complete' the order if the payment is taken successfully.

 * @package    Red61\Via * @{inheritdoc}
 */
class ViaApiService {

  /**
   * @var ViaApiClient $api_client
   */
  protected $api_client;

  /**
   * @param ViaApiClient $api_client
   */
  public function __construct(ViaApiClient $api_client)
  {
    $this->api_client = $api_client;
  }

  /**
   * Lists prices for the specified performance. This method will attempt translation of it's returned objects, where
   * available.
   *
   * @param \Red61\Via\ApiRequest\apiGetPerformancePricesRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiPerformancePrices
   */
  public function getPerformancePrices(\Red61\Via\ApiRequest\apiGetPerformancePricesRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Lists performances for the supplied event which are participating in the supplied schemes. This method will attempt
   * translation of it's returned objects, where available.
   *
   * @deprecated This method will probably be replaced when the new more flexible concessions system is added in version 5.1
   *
   * @param \Red61\Via\ApiRequest\apiGetPerformancesBySchemeRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiPerformance[]
   */
  public function getPerformancesByScheme(\Red61\Via\ApiRequest\apiGetPerformancesBySchemeRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Lists performances for the supplied event. This method will attempt translation of it's returned objects, where
   * available.
   *
   * @param \Red61\Via\ApiRequest\apiGetPerformancesRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiPerformance[]
   */
  public function getPerformances(\Red61\Via\ApiRequest\apiGetPerformancesRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Lists unique performance dates for the supplied event. Designed to be used to build a calendar of performances -
   * when the user selects a date then use `apiGetPerformancesByDate` to fetch the performances on that date. Digital
   * performances which have no date and time are returned grouped in the first element with a `null`
   * `ViaApiPerformanceDate#date`. Digital performances will also have an empty `ViaApiPerformanceDate#times`
   * collection. This method will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetPerformanceDatesRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiPerformanceDate[]
   */
  public function getPerformanceDates(\Red61\Via\ApiRequest\apiGetPerformanceDatesRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Lists all visible events. This method will attempt translation of it's returned objects, where available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiEvent[]
   */
  public function getEvents()
  {
    $request = new \Red61\Via\ApiRequest\apiGetEventsRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Returns the event details for a given event. This method will attempt translation of it's returned objects, where
   * available.
   *
   * @param \Red61\Via\ApiRequest\apiGetEventDetailsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiEvent
   */
  public function getEventDetails(\Red61\Via\ApiRequest\apiGetEventDetailsRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Clears any items from the current basket (such as tickets and schemes). This sets any seats in the basket back on
   * sale. If an order has been created line items are deleted too and any remote inventory server records relating to
   * the order. If `removeBasket` is `true`, it removes the basket so the `sessionId` is no longer valid after this call
   * has succeeded.
   *
   * @param \Red61\Via\ApiRequest\apiClearBasketRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function clearBasket(\Red61\Via\ApiRequest\apiClearBasketRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Removes tickets from the basket identified by `sessionId`.
   *
   * @param \Red61\Via\ApiRequest\apiRemoveTicketRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function removeTicket(\Red61\Via\ApiRequest\apiRemoveTicketRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * This method may be used to change the priceband concession for items already in the basket.
   *
   * @param \Red61\Via\ApiRequest\apiUpdateTicketConcessionRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function updateTicketConcession(\Red61\Via\ApiRequest\apiUpdateTicketConcessionRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Creates or updates a customer record. If `create` is `true` and `customerId` is 0 then a new account is created.
   * Otherwise, if `create` is `false` then the record identified by `customerId` is updated. If there is an existing
   * customer record with the same email and password, the existing record will be silently updated rather than creating
   * a new customer, even though create == true
   *
   * @param \Red61\Via\ApiRequest\apiCreateAccountRequest $request
   *
   * @return int
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function createAccount(\Red61\Via\ApiRequest\apiCreateAccountRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Creates the entire order => order record, line items and processes payment. Processes the payment using the
   * relevant Payment Processor linked to the WEB location. If the transaction total is ZERO (items + handling and
   * postage fee) then the card processing is bypassed and a complimentary payment is created instead. Currently though
   * card fields are required for a complimentary payment so pass in `4111111111111111` as the card number, a valid
   * expiry date and any three digit cv2 number (123 for example). This will pass the card checking.
   *
   * @param \Red61\Via\ApiRequest\apiCreateOrderRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiOrderDetails
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   * @throws \Red61\Via\Exception\UnavailablePrintOptionException If the given delivery option is not possible with the current basket
   */
  public function createOrder(\Red61\Via\ApiRequest\apiCreateOrderRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * 
   *
   * @param \Red61\Via\ApiRequest\apiCreateRepeatPaymentOrderRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiOrderDetails
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   * @throws \Red61\Via\Exception\UnavailablePrintOptionException If the given delivery option is not possible with the current basket
   */
  public function createRepeatPaymentOrder(\Red61\Via\ApiRequest\apiCreateRepeatPaymentOrderRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Creates the entire initial order => order record, line items and 'unpaid' payment record. If this does not return a
   * valid payment Id (greater than ZERO) then DO NOT proceed to take payment. If this method returns a valid payment Id
   * then it is safe to proceed to take payment. The result of that payment attempt must be passed to `apiCompleteOrder`
   * to complete the order. If the payment fails then this method can be called again and will persist the new unpaid
   * payment against the original for processing. Remember to only proceed to payment if this method returns a valid
   * paymentId (greater than ZERO).
   *
   * @param \Red61\Via\ApiRequest\apiCreateInitialUnpaidOrderRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiOrderDetails
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   * @throws \Red61\Via\Exception\UnavailablePrintOptionException If the given delivery option is not possible with the current basket
   */
  public function createInitialUnpaidOrder(\Red61\Via\ApiRequest\apiCreateInitialUnpaidOrderRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Creates the initial order record and provides html for display to allow the Payment Service Provider to capture the
   * card details. Card Details are captured and send directly to the Payment Service Provider, so this method may be
   * used to provide solution where website hosting does not need to be PCI:DSS complaint
   * 
   * NOTE: Must call `apiSetDeliveryOption` before calling this method.
   * 
   * After `ViaApiCardPaymentRequest#paymentFormHTML` has been displayed and payment has been taken call
   * `apiSendCardPaymentReturn` to complete the order.
   * 
   * If the payment fails then this method can be called again and will prepare a new payment against the original order
   * for processing. Remember to only proceed to payment if this method returns a populated
   * `ViaApiCardPaymentRequest#paymentFormHTML`
   *
   * @param \Red61\Via\ApiRequest\apiCreateCardPaymentOrderRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiCardPaymentRequest
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   * @throws \Red61\Via\Exception\UnavailablePrintOptionException If the given delivery option is not possible with the current basket
   */
  public function createCardPaymentOrder(\Red61\Via\ApiRequest\apiCreateCardPaymentOrderRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Completes the order and payment flow after a successful payment has been taken by the Payment Service Provider.
   * This method persists the payment information to the database, and completes the order depending on whether the
   * payment was taken successfully or not. If payment was taken successfully all remote inventory servers in the order
   * are dispatched with payment and order information. If no errors occur then the basket in removed. If the payment
   * did not succeed then returning code will be a paymentError allowing another payment to be attempted by by starting
   * the whole process again calling `apiCreateCardPaymentOrder`
   *
   * @param \Red61\Via\ApiRequest\apiSendCardPaymentReturnRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiOrderDetails
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function sendCardPaymentReturn(\Red61\Via\ApiRequest\apiSendCardPaymentReturnRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Creates the entire unpaid account order => order record, line items and 'unpaid' account payment record. If this
   * does not return a valid order Id (greater than ZERO) then assume that the unpaid account order has not been
   * created. If this method returns a valid order Id then it is safe to proceed to call `apiCompleteAccountOrder` to
   * complete the order. If the payment fails then this method can be called again and will persist a new unpaid payment
   * against the original transaction for processing. This will only work however if you first call
   * `apiCompleteAccountOrder` to process the 'failed' payment. Remember to only proceed to take payment if this method
   * returns a valid orderId (greater than ZERO). The webuser identified by `webkey` must have system permission to
   * create transactions with account payments
   *
   * @param \Red61\Via\ApiRequest\apiCreateUnpaidAccountOrderRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiOrderDetails
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   * @throws \Red61\Via\Exception\UnavailablePrintOptionException If the given delivery option is not possible with the current basket
   */
  public function createUnpaidAccountOrder(\Red61\Via\ApiRequest\apiCreateUnpaidAccountOrderRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * This method persists the payment information to the database, and completes the order depending on whether the
   * payment was taken successfully or not. If payment was taken successfully all remote inventory servers in the order
   * are dispatched with payment and order information. If no errors occur then the basket in removed. If the payment
   * did not succeed then returning code will be a paymentError allowing another payment to be attempted by by starting
   * the whole process again calling `apiCreateInitialUnpaidOrder` -> process payment independently ->
   * `apiCompleteOrder`
   *
   * @param \Red61\Via\ApiRequest\apiCompleteOrderRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiOrderDetails
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function completeOrder(\Red61\Via\ApiRequest\apiCompleteOrderRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * This method marks the account payment as paid or declined depending on the flag `paymentSuccess`. If payment was
   * taken successfully all remote inventory servers in the order are dispatched with payment and order information. If
   * no errors occur then the basket in removed. If the payment did not succeed then returning code will be a
   * paymentError allowing another payment to be attempted by by starting the whole process again calling
   * `apiCreateUnpaidAccountOrder` -> process payment independently -> `apiCompleteAccountOrder`
   *
   * @param \Red61\Via\ApiRequest\apiCompleteAccountOrderRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiOrderDetails
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function completeAccountOrder(\Red61\Via\ApiRequest\apiCompleteAccountOrderRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Creates a reservation with the items in the given basket for the given customer. This call may fail for several
   * reasons. Reason for failure will be indicated in the returned objects code and status messages. Calls may fail for
   * the following reasons.
   * 
   * - The web user does not have permission to create reservations.
   * - The basket contains non ticket items.
   * - The user does not have permission to reserve one or more items in the basket.
   * - One or more of the tickets in the basket are no longer on sale.
   * - The given customer is not valid.
   *
   * @param \Red61\Via\ApiRequest\apiCreateReservationRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiOrderDetails
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function createReservation(\Red61\Via\ApiRequest\apiCreateReservationRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * For completing the 3-D secure process. If `apiCreateOrder` returns a re-direct and code of 'serverId:-302' then the
   * card being processed requires extra 3-D secure authorisation - currently only Maestro cards require 3-D secure
   * authorisation. The resulting `md` and `paRes` fields must be passed to this call to complete the 3-D secure
   * transaction and, if payment succeeds, this call will complete the order and 'remove' the basket.
   *
   * @param \Red61\Via\ApiRequest\apiSend3DSecureReturnRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiOrderDetails
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function send3DSecureReturn(\Red61\Via\ApiRequest\apiSend3DSecureReturnRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Gets details about a given performance. This method will attempt translation of it's returned objects, where
   * available.
   *
   * @param \Red61\Via\ApiRequest\apiGetPerformanceDetailsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiPerformance
   */
  public function getPerformanceDetails(\Red61\Via\ApiRequest\apiGetPerformanceDetailsRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Gets details about the tickets in a given order. This method will attempt translation of it's returned objects,
   * where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetOrderTicketDetailsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiTicketDetails[]
   */
  public function getOrderTicketDetails(\Red61\Via\ApiRequest\apiGetOrderTicketDetailsRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Gets details about the tickets and their scanned status for a given order. This differs to
   * `apiGetOrderTicketDetails` due to the returned object `ViaApiTicketDetailsWithStatus` containing a `List` of
   * `ViaApiTicketItemDetailsWithStatus` objects which contain the field
   * `ViaApiTicketItemDetailsWithStatus#availableToScan` This method will attempt translation of it's returned objects,
   * where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetOrderTicketDetailsWithStatusRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiTicketDetailsWithStatus[]
   */
  public function getOrderTicketDetailsWithStatus(\Red61\Via\ApiRequest\apiGetOrderTicketDetailsWithStatusRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Creates a new basket for the supplied `webKey`
   *
   *
   * @return string
   * @throws \Red61\Via\Exception\TooManyCartsException If the system is under high load and will not create another basket.
   */
  public function getBasketSessionId()
  {
    $request = new \Red61\Via\ApiRequest\apiGetBasketSessionIdRequest;
    return $this->api_client->send($request);
  }

  /**
   * Adds the supplied tickets with the attributes to the basket identified by `sessionId`.
   *
   * @param \Red61\Via\ApiRequest\apiAddTicketsWithAttributesToBasketRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   * @throws \Red61\Via\Exception\AddSeatsException If there was an issue when attempting to add the requested seats to the customer's basket.
   */
  public function addTicketsWithAttributesToBasket(\Red61\Via\ApiRequest\apiAddTicketsWithAttributesToBasketRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Adds the supplied tickets to the basket identified by `sessionId`.
   *
   * @param \Red61\Via\ApiRequest\apiAddTicketsToBasketRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   * @throws \Red61\Via\Exception\AddSeatsException If there was an issue when attempting to add the requested seats to the customer's basket.
   */
  public function addTicketsToBasket(\Red61\Via\ApiRequest\apiAddTicketsToBasketRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * This method may be used to assign ticket(s) as wheelchair users for an unreserved performance already in the
   * basket.
   *
   * @param \Red61\Via\ApiRequest\apiUpdateWheelChairTicketsInBasketRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   * @throws \Red61\Via\Exception\WheelchairUpdateException If there is an issue updating the wheelchair status of a ticket.
   * @throws \Red61\Via\Exception\FailedConvertException If there is an issue converting in order tickets back on sale.
   * @throws \Red61\Via\Exception\NoOffersException If there is an issue adding the newly requested Wheelchair seats to cache.
   */
  public function updateWheelChairTicketsInBasket(\Red61\Via\ApiRequest\apiUpdateWheelChairTicketsInBasketRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Adds and updates the specified access note for performance(s) already in the basket identified by `sessionId`. For
   * the note to be added to a performance: 1. A customer should already be attached to the basket. 2. A ticket for the
   * target performance should already be in the basket. 3. At least one `accessNoteCategoryId` should be specified.
   *
   * @param \Red61\Via\ApiRequest\apiAddPerformanceAccessNotesToBasketRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\AccessNotesException If there is an issue preventing the access note from being added to the basket
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function addPerformanceAccessNotesToBasket(\Red61\Via\ApiRequest\apiAddPerformanceAccessNotesToBasketRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Removes access note from the specified performances in the basket identified by `sessionId`.
   *
   * @param \Red61\Via\ApiRequest\apiRemovePerformanceAccessNotesFromBasketRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\AccessNotesException If there is an issue preventing the access note from being removed from the basket
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function removePerformanceAccessNotesFromBasket(\Red61\Via\ApiRequest\apiRemovePerformanceAccessNotesFromBasketRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Adds the specified scheme to the basket identified by `sessionId`.
   *
   * @param \Red61\Via\ApiRequest\apiAddSchemeToBasketRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   * @throws \Red61\Via\Exception\SchemeUnavailableException If the scheme tier requested or the scheme is not available to the web
   */
  public function addSchemeToBasket(\Red61\Via\ApiRequest\apiAddSchemeToBasketRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Removes the specified scheme from the basket identified by `sessionId`.
   *
   * @param \Red61\Via\ApiRequest\apiRemoveSchemeFromBasketRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function removeSchemeFromBasket(\Red61\Via\ApiRequest\apiRemoveSchemeFromBasketRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Returns the details for a particular scheme
   * 
   * ViaApiSchemeDetails contains a nested array of ViaApiSchemeTierDetails objects This method will attempt translation
   * of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetSchemeDetailsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiSchemeDetails
   * @throws \Red61\Via\Exception\SchemeUnavailableException if the scheme is not available on the web
   */
  public function getSchemeDetails(\Red61\Via\ApiRequest\apiGetSchemeDetailsRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Returns the valid and current schemeRefs that a customer is subscribed to. This reflects the scheme and the
   * selected tier. - serverid:schemeId:tierId
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerSchemesRequest $request
   *
   * @return string[]
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function getCustomerSchemes(\Red61\Via\ApiRequest\apiGetCustomerSchemesRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Returns the ViaApiSchemeTierDetails for a specified scheme and tier This method will attempt translation of it's
   * returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetSchemeTierDetailsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiSchemeTierDetails
   * @throws \Red61\Via\Exception\SchemeUnavailableException if the scheme or scheme tier is not available on the web
   */
  public function getSchemeTierDetails(\Red61\Via\ApiRequest\apiGetSchemeTierDetailsRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Since version 4.1 this method is the method for fetching the basket contents This method will attempt translation
   * of it's returned objects, where available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiBasket
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function getBasketItems()
  {
    $request = new \Red61\Via\ApiRequest\apiGetBasketItemsRequest;
    return $this->api_client->send($request);
  }

  /**
   * Returns a list of the donation funds linked to a specific event This method will attempt translation of it's
   * returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetDonationsForEventRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiDonationFund[]
   */
  public function getDonationsForEvent(\Red61\Via\ApiRequest\apiGetDonationsForEventRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Returns a list of the suggested donations for this basket with `ViaApiDonationFund#amount` set to the suggested
   * amount. This method will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetBasketSuggestedDonationsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiDonationFund[]
   * @throws \Red61\Via\Exception\CartNotFoundException If `sessionId` does not reference an active basket.
   */
  public function getBasketSuggestedDonations(\Red61\Via\ApiRequest\apiGetBasketSuggestedDonationsRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Returns the text content of the Gift Aid statement, this must be displayed next to the option on the website or
   * application next to the option to gift aid donation option
   *
   *
   * @return string
   */
  public function getGiftAidText()
  {
    $request = new \Red61\Via\ApiRequest\apiGetGiftAidTextRequest;
    return $this->api_client->send($request);
  }

  /**
   * Sets the createGiftAidDeclaration flag, this flag will determine if a gift aid declaration will be created for that
   * customer (if required).
   *
   * @param \Red61\Via\ApiRequest\apiSetCreateGiftAidDeclarationRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   * @throws \Red61\Via\Exception\IllegalGiftAidException if gift aid is not applicable
   */
  public function setCreateGiftAidDeclaration(\Red61\Via\ApiRequest\apiSetCreateGiftAidDeclarationRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Returns information on the current cart count and current cart limit. This may be used if you wish to queue access
   * to your website during periods of high demand
   *
   *
   * @return \Red61\Via\DataObject\ViaApiBasketLimits
   */
  public function getBasketLimits()
  {
    $request = new \Red61\Via\ApiRequest\apiGetBasketLimitsRequest;
    return $this->api_client->send($request);
  }

  /**
   * Gets the customer id associated with the supplied e-mail address and password.
   *
   * @param \Red61\Via\ApiRequest\apiLoginRequest $request
   *
   * @return int
   * @throws \Red61\Via\Exception\LoginDisabledException if the customer has had their ability to login disabled
   */
  public function login(\Red61\Via\ApiRequest\apiLoginRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Gets the full name (including title) for a given customer.
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerNameRequest $request
   *
   * @return string
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function getCustomerName(\Red61\Via\ApiRequest\apiGetCustomerNameRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Returns events within a specified time. This method will attempt translation of it's returned objects, where
   * available.
   *
   * @param \Red61\Via\ApiRequest\apiFilterEventsByTimeRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiEvent[]
   */
  public function filterEventsByTime(\Red61\Via\ApiRequest\apiFilterEventsByTimeRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Gets the customers friend status
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerFriendStatusRequest $request
   *
   * @return int
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function getCustomerFriendStatus(\Red61\Via\ApiRequest\apiGetCustomerFriendStatusRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Sets the customers friend status
   *
   * @param \Red61\Via\ApiRequest\apiSetCustomerFriendStatusRequest $request
   *
   * @return int
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function setCustomerFriendStatus(\Red61\Via\ApiRequest\apiSetCustomerFriendStatusRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Updates the customers access requirements
   *
   * @param \Red61\Via\ApiRequest\apiUpdateCustomerAccessRequirementRequest $request
   *
   * @return int
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function updateCustomerAccessRequirement(\Red61\Via\ApiRequest\apiUpdateCustomerAccessRequirementRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Gets all customer details for a supplied customer ID.
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerDetailsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiCustomerDetails
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function getCustomerDetails(\Red61\Via\ApiRequest\apiGetCustomerDetailsRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Updates the customer's extended details returned by apiGetCustomerDetails
   *
   * @param \Red61\Via\ApiRequest\apiUpdateCustomerOtherDetailsRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function updateCustomerOtherDetails(\Red61\Via\ApiRequest\apiUpdateCustomerOtherDetailsRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Checks to see that the earliest performance in the shopping basket is outwith the post cut off time. `cutoff` is
   * not currently stored in the db so is hardcoded and stored within session management. It is used to decide whether
   * to give the customer the option of having there tickets posted to them.
   *
   * @deprecated 
   *
   * @param \Red61\Via\ApiRequest\apiIsPostAllowedRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function isPostAllowed(\Red61\Via\ApiRequest\apiIsPostAllowedRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Returns a `ViaApiAvailablePrintOptionReturn` object containing a list of valid delivery options available
   * calculated against the current contents of the basket, also contains a String basketChangedMessage the presence of
   * which indicates that the basket's print option has been changed due to the basket contents not allowing the
   * originally selected print option. It is advisable to refresh the basket contents if this exception is populated.
   * 
   * Post is only available if all performance's in the basket are not after the `postCutOff` set and E-Ticket is only
   * available if all the sub venues in the basket are enabled for E-Ticket. Therefore these delivery options can change
   * after any basket contents change.
   *
   * @deprecated use `apiGetDeliveryOptions` to fetch `ViaApiBasketDeliveryOptions` instead
   *
   * @param \Red61\Via\ApiRequest\apiGetBasketPrintOptionsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiAvailablePrintOptionReturn
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function getBasketPrintOptions(\Red61\Via\ApiRequest\apiGetBasketPrintOptionsRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Searches for events based on a supplied keyword. Matches are based on:
   * 
   * - event titles;
   * - performance dates;
   * - venue names;
   * - sub venue names;
   * - companies;
   * - artists;
   * - promoters;
   * - performance dates (DD/MM/YY) This method will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiEventQuickSearchRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiEvent[]
   */
  public function eventQuickSearch(\Red61\Via\ApiRequest\apiEventQuickSearchRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Searches for events based on the supplied parameters. `typeid` and `subtypeid` must be used in conjunction with
   * other parameters (otherwise no results are returned). This method will attempt translation of it's returned
   * objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiEventDetailedSearchRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiEvent[]
   */
  public function eventDetailedSearch(\Red61\Via\ApiRequest\apiEventDetailedSearchRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * This method will attempt translation of it's returned objects, where available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiSeasonDetails[]
   */
  public function getActiveSeasons()
  {
    $request = new \Red61\Via\ApiRequest\apiGetActiveSeasonsRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Searches for events based on the supplied parameters. This method will attempt translation of it's returned
   * objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiFilterEventsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiEvent[]
   */
  public function filterEvents(\Red61\Via\ApiRequest\apiFilterEventsRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Gets a purchase summary for a given order ID. This method will attempt translation of it's returned objects, where
   * available.
   *
   * @param \Red61\Via\ApiRequest\apiGetOrderSummaryRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiPurchaseSummary
   */
  public function getOrderSummary(\Red61\Via\ApiRequest\apiGetOrderSummaryRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Gets the full transaction details for a given order ID. Identical to `apiGetOrderSummary` but includes ticket
   * scanning information `ViaApiTicketItemDetailsWithStatus#availableToScan` This method will attempt translation of
   * it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetTransactionDetailsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiTransactionDetails
   */
  public function getTransactionDetails(\Red61\Via\ApiRequest\apiGetTransactionDetailsRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Gets the venues visible using the supplied authorisation code `webKey`. This method will attempt translation of
   * it's returned objects, where available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiListElement[]
   */
  public function getVenues()
  {
    $request = new \Red61\Via\ApiRequest\apiGetVenuesRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Gets the sub venues visible or sub venues of the venue if specified, using the supplied authorisation code
   * `webKey`. This method will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetSubVenuesRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiListElement[]
   */
  public function getSubVenues(\Red61\Via\ApiRequest\apiGetSubVenuesRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Returns details on a given venue This method will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetVenueDetailsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiVenueDetails
   */
  public function getVenueDetails(\Red61\Via\ApiRequest\apiGetVenueDetailsRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Returns Details on a given subvenue This method will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetSubVenueDetailsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiSubVenueDetails
   */
  public function getSubVenueDetails(\Red61\Via\ApiRequest\apiGetSubVenueDetailsRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Gets all valid schemes.
   * 
   * Does not query other servers since only a schemes owning server can sell schemes anyway. This method will attempt
   * translation of it's returned objects, where available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiListElement[]
   */
  public function getSchemes()
  {
    $request = new \Red61\Via\ApiRequest\apiGetSchemesRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Return a list of all customer categories. With the provided customers categories being marked as active. This
   * method will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerCategoriesRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiCustomerCategory[]
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function getCustomerCategories(\Red61\Via\ApiRequest\apiGetCustomerCategoriesRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Set the customer categories of this customer
   *
   * @param \Red61\Via\ApiRequest\apiSetCustomerCategoriesRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function setCustomerCategories(\Red61\Via\ApiRequest\apiSetCustomerCategoriesRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Get all the available postage charges for tickets
   * 
   * **WARNING: this may return different Postage Fees for tickets compared to `apiGetDeliveryOptions` as it is not
   * aware of the basket** This method will attempt translation of it's returned objects, where available.
   *
   * @deprecated instead use `apiGetDeliveryOptions` which is basket aware
   *
   *
   * @return \Red61\Via\DataObject\ViaApiPostageCharge[]
   */
  public function getPostageCharges()
  {
    $request = new \Red61\Via\ApiRequest\apiGetPostageChargesRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Sets the postage charge for the basket referenced by sessionId.
   *
   * @deprecated use `apiSetDeliveryOption` instead.
   *
   * @param \Red61\Via\ApiRequest\apiSetPostageChargeRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   * @throws \Red61\Via\Exception\IllegalArgumentException If the session id is null or if chargeId is not 0 or a valid postage id.
   */
  public function setPostageCharge(\Red61\Via\ApiRequest\apiSetPostageChargeRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Gets all the event types. This method will attempt translation of it's returned objects, where available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiTypeListElement[]
   */
  public function getEventTypes()
  {
    $request = new \Red61\Via\ApiRequest\apiGetEventTypesRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Gets all artists (companies).
   *
   *
   * @return \Red61\Via\DataObject\ViaApiListElement[]
   */
  public function getArtists()
  {
    $request = new \Red61\Via\ApiRequest\apiGetArtistsRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Gets a summary of the contents of the supplied basket which includes total, handling fee, total and number of
   * tickets in the basket. This method will attempt translation of it's returned objects, where available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiBasketSummary
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function getBasketSummary()
  {
    $request = new \Red61\Via\ApiRequest\apiGetBasketSummaryRequest;
    return $this->api_client->send($request);
  }

  /**
   * Clean up an unpaid api order.
   *
   * @param \Red61\Via\ApiRequest\apiCleanUnpaidOrderRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function cleanUnpaidOrder(\Red61\Via\ApiRequest\apiCleanUnpaidOrderRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Search for full address using the current Address lookup provider
   *
   * @param \Red61\Via\ApiRequest\apiSearchAddressRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiAddressResult[]
   */
  public function searchAddress(\Red61\Via\ApiRequest\apiSearchAddressRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Convert the address identifier from the apiSearchAddress() call into a fully populated AddressDetails object
   *
   * @param \Red61\Via\ApiRequest\apiGetAddressRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiAddressDetails
   */
  public function getAddress(\Red61\Via\ApiRequest\apiGetAddressRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Create a gift aid declaration for the customer identified by `customerId`. The start date is the system default or
   * date now and the end date is assumed to be forever.
   *
   * @param \Red61\Via\ApiRequest\apiCreateGiftAidDeclarationRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\GiftAidDeclarationOverlapException if the customer already has an active declaration covering date now.
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function createGiftAidDeclaration(\Red61\Via\ApiRequest\apiCreateGiftAidDeclarationRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * 
   *
   * @deprecated use `apiSendPasswordResetEmail` instead.
   *
   * @param \Red61\Via\ApiRequest\apiCustomerPasswordResetRequest $request
   *
   * @return string
   * @throws \Red61\Via\Exception\DuplicateRecordException if there is more than one customer with this email address.
   * @throws \Red61\Via\Exception\NoResultsException If an account with the given email address could not be found.
   */
  public function customerPasswordReset(\Red61\Via\ApiRequest\apiCustomerPasswordResetRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * 
   *
   * @param \Red61\Via\ApiRequest\apiCustomerEmailExistsRequest $request
   *
   * @return bool
   */
  public function customerEmailExists(\Red61\Via\ApiRequest\apiCustomerEmailExistsRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Check if the customer record exists for the given email password combination, is active and is web login enabled.
   *
   * @param \Red61\Via\ApiRequest\apiCustomerEmailVerifiedActiveRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\FinderException 
   * @throws \Red61\Via\Exception\CustomerException 
   * @throws \Red61\Via\Exception\BadCustomerException 
   * @throws \Red61\Via\Exception\CustomerMergedException 
   */
  public function customerEmailVerifiedActive(\Red61\Via\ApiRequest\apiCustomerEmailVerifiedActiveRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Returns a list of the HPH performances
   *
   * @param \Red61\Via\ApiRequest\apiGetHPHPerformancesRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiHPHPerformance[]
   */
  public function getHPHPerformances(\Red61\Via\ApiRequest\apiGetHPHPerformancesRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Fetches basic information for performances that use a given ticket offer (by concession code), between start and
   * end date/time parameters This method will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetOfferPerformancesRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiOfferPerformance[]
   */
  public function getOfferPerformances(\Red61\Via\ApiRequest\apiGetOfferPerformancesRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * This method creates an xml document from which a seating plan can be rendered. It can also return only the seats
   * that have recently changed to facilitate seating plan refreshing
   *
   * @param \Red61\Via\ApiRequest\apiGetSeatingPlanRequest $request
   *
   * @return string
   */
  public function getSeatingPlan(\Red61\Via\ApiRequest\apiGetSeatingPlanRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Sends a confirmation order for the specified order id
   *
   * @param \Red61\Via\ApiRequest\apiSendConfirmationEmailRequest $request
   *
   * @return bool
   */
  public function sendConfirmationEmail(\Red61\Via\ApiRequest\apiSendConfirmationEmailRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Get a transaction summary for the given customer for paid orders.
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerTransactionsSummaryRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiTransactionSummary[]
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function getCustomerTransactionsSummary(\Red61\Via\ApiRequest\apiGetCustomerTransactionsSummaryRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Return All the event and performances which have a digital asset. i.e physical event using eTickets or online
   * content
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerDigitalAssetsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiDigitalEventPurchase[]
   * @throws \Red61\Via\Exception\BadCustomerException 
   * @throws \Red61\Via\Exception\CustomerMergedException 
   */
  public function getCustomerDigitalAssets(\Red61\Via\ApiRequest\apiGetCustomerDigitalAssetsRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Get transaction summary objects for the given customer, between the given dates. Orders not filtered by 'paid'
   * status.
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerTransactionsSummaryByDateRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiTransactionSummary[]
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function getCustomerTransactionsSummaryByDate(\Red61\Via\ApiRequest\apiGetCustomerTransactionsSummaryByDateRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Gets a list of all countries with the ISO 3166 code mapped to the countries name. This method will attempt
   * translation of it's returned objects, where available.
   *
   * @deprecated 
   *
   *
   * @return \Red61\Via\DataObject\ViaApiListElement[]
   */
  public function getCountries()
  {
    $request = new \Red61\Via\ApiRequest\apiGetCountriesRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * This method will attempt translation of it's returned objects, where available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiCountryDetails[]
   */
  public function getCountryDetails()
  {
    $request = new \Red61\Via\ApiRequest\apiGetCountryDetailsRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Sets the affiliate of this customer visit. if the sale get processed, the affiliate code and token gets stored with
   * the order.
   *
   * @param \Red61\Via\ApiRequest\apiSetAffiliateRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function setAffiliate(\Red61\Via\ApiRequest\apiSetAffiliateRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Sets an id which is used to track which campaign was used to initiate the current basket.
   * 
   * campaignTrackingFields is optional and stores any extra information used for tracking.
   *
   * @param \Red61\Via\ApiRequest\apiSetCampaignTrackingIdRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function setCampaignTrackingId(\Red61\Via\ApiRequest\apiSetCampaignTrackingIdRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Remove the donation from the basket
   *
   * @param \Red61\Via\ApiRequest\apiRemoveDonationFromBasketRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function removeDonationFromBasket(\Red61\Via\ApiRequest\apiRemoveDonationFromBasketRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Add a donation of any positive amount to the basket. Use this call to update a donation value.
   *
   * @param \Red61\Via\ApiRequest\apiAddDonationToBasketRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function addDonationToBasket(\Red61\Via\ApiRequest\apiAddDonationToBasketRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Fetch the donation details for a specific order This method will attempt translation of it's returned objects,
   * where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetOrderDonationDetailsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiDonationDetails[]
   */
  public function getOrderDonationDetails(\Red61\Via\ApiRequest\apiGetOrderDonationDetailsRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Returns an array of `ViaApiListElement`s representing Questionnaires that are suitable to be displayed to the
   * customer. OrderId is used to see if any questionnaires are to be fired based on the contents of the order such as a
   * particular event. If the orderId is null then this information is not used.
   *
   * @param \Red61\Via\ApiRequest\apiGetDisplayableQuestionnairesRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiListElement[]
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function getDisplayableQuestionnaires(\Red61\Via\ApiRequest\apiGetDisplayableQuestionnairesRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Lists all the available donation funds. This method will attempt translation of it's returned objects, where
   * available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiDonationFund[]
   */
  public function getDonationFunds()
  {
    $request = new \Red61\Via\ApiRequest\apiGetDonationFundsRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Returns the details of a given Questionnaire. If the customer ID parameter is supplied, all question definitions
   * should include that customer’s answers to any linked questionnaire type. This method will attempt translation of
   * it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetQuestionnaireDetailsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiQuestionnaire
   */
  public function getQuestionnaireDetails(\Red61\Via\ApiRequest\apiGetQuestionnaireDetailsRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Called to indicate that the given customer has declined to answer this questionnaire. This customer wont be
   * prompted with this questionnaire in the future.
   *
   * @param \Red61\Via\ApiRequest\apiDeclineQuestionnaireRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function declineQuestionnaire(\Red61\Via\ApiRequest\apiDeclineQuestionnaireRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Stores the answers of a customer against the given questionnaire.
   *
   * @param \Red61\Via\ApiRequest\apiAnswerQuestionnaireRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function answerQuestionnaire(\Red61\Via\ApiRequest\apiAnswerQuestionnaireRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Updates the given customers custom fields based on the given entries.
   *
   * @param \Red61\Via\ApiRequest\apiUpdateCustomerCustomFieldsRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function updateCustomerCustomFields(\Red61\Via\ApiRequest\apiUpdateCustomerCustomFieldsRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Returns a list of all custom customer fields. If customer id is supplied then the returned array will also contain
   * the values for the given customer. This method will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomCustomerFieldsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiCustomCustomerFieldDetails[]
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function getCustomCustomerFields(\Red61\Via\ApiRequest\apiGetCustomCustomerFieldsRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Returns the cards saved against this customer/
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerSavedCardsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiSavedCardDetails[]
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function getCustomerSavedCards(\Red61\Via\ApiRequest\apiGetCustomerSavedCardsRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Saves a customer's card without having to process an order and to allow the card details be saved for future
   * purchases. Only available if the Payment Provider supports this functionality.
   *
   * @param \Red61\Via\ApiRequest\apiSaveCardDetailsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiSavedCardDetails
   * @throws \Red61\Via\Exception\ViaApiOrderFailException 
   * @throws \Red61\Via\Exception\BadCustomerException 
   */
  public function saveCardDetails(\Red61\Via\ApiRequest\apiSaveCardDetailsRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Returns an array of all Event Tags. This method will attempt translation of it's returned objects, where available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiListElement[]
   */
  public function getEventTags()
  {
    $request = new \Red61\Via\ApiRequest\apiGetEventTagsRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Lists all the voucher types available to purchase. This method will attempt translation of it's returned objects,
   * where available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiVoucherType[]
   */
  public function getVoucherTypes()
  {
    $request = new \Red61\Via\ApiRequest\apiGetVoucherTypesRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Adds a specified quantity of the given voucher type to the basket for purchasing.
   *
   * @param \Red61\Via\ApiRequest\apiAddVoucherTypeToBasketRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active basket.
   * @throws \Red61\Via\Exception\VoucherTypeUnavailableException If the voucher type is off sale, is not available to the be sold on the web or is restrict to a customer and the basket has no customer attached.
   */
  public function addVoucherTypeToBasket(\Red61\Via\ApiRequest\apiAddVoucherTypeToBasketRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Removes the given quantity of the given voucher types from the basket.
   *
   * @param \Red61\Via\ApiRequest\apiRemoveVoucherTypeFromBasketRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function removeVoucherTypeFromBasket(\Red61\Via\ApiRequest\apiRemoveVoucherTypeFromBasketRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Removes all voucher types for purchase from the basket.
   *
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function removeVoucherTypesFromBasket()
  {
    $request = new \Red61\Via\ApiRequest\apiRemoveVoucherTypesFromBasketRequest;
    return $this->api_client->send($request);
  }

  /**
   * Redeems a voucher code against the given basket for the given customer.
   *
   * @param \Red61\Via\ApiRequest\apiRedeemVoucherRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   * @throws \Red61\Via\Exception\InvalidCodeException If there was a problem with the supplied voucher code or voucher linked to the code, see the message for details.
   */
  public function redeemVoucher(\Red61\Via\ApiRequest\apiRedeemVoucherRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Removes the given redeemed voucher from the basket.
   *
   * @param \Red61\Via\ApiRequest\apiRemoveVoucherRedeemFromBasketRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function removeVoucherRedeemFromBasket(\Red61\Via\ApiRequest\apiRemoveVoucherRedeemFromBasketRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Removes all the vouchers redeemed against this basket.
   *
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function removeVoucherRedeemsFromBasket()
  {
    $request = new \Red61\Via\ApiRequest\apiRemoveVoucherRedeemsFromBasketRequest;
    return $this->api_client->send($request);
  }

  /**
   * Returns all the vouchers a customer has redeemed which still have credit. This method will attempt translation of
   * it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerVouchersRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiVoucherDetails[]
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function getCustomerVouchers(\Red61\Via\ApiRequest\apiGetCustomerVouchersRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Lists all the access note categories. This method will attempt translation of it's returned objects, where
   * available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiAccessNoteCategory[]
   * @throws \Red61\Via\Exception\AccessNotesException If there is an issue loading the access note categories
   */
  public function getAccessNoteCategories()
  {
    $request = new \Red61\Via\ApiRequest\apiGetAccessNoteCategoriesRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Lists all the merchandise categories. This method will attempt translation of it's returned objects, where
   * available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiMerchandiseCategory[]
   */
  public function getMerchandiseCategories()
  {
    $request = new \Red61\Via\ApiRequest\apiGetMerchandiseCategoriesRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Returns the merchandise for the given merchandiseRef. This method will attempt translation of it's returned
   * objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetMerchandiseRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiMerchandise
   * @throws \Red61\Via\Exception\MerchandiseUnavailableException If the merchandise is not available to the web, or is disabled
   */
  public function getMerchandise(\Red61\Via\ApiRequest\apiGetMerchandiseRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Lists all the merchandise available to purchase in a particular category. This method will attempt translation of
   * it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetMerchandiseByCategoryRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiMerchandise[]
   */
  public function getMerchandiseByCategory(\Red61\Via\ApiRequest\apiGetMerchandiseByCategoryRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Adds a specified quantity of the given merchandise to the basket for purchasing.
   *
   * @param \Red61\Via\ApiRequest\apiAddMerchandiseToBasketRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   * @throws \Red61\Via\Exception\MerchandiseUnavailableException If the merchandise is not available to the web, or is disabled
   */
  public function addMerchandiseToBasket(\Red61\Via\ApiRequest\apiAddMerchandiseToBasketRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Removes the given quantity of the given merchandise from the basket.
   *
   * @param \Red61\Via\ApiRequest\apiRemoveMerchandiseFromBasketRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\CartNotFoundException If sessionId doesn't reference an active cart.
   */
  public function removeMerchandiseFromBasket(\Red61\Via\ApiRequest\apiRemoveMerchandiseFromBasketRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * This method is used to register/remove an event interest of a customer if @param interested is true/false. NOTE:
   * local events only
   *
   * @param \Red61\Via\ApiRequest\apiRegisterInterestInEventRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function registerInterestInEvent(\Red61\Via\ApiRequest\apiRegisterInterestInEventRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * This method is used to register/remove a performance interest of a customer if @param interested is true/false.
   * NOTE: local events only
   *
   * @param \Red61\Via\ApiRequest\apiRegisterInterestInPerformanceRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function registerInterestInPerformance(\Red61\Via\ApiRequest\apiRegisterInterestInPerformanceRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * This method is used to get customer event interests. This method will attempt translation of it's returned objects,
   * where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerInterestsEventRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiCustomerEventInterest[]
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function getCustomerInterestsEvent(\Red61\Via\ApiRequest\apiGetCustomerInterestsEventRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * This method is used to get customer performance interests. This method will attempt translation of it's returned
   * objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerInterestsPerformanceRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiCustomerPerformanceInterest[]
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function getCustomerInterestsPerformance(\Red61\Via\ApiRequest\apiGetCustomerInterestsPerformanceRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * This method is used to return the current number of seats available for sale to the current user for the selected
   * performance. NOTE: should only be used after consultation with Red61 regarding it's use
   *
   * @param \Red61\Via\ApiRequest\apiGetPerformanceAvailabilityRequest $request
   *
   * @return int
   */
  public function getPerformanceAvailability(\Red61\Via\ApiRequest\apiGetPerformanceAvailabilityRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * This method is used to return the current number of seats available for sale to the current user for the supplied
   * list of performances. NOTE: should only be used after consultation with Red61 regarding it's use
   *
   * @param \Red61\Via\ApiRequest\apiGetPerformanceAvailabilityListRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiListValue[]
   */
  public function getPerformanceAvailabilityList(\Red61\Via\ApiRequest\apiGetPerformanceAvailabilityListRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * This method is used for fetching the current number of seats for sale to the current user for the supplied list of
   * performances, broken down by price band. NOTE: should only be used after consultation with Red61 regarding it's use
   * This method will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetPriceBandAvailabilityRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiPerformanceAvailability
   */
  public function getPriceBandAvailability(\Red61\Via\ApiRequest\apiGetPriceBandAvailabilityRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * This method is used to obtain the VIA version
   *
   *
   * @return string
   */
  public function getVersion()
  {
    $request = new \Red61\Via\ApiRequest\apiGetVersionRequest;
    return $this->api_client->send($request);
  }

  /**
   * This method is used to fetch a unique transaction token for a set of items from a transaction. This token can be
   * used to claim the items by calling `apiClaimFromFriend`
   *
   * @param \Red61\Via\ApiRequest\apiSendToAFriendRequest $request
   *
   * @return string
   */
  public function sendToAFriend(\Red61\Via\ApiRequest\apiSendToAFriendRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * This method is used to list the transaction items that can be claimed using the token generated using
   * `apiSendToAFriend` for the transaction provided. To claim the items for a customer call `apiClaimFromFriend` This
   * method will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiListSentByFriendRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiTicketDetails[]
   * @throws \Red61\Via\Exception\InvalidTokenException if there are no items for this order matching the token, the transaction is invalid or the token is malformed
   */
  public function listSentByFriend(\Red61\Via\ApiRequest\apiListSentByFriendRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * This method is used to claim the items that have had a token generated for them using `apiSendToAFriend`
   *
   * @param \Red61\Via\ApiRequest\apiClaimFromFriendRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\InvalidTokenException 
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function claimFromFriend(\Red61\Via\ApiRequest\apiClaimFromFriendRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * This method is used to fetch the details of the items that have been claimed by this customer, defined by
   * `customerId` using `apiClaimFromFriend` This method will attempt translation of it's returned objects, where
   * available.
   *
   * @param \Red61\Via\ApiRequest\apiGetMyClaimedTicketsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiTicketDetails[]
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function getMyClaimedTickets(\Red61\Via\ApiRequest\apiGetMyClaimedTicketsRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * This method is used to see the details of the items that you about to claim using `apiClaimFromFriend` This method
   * will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetTicketsByClaimTokenRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiTicketDetails[]
   * @throws \Red61\Via\Exception\InvalidTokenException If the token does not match any unclaimed items
   */
  public function getTicketsByClaimToken(\Red61\Via\ApiRequest\apiGetTicketsByClaimTokenRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * This method is used to associate a customer with the current basket. The purpose of this is to bring customer
   * specific ticket offers into scope so they can be applied to items in the basket. An example of this being scheme
   * membership benefits.
   *
   * @param \Red61\Via\ApiRequest\apiSetBasketCustomerRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CartNotFoundException if the basket cannot be found, usually because it has expired
   * @throws \Red61\Via\Exception\BasketChangedException When associating a customer with the current basket the basket offers are refreshed and this may result in different offers being applied to the items in the basket. If these new offers mean that the basket total has changed this exception will be thrown. It is advisable to refresh the basket contents if receiving this exception to display the new offers on the items in the basket.
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function setBasketCustomer(\Red61\Via\ApiRequest\apiSetBasketCustomerRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * This method is used to add a promocode to the current basket. The purpose of this is to bring promocode specific
   * ticket offers into scope so that they can be applied to the basket.
   *
   * @param \Red61\Via\ApiRequest\apiAddPromocodeToBasketRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CartNotFoundException if the basket cannot be found, usually because it has expired.
   * @throws \Red61\Via\Exception\BasketChangedException When adding a promo code to the current basket the basket's offers are refreshed and this may result in different offers being applied to the items in the basket. If these new offers mean that the basket total has changed this exception will be thrown. It is advisable to refresh the basket contents if receiving this exception to display the new offers on the items in the basket.
   */
  public function addPromocodeToBasket(\Red61\Via\ApiRequest\apiAddPromocodeToBasketRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * This method is used to stop a customer's card from a previous payment from being included in the list of repeat
   * payments for a customer, see `apiGetCustomerSavedCards` If this call is successful then calling
   * `apiGetCustomerSavedCards` should no longer contain the repeat payment identified by `repeatPaymentToken`
   *
   * @param \Red61\Via\ApiRequest\apiRemoveRepeatabilityCustomerSavedCardRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\IllegalArgumentException if the arguments passed in are null or empty
   * @throws \Red61\Via\Exception\EJBException if the system is unable to find the card record linked to the repeat payment token or the card record linked to the repeat token does not match the `customerId` argument
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function removeRepeatabilityCustomerSavedCard(\Red61\Via\ApiRequest\apiRemoveRepeatabilityCustomerSavedCardRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Lists performances for the supplied event within the given date-time range. Note: If the web user does not have
   * permission to list past performances then the start date supplied should be greater than the current date and time
   * minus the allowance. Otherwise the start date will be set to the current date and time minus the allowance. This
   * method will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetPerformancesByDateRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiPerformance[]
   */
  public function getPerformancesByDate(\Red61\Via\ApiRequest\apiGetPerformancesByDateRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Lists performances based on the supplied parameters.Returned list is ordered by ascending performance date.`typeid`
   * and `subtypeid` must be used in conjunction with other parameters (otherwise no results are returned). This method
   * will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetPerformancesByTimeRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiEventPerformance[]
   */
  public function getPerformancesByTime(\Red61\Via\ApiRequest\apiGetPerformancesByTimeRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Method lists companies for which the customer has not yet decided mailing permission. Note: This method requires
   * the customerId to be set against the basket as well, otherwise an empty list will be returned.
   * 
   * For updating the customers company mail permissions see `apiUpdateCustomerCompanyMailPermissions`
   *
   * @deprecated in favour of `apiGetCustomerUndecidedMailPermissionCompanies`
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerCompanyDetailsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiCompany[]
   * @throws \Red61\Via\Exception\CartNotFoundException If the basket cannot be found, usually because it has expired.
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   * @throws \Red61\Via\Exception\EJBException If the given customerId does not match the customer currently assigned to the basket.
   * @throws \Red61\Via\Exception\IllegalArgumentException If the given customer cannot be loaded.
   */
  public function getCustomerCompanyDetails(\Red61\Via\ApiRequest\apiGetCustomerCompanyDetailsRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Method lists companies for which the customer has not yet decided mailing permission. Note: This method requires
   * the customerId to be set against the basket as well, otherwise an empty list will be returned.
   * 
   * For updating the customers company mail permissions see `apiUpdateCustomerCompanyMailPermissions`
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerUndecidedMailPermissionCompaniesRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiCompany[]
   * @throws \Red61\Via\Exception\CartNotFoundException If the basket cannot be found, usually because it has expired.
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   * @throws \Red61\Via\Exception\EJBException If the given customerId does not match the customer currently assigned to the basket.
   * @throws \Red61\Via\Exception\IllegalArgumentException If the given customer cannot be loaded.
   */
  public function getCustomerUndecidedMailPermissionCompanies(\Red61\Via\ApiRequest\apiGetCustomerUndecidedMailPermissionCompaniesRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * This method is used to update the customer's mailing permissions to include the new responses given.
   * 
   * For retrieving the list of companies that the customer has yet to be asked about see
   * `apiGetCustomerUndecidedMailPermissionCompanies`
   *
   * @param \Red61\Via\ApiRequest\apiUpdateCustomerCompanyMailPermissionsRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   * @throws \Red61\Via\Exception\EJBException If there is an issue updating the customer's account to contain the new mailing permissions.
   * @throws \Red61\Via\Exception\IllegalArgumentException If the given customer cannot be loaded, or invalid or duplicate responses were passed in
   */
  public function updateCustomerCompanyMailPermissions(\Red61\Via\ApiRequest\apiUpdateCustomerCompanyMailPermissionsRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * This method is used to retrieve the Card Type of a given card using its BIN
   *
   * @param \Red61\Via\ApiRequest\apiGetCardTypeFromBINRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiCardType
   */
  public function getCardTypeFromBIN(\Red61\Via\ApiRequest\apiGetCardTypeFromBINRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * This method is used to retrieve details about the schemes in a given order. This method will attempt translation of
   * it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetOrderSchemeDetailsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiSchemeTierDetails[]
   */
  public function getOrderSchemeDetails(\Red61\Via\ApiRequest\apiGetOrderSchemeDetailsRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * This method is used to retrieve the instance's offset from the end of the day.
   *
   *
   * @return int
   */
  public function getDayChangeOffset()
  {
    $request = new \Red61\Via\ApiRequest\apiGetDayChangeOffsetRequest;
    return $this->api_client->send($request);
  }

  /**
   * This method is used to extend the life of a customer's basket by a further 30 minutes in the event that they are
   * not finished shopping but have also not added anything to the basket within the last 30 minutes.
   *
   *
   * @return void
   * @throws \Red61\Via\Exception\CartNotFoundException If the basket cannot be found, usually because it has expired.
   */
  public function basketKeepAlive()
  {
    $request = new \Red61\Via\ApiRequest\apiBasketKeepAliveRequest;
    $this->api_client->send($request);
  }

  /**
   * Add a category to the given customer.
   *
   * @param \Red61\Via\ApiRequest\apiAddCustomerCategoryRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function addCustomerCategory(\Red61\Via\ApiRequest\apiAddCustomerCategoryRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Remove a category from the given customer.
   *
   * @param \Red61\Via\ApiRequest\apiRemoveCustomerCategoryRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function removeCustomerCategory(\Red61\Via\ApiRequest\apiRemoveCustomerCategoryRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Returns the valid and current subscriptions for the given customer. This method will attempt translation of it's
   * returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerSubscriptionDetailsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiSubscriptionDetails[]
   * @throws \Red61\Via\Exception\CustomerMergedException If the account for the given customerId has been merged into another customer account.
   * @throws \Red61\Via\Exception\BadCustomerException If the account for the given customerId is no longer active.
   */
  public function getCustomerSubscriptionDetails(\Red61\Via\ApiRequest\apiGetCustomerSubscriptionDetailsRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * This method is used to retrieve information on the gift aid declarations that this customer has made.
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerGiftAidDeclarationsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiGiftAidDeclarationDetails[]
   */
  public function getCustomerGiftAidDeclarations(\Red61\Via\ApiRequest\apiGetCustomerGiftAidDeclarationsRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * This method is used to retrieve all information on the gift aid declaration for the given customer and gift aid
   * declaration Id.
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerGiftAidDeclarationRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiGiftAidDeclarationFullDetails
   */
  public function getCustomerGiftAidDeclaration(\Red61\Via\ApiRequest\apiGetCustomerGiftAidDeclarationRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Finds the order ID for any order created by the given session ID, if one exists
   *
   * @param \Red61\Via\ApiRequest\apiGetOrderIdForSessionIdRequest $request
   *
   * @return string
   */
  public function getOrderIdForSessionId(\Red61\Via\ApiRequest\apiGetOrderIdForSessionIdRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * This method is used to determine whether or not the underlying payment processor supports saving the customers card
   * information for ease of future purchases.
   *
   *
   * @return bool
   */
  public function doesProcessorSupportRepeatPayments()
  {
    $request = new \Red61\Via\ApiRequest\apiDoesProcessorSupportRepeatPaymentsRequest;
    return $this->api_client->send($request);
  }

  /**
   * This method will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetUserSimilarEventsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiEvent[]
   * @throws \Red61\Via\Exception\RecommendationException if there was an issue finding recommended items
   */
  public function getUserSimilarEvents(\Red61\Via\ApiRequest\apiGetUserSimilarEventsRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * This method is used to retrieve a `ViaApiSchemeUsageDetails` object containing information on the number and
   * details of discounted tickets for the customer's subscription
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerSchemeUsagesRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiSchemeUsageDetails
   */
  public function getCustomerSchemeUsages(\Red61\Via\ApiRequest\apiGetCustomerSchemeUsagesRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Get all the currently available delivery options for a basket, and the currently selected delivery option.
   * 
   * If no selection has been made, then the default selection for the web user for the given webKey will be returned
   * This method will attempt translation of it's returned objects, where available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiBasketDeliveryOptions
   * @throws \Red61\Via\Exception\CartNotFoundException 
   */
  public function getDeliveryOptions()
  {
    $request = new \Red61\Via\ApiRequest\apiGetDeliveryOptionsRequest;
    return $this->api_client->send($request);
  }

  /**
   * Set the delivery option, and the selected delivery fees for each given item type
   *
   * @param \Red61\Via\ApiRequest\apiSetDeliveryOptionRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CartNotFoundException 
   * @throws \Red61\Via\Exception\UnavailablePrintOptionException if the `deliverOptionId` cannot be set on this basket as it is not available. For example POST is not available if the date of the earliest performance in the basket is before the the post cut off. ETICKET is not available unless all performances in the basket are in sub venues that support e-ticket.
   * @throws \Red61\Via\Exception\IllegalArgumentException If the given delivery option does not exist, or the fees do not match the delivery option
   */
  public function setDeliveryOption(\Red61\Via\ApiRequest\apiSetDeliveryOptionRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Attaches a note of given type to this basket that will be saved against the order
   *
   * @param \Red61\Via\ApiRequest\apiSetBasketOrderNoteRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CartNotFoundException 
   */
  public function setBasketOrderNote(\Red61\Via\ApiRequest\apiSetBasketOrderNoteRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Returns the current notes attached to this basket that will be saved against the order
   *
   *
   * @return \Red61\Via\DataObject\ViaApiListElement[]
   * @throws \Red61\Via\Exception\CartNotFoundException 
   */
  public function getBasketOrderNotes()
  {
    $request = new \Red61\Via\ApiRequest\apiGetBasketOrderNotesRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Refunds the selected complimentary items (i.e. free tickets) from the given order.
   * 
   * Can not be used to refund anything that cost money.
   *
   * @param \Red61\Via\ApiRequest\apiRefundCompItemsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiOrderDetails
   */
  public function refundCompItems(\Red61\Via\ApiRequest\apiRefundCompItemsRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Adds a transaction note of the specified type to the transaction for the given Id.
   *
   * @param \Red61\Via\ApiRequest\apiAddOrderNoteRequest $request
   *
   * @return void
   */
  public function addOrderNote(\Red61\Via\ApiRequest\apiAddOrderNoteRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Marks the voucher for the specified order and line item as "gifted"
   *
   * @param \Red61\Via\ApiRequest\apiMarkVoucherAsGiftedRequest $request
   *
   * @return void
   */
  public function markVoucherAsGifted(\Red61\Via\ApiRequest\apiMarkVoucherAsGiftedRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Returns a List of vouchers for the given transaction id, the voucher details will contain additional information on
   * the voucherCode, giftedStatus, description and initalValue of each voucher. This method will attempt translation of
   * it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetVoucherGiftDetailsForOrderRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiVoucherGiftDetails[]
   */
  public function getVoucherGiftDetailsForOrder(\Red61\Via\ApiRequest\apiGetVoucherGiftDetailsForOrderRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Fetch all global data protection consent questions to ask a new customer at customer creation This method will
   * attempt translation of it's returned objects, where available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiConsentQuestion[]
   * @throws \Red61\Via\Exception\CustomerMergedException If the current customer account has been merged into another account
   * @throws \Red61\Via\Exception\BadCustomerException If the customer account is inactive
   */
  public function getConsentQuestionsForNewCustomer()
  {
    $request = new \Red61\Via\ApiRequest\apiGetConsentQuestionsForNewCustomerRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Fetch any unanswered, global data protection consent questions to ask an existing customer This method will attempt
   * translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetConsentQuestionsForExistingCustomerRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiConsentQuestion[]
   * @throws \Red61\Via\Exception\CustomerMergedException If the current customer account has been merged into another account
   * @throws \Red61\Via\Exception\BadCustomerException If the customer account is inactive
   */
  public function getConsentQuestionsForExistingCustomer(\Red61\Via\ApiRequest\apiGetConsentQuestionsForExistingCustomerRequest $request)
  {
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Fetch any unanswered data protection consent questions for the basket's customer, including any questions specific
   * to the basket contents This method will attempt translation of it's returned objects, where available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiConsentQuestions
   * @throws \Red61\Via\Exception\CartNotFoundException If `sessionId` does not reference an active basket.
   * @throws \Red61\Via\Exception\CustomerMergedException If the current customer account has been merged into another account
   * @throws \Red61\Via\Exception\BadCustomerException If the customer account is inactive
   */
  public function getConsentQuestionsForBasket()
  {
    $request = new \Red61\Via\ApiRequest\apiGetConsentQuestionsForBasketRequest;
    return $this->api_client->send($request);
  }

  /**
   * Fetch all answered data protection consent questions for a customer, with their given answers. Optionally return
   * any additional un-answered questions for this customer. This method will attempt translation of it's returned
   * objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerConsentQuestionsRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiCustomerConsentQuestions
   * @throws \Red61\Via\Exception\CustomerMergedException If the current customer account has been merged into another account
   * @throws \Red61\Via\Exception\BadCustomerException If the customer account is inactive
   */
  public function getCustomerConsentQuestions(\Red61\Via\ApiRequest\apiGetCustomerConsentQuestionsRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Save new answers to data protection consent questions, for the given customer.
   *
   * @param \Red61\Via\ApiRequest\apiSetCustomerConsentAnswersRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CustomerMergedException If the current customer account has been merged into another account
   * @throws \Red61\Via\Exception\BadCustomerException If the customer account is inactive
   */
  public function setCustomerConsentAnswers(\Red61\Via\ApiRequest\apiSetCustomerConsentAnswersRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Converts tickets within a reservation. If the `itemIds` List is empty or `null` the system will convert all
   * remaining active tickets within the reservation. An active ticket is one that has not been released or already
   * converted. Converting a reservation will take the specified tickets and add them to the basket defined by
   * `sessionId`. Converting **IS POSSIBLE** if this basket already has items in it. The `convertingBehaviour` controls
   * how the system handles the ticket offers present against the tickets being converted. This has 3 values:
   * 
   * - **disregard** : all tickets will be added at full price
   * - **available** : all tickets will be added with best ticket offers currently available
   * - **forced** : all tickets will be added with the ticket offer reserved at the time of the reservation, if still
   * possible. If not it will fail
   * 
   * Different `convertingBehaviour` values will impose different restrictions on the converting process, with `forced`
   * being the most restrictive.
   * 
   * **WARNING:** The active basket will have its customer set to the customer of the reservation.
   *
   * @param \Red61\Via\ApiRequest\apiConvertReservationRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\CartNotFoundException If `sessionId` does not reference an active basket.
   * @throws \Red61\Via\Exception\CustomerMergedException If the current customer account has been merged into another account
   * @throws \Red61\Via\Exception\BadCustomerException If the customer account is inactive
   * @throws \Red61\Via\Exception\FailedConvertException If there was a problem with the chosen `convertingBehaviour` when adding the active items from the reservation to the basket. If this is triggered the original reservation will be unchanged
   * @throws \Red61\Via\Exception\AddSeatsException If there was a problem adding the active items from the reservation to the basket. If this is triggered the original reservation will be unchanged
   * @throws \Red61\Via\Exception\FailedRevertException If there was any problem with the basket calls required for converting the system will revert the convert of the reservation. This error is thrown in the rare case that this revert fails
   */
  public function convertReservation(\Red61\Via\ApiRequest\apiConvertReservationRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Releases tickets within a reservation. If the `itemIds` List is empty or `null` the system will release all
   * remaining active tickets within the reservation. An active ticket is one that has not been released or already
   * converted.
   *
   * @param \Red61\Via\ApiRequest\apiReleaseTicketsFromReservationRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\FailedReleaseException If an error occurred releasing the tickets from the reservation. See the message for details
   */
  public function releaseTicketsFromReservation(\Red61\Via\ApiRequest\apiReleaseTicketsFromReservationRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Generates a password reset token and sends a customer password reset email to the supplied email address
   *
   * @param \Red61\Via\ApiRequest\apiSendPasswordResetEmailRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\PasswordResetException - if we fail to find the customer account for the given email or if there was an issue sending the email
   * @throws \Red61\Via\Exception\CustomerMergedException If the current customer account has been merged into another account
   * @throws \Red61\Via\Exception\BadCustomerException - If the customer account is inactive.
   */
  public function sendPasswordResetEmail(\Red61\Via\ApiRequest\apiSendPasswordResetEmailRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Generates a password reset token and sends a customer password reset email to the email address for the given
   * customerId
   *
   * @param \Red61\Via\ApiRequest\apiSendPasswordResetEmailByCustomerIdRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\PasswordResetException - if we fail to find the customer account for the given email or if there was an issue sending the email
   * @throws \Red61\Via\Exception\FinderException - If the no customer matches the given customerId.
   * @throws \Red61\Via\Exception\BadCustomerException - If the customer account is inactive.
   * @throws \Red61\Via\Exception\CustomerMergedException If the current customer account has been merged into another account
   */
  public function sendPasswordResetEmailByCustomerId(\Red61\Via\ApiRequest\apiSendPasswordResetEmailByCustomerIdRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Finds the customer for the given token, then updates their password to the one supplied.
   *
   * @param \Red61\Via\ApiRequest\apiUpdateCustomerPasswordRequest $request
   *
   * @return void
   * @throws \Red61\Via\Exception\PasswordResetException - if there was an issue finding the customer for the supplied token
   * @throws \Red61\Via\Exception\CustomerMergedException If the current customer account has been merged into another account
   * @throws \Red61\Via\Exception\BadCustomerException - If the customer account is inactive.
   */
  public function updateCustomerPassword(\Red61\Via\ApiRequest\apiUpdateCustomerPasswordRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Method checks if the customer password reset token is valid and has not expired
   *
   * @param \Red61\Via\ApiRequest\apiCheckPasswordResetTokenValidityRequest $request
   *
   * @return bool
   */
  public function checkPasswordResetTokenValidity(\Red61\Via\ApiRequest\apiCheckPasswordResetTokenValidityRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Gets the all the (non hidden) `ViaApiAgeSuitabilityDetails` in the system. This method will attempt translation of
   * it's returned objects, where available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiAgeSuitabilityDetails[]
   */
  public function getEventAgeSuitabilities()
  {
    $request = new \Red61\Via\ApiRequest\apiGetEventAgeSuitabilitiesRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Method Lists all accreditation types to which the API user has access. This method will attempt translation of it's
   * returned objects, where available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiAccreditationType[]
   */
  public function getAccreditationTypes()
  {
    $request = new \Red61\Via\ApiRequest\apiGetAccreditationTypesRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Method returns the accreditation that corresponds to the supplied customer and accreditation type ID. This method
   * will attempt translation of it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetAccreditationByCustomerAndTypeRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiAccreditation
   */
  public function getAccreditationByCustomerAndType(\Red61\Via\ApiRequest\apiGetAccreditationByCustomerAndTypeRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Method returns a questionnaire response with the provided questionnaire and customer IDs.
   *
   * @param \Red61\Via\ApiRequest\apiGetQuestionnaireResponseByQuestionnaireAndCustomerRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiQuestionnaireResponse
   */
  public function getQuestionnaireResponseByQuestionnaireAndCustomer(\Red61\Via\ApiRequest\apiGetQuestionnaireResponseByQuestionnaireAndCustomerRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Method submits an accreditation for a customer. If the ‘draft’ parameter is set to true, the accreditation
   * should be created with a status of ‘INCOMPLETE’ and it should skip the validation of the questionnaire
   * response. Otherwise, the accreditation should be created with a status of ‘PENDING’ and questionnaire
   * validation should occur as usual.
   *
   * @param \Red61\Via\ApiRequest\apiSubmitAccreditationRequest $request
   *
   * @return bool
   * @throws \Red61\Via\Exception\QuestionnaireValidationException - if there is an issue validating the questionnaire response
   */
  public function submitAccreditation(\Red61\Via\ApiRequest\apiSubmitAccreditationRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Method requests an AWS pre-signed URL and a media ID from the back end. The front end then uploads the image to the
   * AWS URL and any subsequent requests to retrieve that image with the media ID should respond with the URL to fetch
   * it from AWS.
   *
   * @param \Red61\Via\ApiRequest\apiCreateMediaRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiCreatedMedia
   */
  public function createMedia(\Red61\Via\ApiRequest\apiCreateMediaRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Method requests the AWS URL for a media object given its ID.
   *
   * @param \Red61\Via\ApiRequest\apiGetMediaRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiMedia
   */
  public function getMedia(\Red61\Via\ApiRequest\apiGetMediaRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * Returns an array of all Presentation Formats. This method will attempt translation of it's returned objects, where
   * available.
   *
   *
   * @return \Red61\Via\DataObject\ViaApiListElement[]
   */
  public function getPresentationFormats()
  {
    $request = new \Red61\Via\ApiRequest\apiGetPresentationFormatsRequest;
    // Wrapped in ternary to ensure even an empty result is iterable
    return ($this->api_client->send($request) ? : array());
  }

  /**
   * Create a customerJWT to "sudo" to this customer within the REST API
   *
   * @param \Red61\Via\ApiRequest\apiGetCustomerJWTRequest $request
   *
   * @return string
   */
  public function getCustomerJWT(\Red61\Via\ApiRequest\apiGetCustomerJWTRequest $request)
  {
    return $this->api_client->send($request);
  }

  /**
   * This method takes a list of tickets identified by `ViaApiItemRefundExemptRequest#itemId` within the transaction and
   * marks them as refund exempt or not depending on `ViaApiItemRefundExemptRequest#refundExemptCode` value.
   * 
   * NOTE this will **override** whatever refund exempt code values were previously stored against these tickets.
   *
   * @param \Red61\Via\ApiRequest\apiSetRefundExemptRequest $request
   *
   * @return void
   */
  public function setRefundExempt(\Red61\Via\ApiRequest\apiSetRefundExemptRequest $request)
  {
    $this->api_client->send($request);
  }

  /**
   * Method Fetches the information for the fund for the specified reference This method will attempt translation of
   * it's returned objects, where available.
   *
   * @param \Red61\Via\ApiRequest\apiGetDonationFundRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiDonationFund
   */
  public function getDonationFund(\Red61\Via\ApiRequest\apiGetDonationFundRequest $request)
  {
    return $this->api_client->send($request);
  }
  /** 
   * This method contacts the payment provider and creates the customers agreed payment schedule for the supplied
   * donation type. The returned `ViaApiCardPaymentRequest#paymentFormHTML` should be displayed to the customer if
   * provided as it allows for initial payment to be captured if required.
   *
   * @param \Red61\Via\ApiRequest\apiCreateDonationScheduleRequest $request
   *
   * @return \Red61\Via\DataObject\ViaApiCardPaymentRequest
   */
  public function createDonationSchedule(\Red61\Via\ApiRequest\apiCreateDonationScheduleRequest $request)
  {
    return $this->api_client->send($request);
  }


}
