<?php
/**
 * Defines Red61\Via\SessionStorage\PhpNativeSessionStorage
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */

namespace Red61\Via\SessionStorage;

/**
 * The simplest session storage engine, this class uses PHP's native session handling to save data in the
 * superglobal $_SESSION array.
 *
 * [!!] Concurrency management
 * PHP's native file sessions hold a filesystem lock on the session for the duration of each request and queues any
 * other requests with the same session id so that they execute in sequence. This makes them concurrency-safe, but can
 * provide poor performance especially if your site has AJAX or dynamic image/asset compilation and thumbnailing.
 *
 * If you are using any other session handler for the native sesions, you MUST ensure that overlapping requests cannot
 * cause a user to lose their basket after it has been assigned, or to keep their basket ID after it has been destroyed (eg during checkout)
 *
 * @package Red61\Via\SessionStorage
 * @see     spec\Red61\Via\SessionStorage\PhpNativeSessionStorageSpec
 */
class PhpNativeSessionStorage implements ViaSessionStorage
{

	/**
	 * @var string key in $_SESSION to use
	 */
	protected $session_key;

	/**
	 * @param string $session_key optional, otherwise stores in $_SESSION['via_basket_id']
	 */
	public function __construct($session_key = 'via_basket_id')
	{
		$this->session_key = $session_key;
	}

	/**
	 * {@inheritdoc}
	 * @throws \BadMethodCallException if session has not been started yet
	 */
	public function loadBasketId()
	{
		$this->throwIfNoSession();

		if (isset($_SESSION[$this->session_key])) {
			return $_SESSION[$this->session_key];
		} else {
			return NULL;
		}
	}

	/**
	 * @throws \BadMethodCallException if the session is not started
	 */
	protected function throwIfNoSession()
	{
		if ( ! session_id()) {
			throw new \BadMethodCallException("Session was not started");
		}
	}

	/**
	 * {@inheritdoc}
	 * @throws \BadMethodCallException if session has not been started yet
	 */
	public function saveBasketId($basket_id)
	{
		$this->throwIfNoSession();
		$_SESSION[$this->session_key] = $basket_id;
	}
}
