<?php
/**
 * Runs basic smoke tests to ensure that applications implemented against the legacy red61_via classes work
 * as expected with the new wrapper.
 *
 * Run it like this:
 *
 *   VIA_WSDL="https://wherever?wsdl" VIA_WEBKEY="5926:your:key:here" php legacy_wrapper_compatability.php
 *
 * If successful it will exit 0. It is deliberately designed as an overall pass/fail script to avoid any
 * external dependencies (which would also introduce the possibility that new-style autoloaders and config
 * etc had made it into global state).
 *
 * @author    Andrew Coulton <andrew@ingenerator.com>
 * @copyright 2014 Red61 Ltd
 * @licence   proprietary
 */
use Red61\Via\ApiRequest\apiUpdateCustomerCustomFieldsRequest;
use Red61\Via\Exception\CartNotFoundException;
use Red61\Via\Exception\ViaException;
use Red61\Via\Exception\ViaSoapFault;

error_reporting(E_ALL);
do_assert(isset($_SERVER['VIA_WSDL']), 'You must define the VIA_WSDL environment variable');
do_assert(isset($_SERVER['VIA_WEBKEY']), 'You must define the VIA_WEBKEY environment variable');
do_assert(isset($_SERVER['MEMCACHE_HOST']), 'You must define the MEMCACHE_HOST environment variable');

define('VIA_TEST_CARD_NUMBER', '5418590012345679');
define('VIA_TEST_CARD_CVV',    '123');
define('VIA_TEST_CARD_START',  '10/12');
define('VIA_TEST_CARD_EXPIRY', '02/'.(date('y') + 1));
define('VIA_TEST_CARD_ISSUE',  '123');

require(__DIR__.'/../red61_via.php');


$_SERVER['REMOTE_ADDR'] = '127.0.0.1';

print 'Connecting to '.$_SERVER['VIA_WSDL'].' with profiling active'.\PHP_EOL;
$via = new red61_via($_SERVER['VIA_WSDL'], $_SERVER['VIA_WEBKEY'], NULL, array('cache' => TRUE, 'profile' => TRUE));

/*-----------------------------------------------------------------------------------------------
 * GENERAL SMOKE TESTS
 * Just interact with VIA a bit and fetch some data, checking we can access some properties on
 * the returned results.
 * ----------------------------------------------------------------------------------------------
 */
do_assert($customer_categories = $via->getCustomerCategories(0), 'Has customer categories for customer 0');
do_assert('\\'.get_class($customer_categories[0]) === Red61\Via\ViaClassmap::$dataClassmap['viaApiCustomerCategory'], 'Classes have been mapped to the new types');
do_assert($customer_categories[0]->code !== "", 'Categories have a code property');

do_assert($venues = $via->getVenues(), "Got venues");
$venue_details = $via->getVenueDetails($venues[0]->id);
do_assert(is_string($venue_details->title), "First venue's details have a title");
do_assert($subvenues = $via->getSubVenues(), "Got subvenues with no filter");
do_assert($via->getEvents(), "Got array of events");

do_assert($via->getEventTypes(), 'Got event types');
do_assert($via->getArtists(), 'Got artists');

/* --------------------------------------------------------------------------------------------
 * ADDRESS SEARCH SMOKE TESTS
 * --------------------------------------------------------------------------------------------
 */
try {
	do_assert($address_results = $via->searchAddress('8', '', '', '', '', 'EH4 2JX'), 'Can search for an address');
	do_assert(preg_match('/Edinburgh/', $address_results[0]->summary), 'Address results have a summary description');
	do_assert($via->getAddress($address_results[0]->reference)->postcode, 'Can get full address for search result');
} catch (ViaException $e) {
	if (preg_match('/daily limit exceeded|account out of credit/i', $e->getMessage())) {
		trigger_error('Could not test address search because the key limit has been reached', E_USER_NOTICE);
	} else {
		throw $e;
	}
}

/*----------------------------------------------------------------------------------------------
 * NO BASKET SMOKE TESTS
 * ----------------------------------------------------------------------------------------------
 */
do_assert($via->getNoOfItemsInBasket() === 0, 'Returns 0 for number of items when no basket');
do_assert($via->getBasketTotal() == 0, 'Returns 0 for basket total when no basket');
do_assert($via->getBasketSubTotal() == 0, 'Returns 0 for basket subtotal when no basket');


/*----------------------------------------------------------------------------------------------
 * CUSTOMER RECORD SMOKE TESTS
 * 1. Create a customer
 * ---------------------------------------------------------------------------------------------
 */
$cust_email = 'test+via.' . uniqid() . '@ingenerator.com';
$cust_forename = uniqid();
$cust_surname = uniqid();
$cust_id = $via->createAccount(TRUE, 'Mr', $cust_forename, $cust_surname, '75 Trafalgar Lane', '', '', 'Edinburgh', '', 'EH6 4DQ',
	'UK', '', '', '', $cust_email, '12345678', '12345678', 'any', 'FALSE', FALSE, FALSE, 0);
do_assert($cust_id, "Returns customer ID from creating record");

/*----------------------------------------------------------------------------------------------
 * CUSTOMER RECORD SMOKE TESTS
 * 2. Get and set their categories and create a gift aid declaration
 * ---------------------------------------------------------------------------------------------
 */
$via->setCustomerCategories($cust_id, array($customer_categories[0]->id));
$customer_categories = $via->getCustomerCategories($cust_id);
do_assert($customer_categories[0]->active, "Customer $cust_id category 0 can be set");

$via->createGiftAidDeclaration($cust_id);

/*----------------------------------------------------------------------------------------------
 * CUSTOMER RECORD SMOKE TESTS
 * 3. Login as customer and reset password
 * ---------------------------------------------------------------------------------------------
 */
do_assert($via->login($cust_email, '12345678') === $cust_id, 'Returns customer ID from logging in');
do_assert($new_pw = $via->customerPasswordReset($cust_email), 'Can reset password');
do_assert($via->login($cust_email, $new_pw), 'Can login with reset password');

/*----------------------------------------------------------------------------------------------
 * CUSTOMER RECORD SMOKE TESTS
 * 4. Get customer details
 * ---------------------------------------------------------------------------------------------
 */
$cust_details = $via->getCustomerDetails($cust_id);
do_assert($cust_details->street1 === '75 Trafalgar Lane', 'Can get customer details');
do_assert($via->getCustomerOtherDetails($cust_details)->email === $cust_email, 'Can get customer other details');
do_assert($via->getCustomerName($cust_id) === "Mr $cust_forename $cust_surname", 'Can get just customer name');

/*----------------------------------------------------------------------------------------------
 * CUSTOMER RECORD SMOKE TESTS
 * 5. Get and set custom customer fields
 * ---------------------------------------------------------------------------------------------
 */
do_assert($custom_fields = $via->getCustomCustomerFields($cust_id), 'Can get customer custom fields');

// We can't be sure what's in the test data
switch ($custom_fields[0]->type) {
	case 'CHOICE':
		$custom_fields[0]->selectedChoice = $custom_fields[0]->choices[0]->choiceId;
	break;

	case 'INTEGER':
	case 'DOUBLE':
		$custom_fields[0]->entry = 9;
	break;

	case 'TEXT':
	case 'TEXTAREA':
	default:
		$custom_fields[0]->entry = 'Some test text';
	break;
}
do_assert(TRUE, 'Can set new value for custom field type '.$custom_fields[0]->type);

$via->updateCustomersCustomFields($cust_id, array($custom_fields[0]));
$new_custom_fields = $via->getCustomCustomerFields($cust_id);

switch ($custom_fields[0]->type) {
	case 'CHOICE':
		do_assert($new_custom_fields[0]->selectedChoice == $custom_fields[0]->selectedChoice, 'Can update custom fields');
	break;

	case 'INTEGER':
	case 'DOUBLE':
	case 'TEXT':
	case 'TEXTAREA':
	default:
		do_assert($new_custom_fields[0]->entry == $custom_fields[0]->entry, 'Can update custom fields');
	break;
}

/*-----------------------------------------------------------------------------------------------
 * PROFILER SMOKE TESTS
 * Check that the profiler has captured stats for the last few tests
 * ----------------------------------------------------------------------------------------------
 */
$stats = $via->getProfileStats(FALSE);
do_assert($stats['success'][apiUpdateCustomerCustomFieldsRequest::SOAP_METHOD_NAME]['count'], 'Counted the update customer fields requests');

$stats = $via->getProfileStats();
do_assert($stats[apiUpdateCustomerCustomFieldsRequest::SOAP_METHOD_NAME]['count'], 'Only returns success stats (not nested) by default');

/*-----------------------------------------------------------------------------------------------
 * BOOKING AND PURCHASE SMOKE TESTS
 * 0. Queue deferred basket metadata setting requests (can't verify these were actually sent, but can verify they don't throw)
 * ----------------------------------------------------------------------------------------------
 */
$via->setAffiliate('TEST-1', 'TEST-2');
$via->setCampaignTrackingId(1, 'foo=bar');

/*-----------------------------------------------------------------------------------------------
 * BOOKING AND PURCHASE SMOKE TESTS
 * 1. Find a bookable event and put some tickets in the basket
 * ----------------------------------------------------------------------------------------------
 */
$tomorrow = new \DateTime('tomorrow');
$future_events = $via->eventDetailedSearch('', '', '', $tomorrow->format('Y-m-d'), '2030-01-10', '', '', '');
do_assert($future_events, 'Has some future events from event detailed search');
$perf_to_book = NULL; $event = NULL; $perf_prices = NULL;

// Have a few goes - there's no guarantee events we find on QA will be on sale etc
try_times(10, function () use ($via, $future_events, & $event, & $perf_to_book, & $perf_prices) {
	$event = $future_events[rand(0, count($future_events) - 1)];
	do_assert($perfs = $via->getPerformances($event->id), 'Can get performances for '.$event->title);
	do_assert($via->groupPerformancesByDate($perfs), 'No errors when grouping perfs by month and day');

	$perf_to_book = $perfs[0];
	do_assert($perf_to_book->status === 'On Sale', $event->title.' has an onsale performance');
	do_assert($via->getPerformanceDetails($perf_to_book->id), 'Can get details for individual performance');
	do_assert($perf_prices = $via->getPerformancePrices($perf_to_book->id), 'Perf '.$perf_to_book->id.' has prices');
	do_assert($perf_prices[0]->seats, 'Has some seats in first priceband');
	do_assert($perf_prices[0]->price > 0, 'Is not free (need to have a value for later basket total tests');
});

do_assert($via->getEventDetails($event->id), "Can get individual event details by id");
do_assert(is_array($via->getDonationsForEvent($event->id)), "Can get donation funds linked to event as array");


$quantities = build_ticket_quantities_for_event($perf_prices);
do_assert($via->addTicketsToBasket($quantities, $perf_to_book->id), 'Can add tickets to basket');
do_assert($via->getBasketItemsSortedByVenue(), 'Can get basket items sorted by venue');

do_assert($flat_basket = $via->getBasketTicketItemsFlat(), 'Can get flat basket with legacy method');
do_assert($flat_basket[0] instanceof \Red61\Via\DataObject\ViaApiProductDetails, 'Flat basket returns ticket as top array element');
do_assert($flat_basket[0]->event instanceof \Red61\Via\DataObject\ViaApiBasketEventItem, 'Flat basket has event as property of each ticket');
do_assert($flat_basket[0]->venue instanceof \Red61\Via\DataObject\ViaApiBasketVenueItem, 'Flat basket has venue as property of each ticket');
do_assert($flat_basket[0]->subvenue instanceof \Red61\Via\DataObject\ViaApiBasketSubVenueItem, 'Flat basket has subvenue as property of each ticket');
do_assert($flat_basket[0]->performance instanceof \Red61\Via\DataObject\ViaApiBasketPerformanceItem, 'Flat basket has performance as property of each ticket');



/*-----------------------------------------------------------------------------------------------
 * BOOKING AND PURCHASE SMOKE TESTS
 * 1a. Increase and reduce ticket quantities
 * ----------------------------------------------------------------------------------------------
 */
$basket = $via->getBasketItems();
$orig_tix_num = $basket->getSummary()->getNotickets();
$flat_basket = $via->getBasketTicketItemsFlat($basket);
$ticket = $flat_basket[0];
do_assert($via->updateTickets($ticket->getTicketids(), $ticket->getPerformanceid(), array($ticket->getPricebandid().'_'.$ticket->getPriceband_concessionid() => $ticket->getQuantity() + 1), $ticket->getQuantity()), 'Can increase number of tix');

$basket = $via->getBasketItems();
do_assert($basket->getSummary()->getNotickets() == ($orig_tix_num + 1), 'Total number of tickets increased by 1');

$flat_basket = $via->getBasketTicketItemsFlat($basket);
$ticket = $flat_basket[0];
do_assert($via->updateTickets($ticket->getTicketids(), $ticket->getPerformanceid(), array($ticket->getPricebandid().'_'.$ticket->getPriceband_concessionid() => $ticket->getQuantity() - 1), $ticket->getQuantity()), 'Can decrease number of tix');
do_assert($via->getBasketItems()->getSummary()->getNotickets() == $orig_tix_num, 'Total number of tickets back to original');

/*-----------------------------------------------------------------------------------------------
 * BOOKING AND PURCHASE SMOKE TESTS
 * 2. Init a new VIA instance with the allocated basket ID to ensure it's set properly
 * ----------------------------------------------------------------------------------------------
 */
print "Switching to new VIA instance to ensure basket ID managed properly on creation".\PHP_EOL;
$basketid = $via->getBasketId();
$via = new red61_via($_SERVER['VIA_WSDL'], $_SERVER['VIA_WEBKEY'], $basketid);
$basket = $via->getBasketItems();
do_assert($basket->events, "Has events still in basket");

do_assert($via->getBasketSubTotal() > 0, 'Returns basket subtotal as direct method');
do_assert($via->getBasketTotal() > 0, 'Returns basket total as direct method');
do_assert($via->getNoOfItemsInBasket() > 0, 'Returns number of items in basket as direct method');
do_assert($via->getBasketSummary()->notickets, 'Returns basket summary with notickets property');

/*-----------------------------------------------------------------------------------------------
 * BOOKING AND PURCHASE SMOKE TESTS
 * 3. Purchase a scheme
 * ----------------------------------------------------------------------------------------------
 */
do_assert($schemes = $via->getSchemes(), "Can list schemes");
do_assert($scheme_details = $via->getSchemeDetails($schemes[0]->id), "Can get details for first scheme");
$scheme_ref = $scheme_details->schemeTiers[0]->tierId;
do_assert($via->addSchemeToBasket($scheme_ref, 0, $cust_id), 'Can add scheme to basket');
do_assert($via->removeSchemeFromBasket($scheme_ref, 0, $cust_id), 'Can remove scheme from basket');

/*-----------------------------------------------------------------------------------------------
 * BOOKING AND PURCHASE SMOKE TESTS
 * 4. Purchase a voucher
 * ----------------------------------------------------------------------------------------------
 */
do_assert(is_array($via->getVoucherTypes()), 'Can list voucher types as array');
// @todo: set up some vouchers on QA for purchase tests


/*-----------------------------------------------------------------------------------------------
 * BOOKING AND PURCHASE SMOKE TESTS
 * 5. Redeem a voucher
 * ----------------------------------------------------------------------------------------------
 */
// @todo redeem some vouchers (though tricky as we can't get the voucher code when they're bought)
do_assert(is_array($via->getVoucherRedeemsFromBasket()), 'Returns array for voucher redeems from basket');

/*-----------------------------------------------------------------------------------------------
 * BOOKING AND PURCHASE SMOKE TESTS
 * 6. Merchandise
 * ----------------------------------------------------------------------------------------------
 */
// @todo setup some merchandise on QA for purchase tests
do_assert(is_array($via->getMerchandiseCategories()), 'Returns array of merchandise categories');

/*-----------------------------------------------------------------------------------------------
 * BOOKING AND PURCHASE SMOKE TESTS
 * 7. Donations
 * ----------------------------------------------------------------------------------------------
 */
do_assert($funds = $via->getDonationFunds(), 'Gets donation funds');
$fund_ref = $funds[0]->donationFundRef;
do_assert($via->addDonationToBasket($fund_ref, 5.90), 'Can add donation to basket');
do_assert($via->removeDonationFromBasket($fund_ref), 'Can remove donation from basket');

/*-----------------------------------------------------------------------------------------------
 * BOOKING AND PURCHASE SMOKE TESTS
 * 8. Clear Basket
 * ----------------------------------------------------------------------------------------------
 */
do_assert($via->clearBasket(TRUE), 'Can remove basket');
do_assert($via->getBasketId() === NULL, 'Basket ID is NULL after removing basket');
do_assert( ! $via->getBasketItems()->getSummary()->getTotal(), 'No basket items after removing basket');

/*-----------------------------------------------------------------------------------------------
 * FULL PURCHASE SMOKE TEST
 * ----------------------------------------------------------------------------------------------
 */
print "Switching to new VIA instance to guarantee clean state for this test".\PHP_EOL;
$via = new red61_via($_SERVER['VIA_WSDL'], $_SERVER['VIA_WEBKEY'], NULL);

do_assert($via->addDonationToBasket($fund_ref, '1.20'), 'Can add donation to basket');
$order_details = $via->createOrder('MR A TEST', VIA_TEST_CARD_NUMBER, VIA_TEST_CARD_START, VIA_TEST_CARD_EXPIRY, VIA_TEST_CARD_ISSUE, VIA_TEST_CARD_CVV, 'test.red61.com', 0, $cust_id, TRUE);
list($order_server, $order_ref) = explode(':', $order_details->code);
do_assert($order_ref > 0, 'Can create order');
do_assert($via->getBasketId() === NULL, 'Basket ID is NULL after completing order');
do_assert( ! $via->getBasketItems()->getSummary()->getTotal(), 'No basket items after completing order');

do_assert($via->sendConfirmationEmail($order_details->code), 'Can send confirmation email');

/*-----------------------------------------------------------------------------------------------
 * CUSTOMER TRANSACTIONS SMOKE TEST - AFTER PURCHASE
 * ----------------------------------------------------------------------------------------------
 */
do_assert($transactions = $via->getCustomerTransactionsSummary($cust_id), 'Can get customer transactions summary');
$first_tran = array_pop($transactions);
do_assert($first_tran->orderId === $order_details->code, 'Test purchase is returned in transaction list');
do_assert($via->getOrderSummary($first_tran->orderId, $first_tran->title), 'Can fetch order summary for given order');

/*-----------------------------------------------------------------------------------------------
 * CUSTOMER SAVED CARDS SMOKE TEST - AFTER PURCHASE
 * ----------------------------------------------------------------------------------------------
 */
do_assert($cards = $via->getCustomerSavedCards($cust_id), 'Can get saved cards');
do_assert($via->addDonationToBasket($fund_ref, '4.90'),   'Can add donation to basket');
$order_details = $via->createRepeatPaymentOrder($cards[0]->id, VIA_TEST_CARD_CVV, 'test.red61.com', 0, $cust_id);
do_assert($order_details->wasSuccessful(), 'Repeat payment order was successful');

/*-----------------------------------------------------------------------------------------------
 * EXCEPTION SMOKE TESTS
 * ----------------------------------------------------------------------------------------------
 */

print "Creating new VIA client with invalid BasketId".\PHP_EOL;
$via = new red61_via($_SERVER['VIA_WSDL'], $_SERVER['VIA_WEBKEY'], '1233456');
do_assert($via->getBasketSummary()->getNotickets() == 0, 'Got empty basket summary from getBasketSummary with invalid basket by default');
do_assert($via->getBasketId() === NULL, 'Basket ID is cleared after CartNotFoundException without rethrow');

print "Creating new VIA client with invalid BasketId and always_rethrow on".\PHP_EOL;
$via = new red61_via($_SERVER['VIA_WSDL'], $_SERVER['VIA_WEBKEY'], '1233456', array('always_rethrow' => TRUE));
try {
	$via->getBasketSummary();
	do_assert(FALSE, 'Should get SoapFault from getBasketSummary when always_rethrow is on');
} catch (SoapFault $e) {
	// Expected
	do_assert(TRUE, 'Got SoapFault from getBasketSummary when always_rethrow is on');
	do_assert($e instanceof ViaException, 'Exception is a ViaException');
	do_assert($via->getBasketId() === NULL, 'Basket ID is cleared after CartNotFoundException with rethrow');
}

print "Creating new VIA client with invalid BasketId and always_rethrow on".\PHP_EOL;
$via = new red61_via($_SERVER['VIA_WSDL'], $_SERVER['VIA_WEBKEY'], '1233456', array('always_rethrow' => TRUE));
try {
	$via->getBasketSummary();
	do_assert(FALSE, 'getBasketSummary should throw exception with invalid basket when always_rethrow is on');
} catch (CartNotFoundException $e) {
	// Expected
	do_assert(TRUE, 'CartNotFoundException is thrown from getBasketSummary when always_rethrow is on');
	do_assert($e instanceof ViaException, 'Exception is a ViaException');
}

$via = new red61_via('https://some.random.server', NULL);
try {
	$via->getArtists();
	do_assert(FALSE, 'Expected exception when connecting to random VIA server that does not exist');
} catch (SoapFault $e) {
	do_assert(TRUE, 'Exception from connection problems can be caught as SoapFault');
	do_assert($e instanceof ViaSoapFault, 'Exception from connection problems is a ViaSoapFault');
	do_assert($e instanceof ViaException, 'Exception is a ViaException');
}

/*-----------------------------------------------------------------------------------------------
 * CACHING SMOKE TESTS - FILESYSTEM DRIVER AND NO CONFIG
 * ----------------------------------------------------------------------------------------------
 */
print 'Wiping temporary API cache in /tmp/api'.\PHP_EOL;
`rm -rf /tmp/api`;

print 'Creating new VIA instance with default (filesystem) cache config '.$_SERVER['VIA_WSDL'].\PHP_EOL;
$via = new red61_via($_SERVER['VIA_WSDL'], $_SERVER['VIA_WEBKEY'], NULL);

$results = assert_second_getevents_faster_by(10, $via);
do_assert($results[0] == $results[1], 'First response is same as second response');
do_assert(glob('/tmp/api/**/*'), "Cache files have been created in /tmp/api");

/*-----------------------------------------------------------------------------------------------
 * CACHING SMOKE TESTS - EXPLICITLY CONFIGURED FILE CACHE DRIVER
 * ----------------------------------------------------------------------------------------------
 */
$path = '/tmp/'.uniqid('via-cache-test');
print 'Creating new VIA instance with filesystem cache in '.$path.\PHP_EOL;
$via = new red61_via($_SERVER['VIA_WSDL'], $_SERVER['VIA_WEBKEY'], NULL, array('cache' => TRUE, 'cache_dir' => 'FS:'.$path));

$results = assert_second_getevents_faster_by(10, $via);
do_assert($results[0] == $results[1], 'First response is same as second response');
do_assert(glob($path.'/**/*'), "Cache files have been created in $path");

/*-----------------------------------------------------------------------------------------------
 * CACHING SMOKE TESTS - EXPLICITLY CONFIGURED MEMCACHE DRIVER
 * ----------------------------------------------------------------------------------------------
 */
print 'Flushing memcache cache in '.$_SERVER['MEMCACHE_HOST'];
$memcache = new Memcache();
$memcache->connect($_SERVER['MEMCACHE_HOST'], 11211);
$memcache->flush();
$mc_stats_before = $memcache->getstats();

print 'Creating new VIA instance with memcache cache in '.$_SERVER['MEMCACHE_HOST'].\PHP_EOL;
$via = new red61_via($_SERVER['VIA_WSDL'], $_SERVER['VIA_WEBKEY'], NULL, array('cache' => TRUE, 'cache_dir' => 'memcache:'.$_SERVER['MEMCACHE_HOST']));

$results = assert_second_getevents_faster_by(10, $via);
do_assert($results[0] == $results[1], 'First response is same as second response');

$mc_stats_after = $memcache->getstats();
// These aren't totally robust, as they don't prove the memcache wasn't written to by something else
do_assert($mc_stats_after['cmd_get'] > $mc_stats_before['cmd_get'], 'Memcache has been fetched from');
do_assert($mc_stats_after['cmd_set'] > $mc_stats_before['cmd_set'], 'Memcache has been written to');

print "=================================================================".\PHP_EOL;
print "* TESTS PASSED OK                                               *".\PHP_EOL;
print "=================================================================".\PHP_EOL;
exit(0);

function do_assert($condition, $message)
{
	if ($condition) {
		print "✓ ".$message.\PHP_EOL;
	} else {
		print "✖ ".$message.\PHP_EOL;
		throw new \UnexpectedValueException('Assertion failed : '.$message);
	}
}

function try_times($iterations, $callable) {
	while ($iterations) {
		$iterations--;
		try {
			call_user_func($callable);
			return;
		} catch (Exception $e) {
			if ($iterations) {
				throw $e;
			} else {
				print "! Swallowing ".$e->getMessage()." - $iterations more attempts".\PHP_EOL;
			}
		}
	}
}

/**
 * @param $perf_prices
 * @return array
 */
function build_ticket_quantities_for_event($perf_prices)
{
	$quantities = array();
	$bandid = $perf_prices[0]->pricebandid;
	if ($perf_prices[0]->hideFullPrice) {
		// Just a concession
		$quantities[$bandid.'_'.$perf_prices[0]->concessions[0]->concessionid] = 1;
	} else {
		// A full price and a random concession if available
		$quantities[$bandid.'_0'] = 1;
		if (isset($perf_prices[0]->concessions)) {
			$concession = $perf_prices[0]->concessions[rand(0, count($perf_prices[0]->concessions) - 1)];
			$quantities[$bandid.'_'.$concession->concessionid] = 1;
		}
	}

	return $quantities;
}

/**
 * Runs a function twice and verifies that the second call was faster than the first by a configurable
 * factor. Used to verify that responses are cached.
 *
 * @param double    $second_faster_by
 * @param Red61_Via $via
 *
 * @return mixed[] returns first and second result for checking
 */
function assert_second_getevents_faster_by($second_faster_by, $via)
{
	$started = microtime(TRUE);
	$results[] = $via->getEvents();
	$after_first = microtime(TRUE);
	$results[] = $via->getEvents();
	$after_second = microtime(TRUE);

	$first_time_ms  = round(($after_first - $started) * 1000);
	$second_time_ms = round(($after_second - $after_first) * 1000);
	$actual_factor  = $first_time_ms / $second_time_ms;

	do_assert(
		$actual_factor > $second_faster_by,
		sprintf(
			'Second getEvents call (%.3f ms) at least %.1f times faster than first call (%.3f ms) - actual factor %.1f',
			$second_time_ms,
			$second_faster_by,
			$first_time_ms,
			$actual_factor
		)
	);
	return $results;
}
