<?php
/**
 * Defines ViaMemcacheDriverSpec - specifications for Red61\Via\Cache\ViaMemcacheDriver
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */

namespace spec\Red61\Via\Cache;

use PhpSpec\Exception\Example\FailureException;
use PhpSpec\Exception\Example\SkippingException;
use PhpSpec\Exception\Exception;
use Red61\Via\Exception\CacheNotAvailableException;
use spec\ObjectBehavior;
use Prophecy\Argument;

if ( ! class_exists('\Memcache')) {
    // Define an empty class to allow us to get past prophecy's mocking and skip the tests
    eval('class Memcache { public static $not_installed = TRUE;}');
}

/**
 *
 * @see Red61\Via\Cache\ViaMemcacheDriver
 */
class ViaMemcacheDriverSpec extends ObjectBehavior
{

	/**
     * Use $this->subject to get proper type hinting for the subject class
     * @var \Red61\Via\Cache\ViaMemcacheDriver
     */
	protected $subject;

	/**
	 * @param \Memcache $memcache
	 */
	function let($memcache)
	{
		if ( isset(\Memcache::$not_installed)) {
		    throw new SkippingException("Skipping - memcache extension is not installed");
		}
		$this->beConstructedWith($memcache, array());
		$memcache->connect(Argument::cetera())->willReturn(TRUE);
		$memcache->pconnect(Argument::cetera())->willReturn(TRUE);
	}

	function it_is_initializable()
    {
		$this->subject->shouldHaveType('Red61\Via\Cache\ViaMemcacheDriver');
	}

	function it_is_a_via_cache_driver()
	{
		$this->subject->shouldHaveType('Red61\Via\Cache\ViaCacheDriver');
	}

	/**
	 * @param \Memcache $memcache
	 */
	function it_connects_to_localhost_memcache_non_persistent_by_default($memcache)
	{
		$this->subject->getWrappedObject();
		$memcache->connect('127.0.0.1', 11211)->shouldHaveBeenCalled();
	}

	/**
	 * @param \Memcache $memcache
	 */
	function it_connects_to_memcache_from_options_if_set($memcache)
	{
		$this->beConstructedWith($memcache, array('host' => '12.23.23.23', 'port' => 12314));
		$this->subject->getWrappedObject();
		$memcache->connect('12.23.23.23', 12314)->shouldHaveBeenCalled();
	}

	/**
	 * @param \Memcache $memcache
	 */
	function it_opens_persistent_connection_if_configured($memcache)
	{
		$this->beConstructedWith($memcache, array('host' => '10.1.23.23', 'persistent' => TRUE, 'port' => 12314));
		$this->subject->getWrappedObject();
		$memcache->pconnect('10.1.23.23', 12314)->shouldHaveBeenCalled();
	}

	/**
	 * @param \Memcache $memcache
	 */
	function it_does_not_attempt_to_connect_if_already_connected_option_set($memcache)
	{
		$this->beConstructedWith($memcache, array('already_connected' => TRUE));
		$this->subject->getWrappedObject();

		$memcache->connect()->shouldNotHaveBeenCalled();
		$memcache->pconnect()->shouldNotHaveBeenCalled();
	}

	/**
	 * @param \Memcache $memcache
	 *
	 * @throws \PhpSpec\Exception\Example\FailureException
	 */
	function it_throws_if_connection_fails($memcache)
	{
		$memcache->connect(Argument::cetera())->willReturn(FALSE);

		try {
			$this->subject->getWrappedObject();
			throw new FailureException("Expected Exception not thrown");
		} catch (CacheNotAvailableException $e) {
			expect($e)->toBeAnInstanceOf('\Red61\Via\Exception\ViaException');
		}
	}

	/**
	 * @param \Memcache $memcache
	 */
	function it_performs_lookup_in_memcache_and_returns_result($memcache)
	{
		$memcache->get('key')->willReturn('some value');

		$this->subject->lookup('apiGetSomething', 'key', 60)->shouldBe('some value');
	}

	/**
	 * @param \Memcache $memcache
	 */
	function it_performs_store_in_memcache($memcache)
	{
		$memcache->set(Argument::cetera())->willReturn(TRUE);
		$this->subject->store('apiGetSomething', 'key', 'value', 3600);
		$memcache->set('key', 'value', 0, 3600)->shouldHaveBeenCalled();
	}

	/**
	 * @param \Memcache $memcache
	 */
	function it_performs_flush_in_memcache($memcache)
	{
		$memcache->flush()->willReturn(TRUE);
		$this->subject->flush();
		$memcache->flush()->shouldHaveBeenCalled();
	}
}
